// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright 2019 NXP
 *
 */

#include <common.h>
#include <dm.h>
#include <dm/device_compat.h>
#include <mipi_dsi.h>
#include <panel.h>
#include <asm/gpio.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <mipi_dsi.h>
#define CMD_TABLE_LEN 2
typedef u8 cmd_set_table[CMD_TABLE_LEN];

/* Write Manufacture Command Set Control */
#define WRMAUCCTR 0xFE

struct rm67191_panel_priv {
	struct gpio_desc reset;
	unsigned int lanes;
	enum mipi_dsi_pixel_format format;
	unsigned long mode_flags;
};

struct rad_platform_data {
	int (*enable)(struct udevice *dev);
};

/* Manufacturer Command Set pages (CMD2) */
static const cmd_set_table mcs_rm67191[] = {
	{0xFE, 0x0B},
	{0x28, 0x40},
	{0x29, 0x4F},
	{0xFE, 0x0E},
	{0x4B, 0x00},
	{0x4C, 0x0F},
	{0x4D, 0x20},
	{0x4E, 0x40},
	{0x4F, 0x60},
	{0x50, 0xA0},
	{0x51, 0xC0},
	{0x52, 0xE0},
	{0x53, 0xFF},
	{0xFE, 0x0D},
	{0x18, 0x08},
	{0x42, 0x00},
	{0x08, 0x41},
	{0x46, 0x02},
	{0x72, 0x09},
	{0xFE, 0x0A},
	{0x24, 0x17},
	{0x04, 0x07},
	{0x1A, 0x0C},
	{0x0F, 0x44},
	{0xFE, 0x04},
	{0x00, 0x0C},
	{0x05, 0x08},
	{0x06, 0x08},
	{0x08, 0x08},
	{0x09, 0x08},
	{0x0A, 0xE6},
	{0x0B, 0x8C},
	{0x1A, 0x12},
	{0x1E, 0xE0},
	{0x29, 0x93},
	{0x2A, 0x93},
	{0x2F, 0x02},
	{0x31, 0x02},
	{0x33, 0x05},
	{0x37, 0x2D},
	{0x38, 0x2D},
	{0x3A, 0x1E},
	{0x3B, 0x1E},
	{0x3D, 0x27},
	{0x3F, 0x80},
	{0x40, 0x40},
	{0x41, 0xE0},
	{0x4F, 0x2F},
	{0x50, 0x1E},
	{0xFE, 0x06},
	{0x00, 0xCC},
	{0x05, 0x05},
	{0x07, 0xA2},
	{0x08, 0xCC},
	{0x0D, 0x03},
	{0x0F, 0xA2},
	{0x32, 0xCC},
	{0x37, 0x05},
	{0x39, 0x83},
	{0x3A, 0xCC},
	{0x41, 0x04},
	{0x43, 0x83},
	{0x44, 0xCC},
	{0x49, 0x05},
	{0x4B, 0xA2},
	{0x4C, 0xCC},
	{0x51, 0x03},
	{0x53, 0xA2},
	{0x75, 0xCC},
	{0x7A, 0x03},
	{0x7C, 0x83},
	{0x7D, 0xCC},
	{0x82, 0x02},
	{0x84, 0x83},
	{0x85, 0xEC},
	{0x86, 0x0F},
	{0x87, 0xFF},
	{0x88, 0x00},
	{0x8A, 0x02},
	{0x8C, 0xA2},
	{0x8D, 0xEA},
	{0x8E, 0x01},
	{0x8F, 0xE8},
	{0xFE, 0x06},
	{0x90, 0x0A},
	{0x92, 0x06},
	{0x93, 0xA0},
	{0x94, 0xA8},
	{0x95, 0xEC},
	{0x96, 0x0F},
	{0x97, 0xFF},
	{0x98, 0x00},
	{0x9A, 0x02},
	{0x9C, 0xA2},
	{0xAC, 0x04},
	{0xFE, 0x06},
	{0xB1, 0x12},
	{0xB2, 0x17},
	{0xB3, 0x17},
	{0xB4, 0x17},
	{0xB5, 0x17},
	{0xB6, 0x11},
	{0xB7, 0x08},
	{0xB8, 0x09},
	{0xB9, 0x06},
	{0xBA, 0x07},
	{0xBB, 0x17},
	{0xBC, 0x17},
	{0xBD, 0x17},
	{0xBE, 0x17},
	{0xBF, 0x17},
	{0xC0, 0x17},
	{0xC1, 0x17},
	{0xC2, 0x17},
	{0xC3, 0x17},
	{0xC4, 0x0F},
	{0xC5, 0x0E},
	{0xC6, 0x00},
	{0xC7, 0x01},
	{0xC8, 0x10},
	{0xFE, 0x06},
	{0x95, 0xEC},
	{0x8D, 0xEE},
	{0x44, 0xEC},
	{0x4C, 0xEC},
	{0x32, 0xEC},
	{0x3A, 0xEC},
	{0x7D, 0xEC},
	{0x75, 0xEC},
	{0x00, 0xEC},
	{0x08, 0xEC},
	{0x85, 0xEC},
	{0xA6, 0x21},
	{0xA7, 0x05},
	{0xA9, 0x06},
	{0x82, 0x06},
	{0x41, 0x06},
	{0x7A, 0x07},
	{0x37, 0x07},
	{0x05, 0x06},
	{0x49, 0x06},
	{0x0D, 0x04},
	{0x51, 0x04},
};

static const cmd_set_table mcs_rm67199[] = {
	{0xFE, 0xA0}, {0x2B, 0x18}, {0xFE, 0x70}, {0x7D, 0x05},
	{0x5D, 0x0A}, {0x5A, 0x79}, {0x5C, 0x00}, {0x52, 0x00},
	{0xFE, 0xD0}, {0x40, 0x02}, {0x13, 0x40}, {0xFE, 0x40},
	{0x05, 0x08}, {0x06, 0x08}, {0x08, 0x08}, {0x09, 0x08},
	{0x0A, 0xCA}, {0x0B, 0x88}, {0x20, 0x93}, {0x21, 0x93},
	{0x24, 0x02}, {0x26, 0x02}, {0x28, 0x05}, {0x2A, 0x05},
	{0x74, 0x2F}, {0x75, 0x1E}, {0xAD, 0x00}, {0xFE, 0x60},
	{0x00, 0xCC}, {0x01, 0x00}, {0x02, 0x04}, {0x03, 0x00},
	{0x04, 0x00}, {0x05, 0x07}, {0x06, 0x00}, {0x07, 0x88},
	{0x08, 0x00}, {0x09, 0xCC}, {0x0A, 0x00}, {0x0B, 0x04},
	{0x0C, 0x00}, {0x0D, 0x00}, {0x0E, 0x05}, {0x0F, 0x00},
	{0x10, 0x88}, {0x11, 0x00}, {0x12, 0xCC}, {0x13, 0x0F},
	{0x14, 0xFF}, {0x15, 0x04}, {0x16, 0x00}, {0x17, 0x06},
	{0x18, 0x00}, {0x19, 0x96}, {0x1A, 0x00}, {0x24, 0xCC},
	{0x25, 0x00}, {0x26, 0x02}, {0x27, 0x00}, {0x28, 0x00},
	{0x29, 0x06}, {0x2A, 0x06}, {0x2B, 0x82}, {0x2D, 0x00},
	{0x2F, 0xCC}, {0x30, 0x00}, {0x31, 0x02}, {0x32, 0x00},
	{0x33, 0x00}, {0x34, 0x07}, {0x35, 0x06}, {0x36, 0x82},
	{0x37, 0x00}, {0x38, 0xCC}, {0x39, 0x00}, {0x3A, 0x02},
	{0x3B, 0x00}, {0x3D, 0x00}, {0x3F, 0x07}, {0x40, 0x00},
	{0x41, 0x88}, {0x42, 0x00}, {0x43, 0xCC}, {0x44, 0x00},
	{0x45, 0x02}, {0x46, 0x00}, {0x47, 0x00}, {0x48, 0x06},
	{0x49, 0x02}, {0x4A, 0x8A}, {0x4B, 0x00}, {0x5F, 0xCA},
	{0x60, 0x01}, {0x61, 0xE8}, {0x62, 0x09}, {0x63, 0x00},
	{0x64, 0x07}, {0x65, 0x00}, {0x66, 0x30}, {0x67, 0x00},
	{0x9B, 0x03}, {0xA9, 0x07}, {0xAA, 0x06}, {0xAB, 0x02},
	{0xAC, 0x10}, {0xAD, 0x11}, {0xAE, 0x05}, {0xAF, 0x04},
	{0xB0, 0x10}, {0xB1, 0x10}, {0xB2, 0x10}, {0xB3, 0x10},
	{0xB4, 0x10}, {0xB5, 0x10}, {0xB6, 0x10}, {0xB7, 0x10},
	{0xB8, 0x10}, {0xB9, 0x10}, {0xBA, 0x04}, {0xBB, 0x05},
	{0xBC, 0x00}, {0xBD, 0x01}, {0xBE, 0x0A}, {0xBF, 0x10},
	{0xC0, 0x11}, {0xFE, 0xA0}, {0x22, 0x00},
};


static const struct display_timing default_timing = {
	.pixelclock.typ		= (1200 + 26 + 26 + 26) * (1920 + 224 + 8 + 38) * 60 ,
	.hactive.typ		= 1200,
	.hfront_porch.typ	= 26,
	.hback_porch.typ	= 26,
	.hsync_len.typ		= 26,
	.vactive.typ		= 1920,
	.vfront_porch.typ	= 224 ,
	.vback_porch.typ	= 38,
	.vsync_len.typ		= 8,
	.flags = DISPLAY_FLAGS_HSYNC_LOW |
		 DISPLAY_FLAGS_VSYNC_LOW ,
		 DISPLAY_FLAGS_DE_LOW |
  		 DISPLAY_FLAGS_PIXDATA_NEGEDGE,
};


static u8 color_format_from_dsi_format(enum mipi_dsi_pixel_format format)
{
	switch (format) {
	case MIPI_DSI_FMT_RGB565:
		return 0x55;
	case MIPI_DSI_FMT_RGB666:
	case MIPI_DSI_FMT_RGB666_PACKED:
		return 0x66;
	case MIPI_DSI_FMT_RGB888:
		return 0x77;
	default:
		return 0x77; /* for backward compatibility */
	}
};

static int rad_panel_push_cmd_list(struct mipi_dsi_device *device,
				   const cmd_set_table *cmd_set,
				   size_t count)
{
	size_t i;
	const cmd_set_table *cmd;
	int ret = 0;

	for (i = 0; i < count; i++) {
		cmd = cmd_set++;
		ret = mipi_dsi_generic_write(device, cmd, CMD_TABLE_LEN);
		if (ret < 0)
			return ret;
	}

	return ret;
};

#define dsi_dcs_write_seq(dsi, seq...) do {				\
		static const u8 d[] = { seq };				\
		int ret;					                        	\
		ret = mipi_dsi_dcs_write_buffer(dsi, d, ARRAY_SIZE(d));	\
		if (ret < 0)						\
			return ret;					\
	} while (0)

static int rm67191_enable(struct udevice *dev)
{
	struct rm67191_panel_priv *priv = dev_get_priv(dev);
	struct mipi_dsi_panel_plat *plat = dev_get_plat(dev);
	struct mipi_dsi_device *dsi = plat->device;
	u8 color_format = color_format_from_dsi_format(priv->format);
	u16 brightness;
	int ret;

	dsi->mode_flags |= MIPI_DSI_MODE_LPM;

	printf("%s: enter!\n", __func__);

	dsi_dcs_write_seq(dsi, 0xB9, 0x83, 0x10, 0x2E);
	dsi_dcs_write_seq(dsi, 0xBD, 0x00);
	dsi_dcs_write_seq(dsi, 0xE1, 0x01, 0x00);
	dsi_dcs_write_seq(dsi, 0xD1, 0x37, 0x0C, 0xFF, 0x05);
	dsi_dcs_write_seq(dsi, 0xB1, 0x10, 0xFA, 0xAF, 0xAF, 0x2D, 0x2D, 0xA2, 0x44, 0x58, 0x38, 0x38, 0x38, 0x38, 0x22, 0x21, 0x15, 0x00);
	dsi_dcs_write_seq(dsi, 0xD2, 0x2D, 0x2D);
	dsi_dcs_write_seq(dsi, 0xB2, 0x00, 0xB0, 0x47, 0x80, 0x00, 0x2C, 0xE0, 0x29, 0x11, 0x11, 0x00, 0x00, 0x15, 0x20, 0xD7, 0x00);
	dsi_dcs_write_seq(dsi, 0xB4, 0x88, 0x64, 0x01, 0x01, 0x88, 0x64, 0x88, 0x64, 0x01, 0x90, 0x01);
	dsi_dcs_write_seq(dsi, 0xBA, 0x70, 0x03, 0xA8, 0x85, 0xF2, 0x00, 0xC0, 0x0D);
	dsi_dcs_write_seq(dsi, 0xBF, 0xFC, 0x85, 0x80);
	dsi_dcs_write_seq(dsi, 0xC0, 0x23, 0x23, 0x22, 0x11, 0xA2, 0x17, 0x00, 0x80, 0x00, 0x00, 0x08, 0x00, 0x63, 0x63);
	dsi_dcs_write_seq(dsi, 0xC7, 0x30);
	dsi_dcs_write_seq(dsi, 0xC8, 0x00, 0x04, 0x04, 0x00, 0x00, 0x82, 0x13, 0x01);
	dsi_dcs_write_seq(dsi, 0xCB, 0x00, 0x13, 0x08, 0x02, 0x24);
	dsi_dcs_write_seq(dsi, 0xCC, 0x02);
	dsi_dcs_write_seq(dsi, 0xD0, 0x07, 0x04, 0x05);
	dsi_dcs_write_seq(dsi, 0xD3, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x04, 0x00, 0x0C, 0x0C, 0x4B, 0x40, 0x44, 0x4F, 0x2B, 0x2B, 0x04, 0x04, 0x32, 0x10, 0x29, 0x00, 0x29, 0x54, 0x17, 0xAB, 0x07, 0xAB, 0x32, 0x10, 0x10, 0x00, 0x10, 0x00, 0x00, 0x21, 0x38, 0x01, 0x55, 0x21, 0x38, 0x01, 0x55, 0x0F);
	dsi_dcs_write_seq(dsi, 0xD5, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x27, 0x26, 0x25, 0x24, 0x1A, 0x1A, 0x1B, 0x1B, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, 0x21, 0x20, 0x18, 0x18, 0x18, 0x18);
	dsi_dcs_write_seq(dsi, 0xD6, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x18, 0x18, 0x26, 0x27, 0x20, 0x21, 0x1A, 0x1A, 0x1B, 0x1B, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x24, 0x25, 0x98, 0x98, 0x98, 0x98);
	dsi_dcs_write_seq(dsi, 0xD8, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xA0, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xA0, 0xAA, 0xAA, 0xAA, 0xEA, 0xAA, 0xA0, 0xAA, 0xAA, 0xAA, 0xEA, 0xAA, 0xA0);
	dsi_dcs_write_seq(dsi, 0xE0, 0x00, 0x05, 0x11, 0x1A, 0x22, 0x3E, 0x56, 0x5E, 0x66, 0x64, 0x7E, 0x84, 0x8B, 0x9B, 0x9A, 0xA2, 0xAA, 0xBB, 0xB8, 0x58, 0x5D, 0x62, 0x63, 0x00, 0x05, 0x11, 0x1A, 0x22, 0x3E, 0x56, 0x5E, 0x66, 0x64, 0x7E, 0x84, 0x8B, 0x9B, 0x9A, 0xA2, 0xAA, 0xBB, 0xB8, 0x58, 0x5D, 0x62, 0x74);
	dsi_dcs_write_seq(dsi, 0xE7, 0x12, 0x13, 0x02, 0x02);
	dsi_dcs_write_seq(dsi, 0xBD, 0x01);
	dsi_dcs_write_seq(dsi, 0xB1, 0x01, 0x9B, 0x01, 0x31);
	dsi_dcs_write_seq(dsi, 0xD3, 0x01, 0x00, 0xFC, 0x00, 0x00, 0x11, 0x10, 0x00, 0x0A, 0x00, 0x01);
	dsi_dcs_write_seq(dsi, 0xD8, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55);
	dsi_dcs_write_seq(dsi, 0xE7, 0x01);
	dsi_dcs_write_seq(dsi, 0xBD, 0x02);
	dsi_dcs_write_seq(dsi, 0xB4, 0x4E, 0x00, 0x33, 0x11, 0x33, 0x88);
	dsi_dcs_write_seq(dsi, 0xBF, 0xF2, 0x00, 0x02);
	dsi_dcs_write_seq(dsi, 0xCB, 0x00, 0x03, 0x00, 0x01, 0x7D);
	dsi_dcs_write_seq(dsi, 0xD8, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0);
	dsi_dcs_write_seq(dsi, 0xBD, 0x03);
	dsi_dcs_write_seq(dsi, 0xD8, 0xAA, 0xAA, 0xAB, 0xEA, 0xAA, 0xA0, 0xAA, 0xAA, 0xAB, 0xEA, 0xAA, 0xA0, 0xAA, 0xAA, 0xBB, 0xFF, 0xFE, 0xA0, 0xAA, 0xAA, 0xBB, 0xFF, 0xFE, 0xA0);
	dsi_dcs_write_seq(dsi, 0xBD, 0x00);
	dsi_dcs_write_seq(dsi, 0xE9, 0xCD);
	dsi_dcs_write_seq(dsi, 0xBB, 0x01);
	dsi_dcs_write_seq(dsi, 0xE9, 0x00);
	dsi_dcs_write_seq(dsi, 0xB8, 0x40, 0x00);
	dsi_dcs_write_seq(dsi, 0xBD, 0x01);
	dsi_dcs_write_seq(dsi, 0xB8, 0x15);
	dsi_dcs_write_seq(dsi, 0xBD, 0x00);
	dsi_dcs_write_seq(dsi, 0x11);
	mdelay(120);
	dsi_dcs_write_seq(dsi, 0x29);
	mdelay(50);

	return 0;
}

static int rm67199_enable(struct udevice *dev)
{
	struct rm67191_panel_priv *priv = dev_get_priv(dev);
	struct mipi_dsi_panel_plat *plat = dev_get_plat(dev);
	struct mipi_dsi_device *dsi = plat->device;
	u8 color_format = color_format_from_dsi_format(priv->format);
	u16 brightness;
	int ret;

	dsi->mode_flags |= MIPI_DSI_MODE_LPM;

	ret = rad_panel_push_cmd_list(dsi, &mcs_rm67199[0],
		sizeof(mcs_rm67199) / CMD_TABLE_LEN);
	if (ret < 0) {
		printf("Failed to send MCS (%d)\n", ret);
		return -EIO;
	}

	/* Select User Command Set table (CMD1) */
	ret = mipi_dsi_generic_write(dsi, (u8[]){ WRMAUCCTR, 0x00 }, 2);
	if (ret < 0)
		return -EIO;

	/* Set DSI mode */
	ret = mipi_dsi_generic_write(dsi, (u8[]){ 0xC2, 0x08 }, 2);
	if (ret < 0) {
		printf("Failed to set DSI mode (%d)\n", ret);
		return -EIO;
	}

	/* Set tear ON */
	ret = mipi_dsi_dcs_set_tear_on(dsi, MIPI_DSI_DCS_TEAR_MODE_VBLANK);
	if (ret < 0) {
		printf("Failed to set tear ON (%d)\n", ret);
		return -EIO;
	}

	/* Set tear scanline */
	ret = mipi_dsi_dcs_set_tear_scanline(dsi, 0x00);
	if (ret < 0) {
		printf("Failed to set tear scanline (%d)\n", ret);
		return -EIO;
	}

	/* Set pixel format */
	ret = mipi_dsi_dcs_set_pixel_format(dsi, color_format);
	if (ret < 0) {
		printf("Failed to set pixel format (%d)\n", ret);
		return -EIO;
	}


	/* Set display brightness */
	brightness = 255; /* Max brightness */
	ret = mipi_dsi_dcs_write(dsi, MIPI_DCS_SET_DISPLAY_BRIGHTNESS, &brightness, 2);
	if (ret < 0) {
		printf("Failed to set display brightness (%d)\n",
				  ret);
		return -EIO;
	}

	/* Exit sleep mode */
	ret = mipi_dsi_dcs_exit_sleep_mode(dsi);
	if (ret < 0) {
		printf("Failed to exit sleep mode (%d)\n", ret);
		return -EIO;
	}

	mdelay(120);

	ret = mipi_dsi_dcs_set_display_on(dsi);
	if (ret < 0) {
		printf("Failed to set display ON (%d)\n", ret);
		return -EIO;
	}

	mdelay(100);

	return 0;
}


static int rm67191_panel_enable_backlight(struct udevice *dev)
{
	struct mipi_dsi_panel_plat *plat = dev_get_plat(dev);
	struct rad_platform_data *data = (struct rad_platform_data *)dev_get_driver_data(dev);
	struct mipi_dsi_device *device = plat->device;
	struct rm67191_panel_priv *priv = dev_get_priv(dev);
	int ret;

	printf("%s: enter!\n", __func__);
	ret = mipi_dsi_attach(device);
	if (ret < 0) {
		printf("rm67191_panel_enable_backlight attach error\n");
		return ret;
	}
	ret = gpio_request_by_name(dev, "reset-gpio", 0, &priv->reset,
				   GPIOD_IS_OUT);
	if (ret) {
		printf("Warning: cannot get reset GPIO\n");
		if (ret != -ENOENT)
			return ret;
	}
 	ret = dm_gpio_set_value(&priv->reset, 0);
	mdelay(10);
	ret = dm_gpio_set_value(&priv->reset, 1);
	mdelay(5);
	ret = dm_gpio_set_value(&priv->reset, 0);
 	return data->enable(dev);
}

static int rm67191_panel_get_display_timing(struct udevice *dev,
					    struct display_timing *timings)
{
	struct mipi_dsi_panel_plat *plat = dev_get_plat(dev);
	struct mipi_dsi_device *device = plat->device;
	struct rm67191_panel_priv *priv = dev_get_priv(dev);

	printf("%s: enter!\n", __func__);
	memcpy(timings, &default_timing, sizeof(*timings));

	/* fill characteristics of DSI data link */
	if (device) {
		device->lanes = priv->lanes;
		device->format = priv->format;
		device->mode_flags = priv->mode_flags;
	}

	return 0;
}

static int rm67191_panel_probe(struct udevice *dev)
{
	struct rm67191_panel_priv *priv = dev_get_priv(dev);
	struct uclass *bus;
	struct udevice *i2c_dev = NULL;
	u32 video_mode;
	int ret;
	int i;
	printf("%s: start!\n", __func__);

	priv->format = MIPI_DSI_FMT_RGB888;
	priv->mode_flags = MIPI_DSI_MODE_VIDEO_HSE | MIPI_DSI_MODE_VIDEO;

	ret = uclass_get(UCLASS_I2C, &bus);
	if (ret) {
		return ret;
	}

	for (i = 0; ; i++) {
		ret = uclass_get_device_by_seq(UCLASS_I2C, i, &i2c_dev);
		if (ret == -ENODEV) {
			printf("%s: get devices error!\n", __func__);
			break;
		}
	}

	uclass_foreach_dev(i2c_dev, bus) {
		ret = device_probe(i2c_dev);
		if (ret) {
			printf("%s - probe failed: %d\n",__func__, ret);
		}
	}

	ret = dev_read_u32(dev, "video-mode", &video_mode);
	if (!ret) {
		switch (video_mode) {
		case 0:
			/* burst mode */
			priv->mode_flags |= MIPI_DSI_MODE_VIDEO_BURST;
			break;
		case 1:
			/* non-burst mode with sync event */
			break;
		case 2:
			/* non-burst mode with sync pulse */
			priv->mode_flags |= MIPI_DSI_MODE_VIDEO_SYNC_PULSE;
			break;
		default:
			dev_warn(dev, "invalid video mode %d\n", video_mode);
			break;
		}
	}

	ret = dev_read_u32(dev, "dsi-lanes", &priv->lanes);
	if (ret) {
		printf("Failed to get dsi-lanes property (%d)\n", ret);
		return ret;
	}
	return 0;
}

static int rm67191_panel_disable(struct udevice *dev)
{
	struct rm67191_panel_priv *priv = dev_get_priv(dev);

	dm_gpio_set_value(&priv->reset, true);

	return 0;
}

static const struct panel_ops rm67191_panel_ops = {
	.enable_backlight = rm67191_panel_enable_backlight,
	.get_display_timing = rm67191_panel_get_display_timing,
};

static const struct rad_platform_data rad_rm67191 = {
	.enable = &rm67191_enable,
};

static const struct rad_platform_data rad_rm67199 = {
	.enable = &rm67199_enable,
};

static const struct udevice_id rm67191_panel_ids[] = {
	{ .compatible = "raydium,rm67191", .data = (ulong)&rad_rm67191 },
	{ .compatible = "raydium,rm67199", .data = (ulong)&rad_rm67199 },
	{ }
};

U_BOOT_DRIVER(rm67191_panel) = {
	.name			  = "rm67191_panel",
	.id			  = UCLASS_PANEL,
	.of_match		  = rm67191_panel_ids,
	.ops			  = &rm67191_panel_ops,
	.probe			  = rm67191_panel_probe,
	.remove			  = rm67191_panel_disable,
	.plat_auto = sizeof(struct mipi_dsi_panel_plat),
	.priv_auto = sizeof(struct rm67191_panel_priv),
};
