// SPDX-License-Identifier: GPL-2.0+
/*
 * Cortina PHY drivers framework
 *
 * Copyright 2018-2022 NXP
 * Copyright (C) 2006-2017 Inphi Corporation, Inc. All rights reserved.
 */

#include "cortina_api.h"

static struct phy_device *cs4223_phydev;
static struct mii_dev    *cs4223_bus;

void cs4223_glue_phydev_set(struct phy_device *phydev)
{
	cs4223_phydev = phydev;
	cs4223_bus = phydev->bus;
}

int cs4224_reg_set(unsigned int slice,
		   unsigned int addr,
		   unsigned short data)
{
	cs4223_bus->write(cs4223_bus, 0, 0, addr, data);
	return CS_OK;
}

int cs4224_reg_get(unsigned int slice,
		   unsigned int addr,
		   unsigned short *data)
{
	*data = cs4223_bus->read(cs4223_bus, 0, 0, addr);
	return CS_OK;
}

unsigned char cs4224_diags_register_can_read(unsigned short addr)
{
	/* don't read from select register ranges */
	if ((addr >= CS4224_PP_LINE_SDS_DSP_MSEQ_IX &&
	     addr <= CS4224_PP_LINE_SDS_DSP_MSEQ_BASE3_INST) ||
	     (addr >= CS4224_PP_HOST_SDS_DSP_MSEQ_IX &&
	      addr <= CS4224_PP_HOST_SDS_DSP_MSEQ_BASE3_INST)) {
		return CS_FALSE;
	}
	return CS_TRUE;
}

#if !defined(CS_SKIP_UCODE_DOWNLOAD)
/* The following array contains the microcode image to download
 * to the device. It consists of a series of register writes
 * in the format:
 *    address, data
 */
unsigned short cs4224_kr_ucode_image[] = {
	0x0011, 0x0001, 0x0017, 0x8080, 0x1091, 0x0001, 0x1007, 0x8004, 0x1225,
	0x0008, 0x1238, 0x0000, 0x0301, 0x9000, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x7c0d, 0x0302, 0x0084, 0x0303, 0x7c02, 0x0302,
	0x0088, 0x0303, 0x010d, 0x0302, 0x0060, 0x0303, 0x7c02, 0x0302, 0x0009,
	0x0303, 0x019f, 0x0302, 0x00e0, 0x0303, 0x7c02, 0x0302, 0x008d, 0x0303,
	0x0187, 0x0302, 0x00e0, 0x0303, 0x7c02, 0x0302, 0x00af, 0x0303, 0x01ea,
	0x0302, 0x0060, 0x0303, 0x7c02, 0x0302, 0x002e, 0x0303, 0x016d, 0x0302,
	0x0060, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0x0180, 0x0302, 0x0060,
	0x0303, 0x016d, 0x0302, 0x0060, 0x0303, 0xe805, 0x0302, 0x008d, 0x0303,
	0x4804, 0x0302, 0x008c, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0xc068,
	0x0302, 0x0083, 0x0303, 0x001b, 0x0302, 0x0000, 0x0303, 0xc067, 0x0302,
	0x0083, 0x0303, 0xc168, 0x0302, 0x0003, 0x0303, 0x6893, 0x0302, 0x0083,
	0x0303, 0x000b, 0x0302, 0x00ba, 0x0303, 0x9002, 0x0302, 0x008c, 0x0303,
	0x000f, 0x0302, 0x003a, 0x0303, 0x03f6, 0x0302, 0x0003, 0x0303, 0x9002,
	0x0302, 0x008a, 0x0303, 0x6793, 0x0302, 0x0083, 0x0303, 0x9006, 0x0302,
	0x0009, 0x0303, 0x0708, 0x0302, 0x003a, 0x0303, 0x03f5, 0x0302, 0x0003,
	0x0303, 0x050b, 0x0302, 0x00ba, 0x0303, 0x03f6, 0x0302, 0x0003, 0x0303,
	0x002d, 0x0302, 0x0000, 0x0303, 0x9003, 0x0302, 0x0028, 0x0303, 0xe2f5,
	0x0302, 0x0003, 0x0303, 0x002d, 0x0302, 0x0000, 0x0303, 0xc8f5, 0x0302,
	0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xe605, 0x0302, 0x002f,
	0x0303, 0xe6f5, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303,
	0xe643, 0x0302, 0x009f, 0x0303, 0x03f6, 0x0302, 0x0003, 0x0303, 0xe7f9,
	0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0x8503, 0x0302,
	0x0086, 0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0x037a, 0x0302, 0x009e,
	0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x003c, 0x0302, 0x0041, 0x0303,
	0x7a5f, 0x0302, 0x009e, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x0045,
	0x0302, 0x0080, 0x0303, 0x7b75, 0x0302, 0x0097, 0x0303, 0x004e, 0x0302,
	0x0041, 0x0303, 0xf970, 0x0302, 0x001c, 0x0303, 0x03f9, 0x0302, 0x0003,
	0x0303, 0x757b, 0x0302, 0x009e, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303,
	0x7b5f, 0x0302, 0x0099, 0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0x003c,
	0x0302, 0x0000, 0x0303, 0x7b75, 0x0302, 0x0097, 0x0303, 0x004e, 0x0302,
	0x0041, 0x0303, 0xf970, 0x0302, 0x009d, 0x0303, 0x03f9, 0x0302, 0x0003,
	0x0303, 0x757b, 0x0302, 0x009e, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303,
	0x5f7b, 0x0302, 0x001e, 0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0x0045,
	0x0302, 0x0080, 0x0303, 0xcc60, 0x0302, 0x0003, 0x0303, 0xdb68, 0x0302,
	0x0083, 0x0303, 0x000b, 0x0302, 0x00ba, 0x0303, 0x03db, 0x0302, 0x0011,
	0x0303, 0x03db, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303,
	0x000f, 0x0302, 0x005a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0xc375,
	0x0302, 0x0083, 0x0303, 0x00f6, 0x0302, 0x0060, 0x0303, 0xf467, 0x0302,
	0x0003, 0x0303, 0x4df4, 0x0302, 0x0097, 0x0303, 0x0056, 0x0302, 0x0041,
	0x0303, 0x0017, 0x0302, 0x009a, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303,
	0x00f6, 0x0302, 0x0060, 0x0303, 0xf40f, 0x0302, 0x001f, 0x0303, 0x2267,
	0x0302, 0x0002, 0x0303, 0x4df4, 0x0302, 0x0097, 0x0303, 0x005b, 0x0302,
	0x00c1, 0x0303, 0x002b, 0x0302, 0x009a, 0x0303, 0x0375, 0x0302, 0x0083,
	0x0303, 0x00f6, 0x0302, 0x0060, 0x0303, 0xf472, 0x0302, 0x0083, 0x0303,
	0x4df4, 0x0302, 0x0097, 0x0303, 0x0062, 0x0302, 0x00c1, 0x0303, 0x003f,
	0x0302, 0x009a, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x00f6, 0x0302,
	0x0060, 0x0303, 0xf40f, 0x0302, 0x001f, 0x0303, 0x2272, 0x0302, 0x0082,
	0x0303, 0x4df4, 0x0302, 0x0097, 0x0303, 0x0068, 0x0302, 0x00c1, 0x0303,
	0x720f, 0x0302, 0x009f, 0x0303, 0x4f75, 0x0302, 0x0082, 0x0303, 0x7567,
	0x0302, 0x0011, 0x0303, 0x03ec, 0x0302, 0x0083, 0x0303, 0x68db, 0x0302,
	0x0083, 0x0303, 0xc081, 0x0302, 0x0003, 0x0303, 0x00e6, 0x0302, 0x00ba,
	0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x03b2, 0x0302, 0x0003, 0x0303,
	0xcffe, 0x0302, 0x0083, 0x0303, 0xc2fd, 0x0302, 0x0003, 0x0303, 0x9015,
	0x0302, 0x000b, 0x0303, 0x780f, 0x0302, 0x009f, 0x0303, 0x0106, 0x0302,
	0x00e0, 0x0303, 0x0083, 0x0302, 0x0080, 0x0303, 0xfd03, 0x0302, 0x0004,
	0x0303, 0xc0fd, 0x0302, 0x0083, 0x0303, 0x0083, 0x0302, 0x0080, 0x0303,
	0xfe70, 0x0302, 0x001d, 0x0303, 0x03fe, 0x0302, 0x0083, 0x0303, 0xc1ae,
	0x0302, 0x0003, 0x0303, 0xc2ae, 0x0302, 0x0003, 0x0303, 0x6f0f, 0x0302,
	0x009f, 0x0303, 0x0106, 0x0302, 0x00e0, 0x0303, 0xc0ae, 0x0302, 0x0083,
	0x0303, 0xb100, 0x0302, 0x00af, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x01f4, 0x0302, 0x003a, 0x0303, 0xb003,
	0x0302, 0x0097, 0x0303, 0xfe02, 0x0302, 0x0009, 0x0303, 0x007e, 0x0302,
	0x0041, 0x0303, 0x00c0, 0x0302, 0x003a, 0x0303, 0x03f4, 0x0302, 0x0083,
	0x0303, 0xc0ea, 0x0302, 0x0083, 0x0303, 0x8403, 0x0302, 0x008d, 0x0303,
	0xc067, 0x0302, 0x0083, 0x0303, 0x01ea, 0x0302, 0x0060, 0x0303, 0xe805,
	0x0302, 0x000a, 0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x9000, 0x0302,
	0x005a, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0x0354, 0x0302, 0x0083,
	0x0303, 0x9000, 0x0302, 0x0018, 0x0303, 0x0390, 0x0302, 0x0003, 0x0303,
	0xe812, 0x0302, 0x0024, 0x0303, 0xe811, 0x0302, 0x00aa, 0x0303, 0xe809,
	0x0302, 0x0009, 0x0303, 0x810f, 0x0302, 0x009f, 0x0303, 0x037d, 0x0302,
	0x001b, 0x0303, 0x0370, 0x0302, 0x001c, 0x0303, 0x0370, 0x0302, 0x0019,
	0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0x7d81, 0x0302, 0x001b, 0x0303,
	0x037d, 0x0302, 0x0017, 0x0303, 0x009f, 0x0302, 0x0021, 0x0303, 0x800f,
	0x0302, 0x00da, 0x0303, 0x0106, 0x0302, 0x00e0, 0x0303, 0x810f, 0x0302,
	0x009f, 0x0303, 0x037d, 0x0302, 0x001b, 0x0303, 0x0370, 0x0302, 0x009d,
	0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0x8184, 0x0302, 0x0028, 0x0303,
	0xc280, 0x0302, 0x0003, 0x0303, 0x07d0, 0x0302, 0x003a, 0x0303, 0x036f,
	0x0302, 0x0011, 0x0303, 0x0106, 0x0302, 0x00e0, 0x0303, 0xc380, 0x0302,
	0x0083, 0x0303, 0x9c40, 0x0302, 0x00ba, 0x0303, 0x036f, 0x0302, 0x0011,
	0x0303, 0x0106, 0x0302, 0x00e0, 0x0303, 0xc280, 0x0302, 0x0003, 0x0303,
	0x00c0, 0x0302, 0x009a, 0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x8503,
	0x0302, 0x0086, 0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0xd294, 0x0302,
	0x0083, 0x0303, 0xd295, 0x0302, 0x0003, 0x0303, 0xb2ef, 0x0302, 0x0083,
	0x0303, 0xc067, 0x0302, 0x0083, 0x0303, 0x01ea, 0x0302, 0x0060, 0x0303,
	0xd100, 0x0302, 0x0090, 0x0303, 0x03e8, 0x0302, 0x001b, 0x0303, 0x03e8,
	0x0302, 0x0003, 0x0303, 0xc0e1, 0x0302, 0x0003, 0x0303, 0x8503, 0x0302,
	0x0086, 0x0303, 0x03c8, 0x0302, 0x009d, 0x0303, 0x03cf, 0x0302, 0x001b,
	0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0xc3df, 0x0302, 0x0083, 0x0303,
	0xe803, 0x0302, 0x0003, 0x0303, 0x03c8, 0x0302, 0x009d, 0x0303, 0x03cf,
	0x0302, 0x001b, 0x0303, 0x0372, 0x0302, 0x0003, 0x0303, 0xc04c, 0x0302,
	0x0083, 0x0303, 0x0053, 0x0302, 0x003a, 0x0303, 0x7503, 0x0302, 0x0097,
	0x0303, 0x00d6, 0x0302, 0x00c1, 0x0303, 0x7572, 0x0302, 0x0019, 0x0303,
	0x0375, 0x0302, 0x0083, 0x0303, 0xdf00, 0x0302, 0x0098, 0x0303, 0x03df,
	0x0302, 0x0083, 0x0303, 0x00ce, 0x0302, 0x0080, 0x0303, 0xe803, 0x0302,
	0x0003, 0x0303, 0x03c9, 0x0302, 0x001d, 0x0303, 0x03cc, 0x0302, 0x001b,
	0x0303, 0x03df, 0x0302, 0x0097, 0x0303, 0x00dc, 0x0302, 0x00c1, 0x0303,
	0x03df, 0x0302, 0x0083, 0x0303, 0xe50f, 0x0302, 0x001f, 0x0303, 0x0358,
	0x0302, 0x009b, 0x0303, 0xdf03, 0x0302, 0x009e, 0x0303, 0x03cf, 0x0302,
	0x001b, 0x0303, 0x03df, 0x0302, 0x0083, 0x0303, 0xe802, 0x0302, 0x008f,
	0x0303, 0x00e8, 0x0302, 0x0000, 0x0303, 0x9003, 0x0302, 0x008b, 0x0303,
	0xdc0f, 0x0302, 0x001f, 0x0303, 0x00e7, 0x0302, 0x0000, 0x0303, 0x0021,
	0x0302, 0x003a, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303, 0xe36e, 0x0302,
	0x001b, 0x0303, 0x03c5, 0x0302, 0x009c, 0x0303, 0x035d, 0x0302, 0x0083,
	0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xf663, 0x0302, 0x009f, 0x0303,
	0x0375, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xf501,
	0x0302, 0x001f, 0x0303, 0x0375, 0x0302, 0x0019, 0x0303, 0x757f, 0x0302,
	0x0003, 0x0303, 0xc0eb, 0x0302, 0x0003, 0x0303, 0xc07c, 0x0302, 0x0083,
	0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0x0180, 0x0302, 0x0060, 0x0303,
	0xcbf4, 0x0302, 0x0003, 0x0303, 0x7580, 0x0302, 0x0003, 0x0303, 0xf400,
	0x0302, 0x0098, 0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x4dd1, 0x0302,
	0x0011, 0x0303, 0x0386, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x8003,
	0x0302, 0x00a4, 0x0303, 0x838a, 0x0302, 0x002b, 0x0303, 0x0105, 0x0302,
	0x0080, 0x0303, 0x838c, 0x0302, 0x00af, 0x0303, 0x0000, 0x0302, 0x00e1,
	0x0303, 0x0386, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0xe821, 0x0302, 0x009f, 0x0303, 0x030c, 0x0302, 0x008b,
	0x0303, 0x8503, 0x0302, 0x0086, 0x0303, 0x038b, 0x0302, 0x0003, 0x0303,
	0x4c00, 0x0302, 0x00ba, 0x0303, 0x038b, 0x0302, 0x0017, 0x0303, 0x011a,
	0x0302, 0x0041, 0x0303, 0x4e00, 0x0302, 0x003a, 0x0303, 0x038b, 0x0302,
	0x0017, 0x0303, 0x011b, 0x0302, 0x00c1, 0x0303, 0x1000, 0x0302, 0x00ba,
	0x0303, 0x03e6, 0x0302, 0x0083, 0x0303, 0x011b, 0x0302, 0x0080, 0x0303,
	0xc0e6, 0x0302, 0x0083, 0x0303, 0xc071, 0x0302, 0x0003, 0x0303, 0xc06a,
	0x0302, 0x0003, 0x0303, 0xc06b, 0x0302, 0x0083, 0x0303, 0xc07f, 0x0302,
	0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xe977, 0x0302, 0x0083,
	0x0303, 0xc16a, 0x0302, 0x0083, 0x0303, 0xc17f, 0x0302, 0x0017, 0x0303,
	0x0162, 0x0302, 0x00a0, 0x0303, 0xc26a, 0x0302, 0x0083, 0x0303, 0xc27f,
	0x0302, 0x0017, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc36a, 0x0302,
	0x0003, 0x0303, 0xc37f, 0x0302, 0x0097, 0x0303, 0xc0e6, 0x0302, 0x0083,
	0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc16b, 0x0302, 0x0003, 0x0303,
	0xc171, 0x0302, 0x0083, 0x0303, 0xc47f, 0x0302, 0x0017, 0x0303, 0x0162,
	0x0302, 0x00a0, 0x0303, 0xc46a, 0x0302, 0x0083, 0x0303, 0xc57f, 0x0302,
	0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc26b, 0x0302, 0x0003,
	0x0303, 0xc67f, 0x0302, 0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303,
	0xc56a, 0x0302, 0x0003, 0x0303, 0xc77f, 0x0302, 0x0017, 0x0303, 0x0162,
	0x0302, 0x00a0, 0x0303, 0xc36b, 0x0302, 0x0083, 0x0303, 0xc87f, 0x0302,
	0x0017, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc66a, 0x0302, 0x0003,
	0x0303, 0x0009, 0x0302, 0x003a, 0x0303, 0x037f, 0x0302, 0x0097, 0x0303,
	0x0162, 0x0302, 0x00a0, 0x0303, 0xc46b, 0x0302, 0x0003, 0x0303, 0x000a,
	0x0302, 0x003a, 0x0303, 0x037f, 0x0302, 0x0097, 0x0303, 0x0162, 0x0302,
	0x00a0, 0x0303, 0xc76a, 0x0302, 0x0083, 0x0303, 0x000b, 0x0302, 0x00ba,
	0x0303, 0x037f, 0x0302, 0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303,
	0xc56b, 0x0302, 0x0083, 0x0303, 0x000c, 0x0302, 0x003a, 0x0303, 0x037f,
	0x0302, 0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc86a, 0x0302,
	0x0083, 0x0303, 0x000d, 0x0302, 0x00ba, 0x0303, 0x037f, 0x0302, 0x0097,
	0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc66b, 0x0302, 0x0083, 0x0303,
	0x000e, 0x0302, 0x00ba, 0x0303, 0x037f, 0x0302, 0x0097, 0x0303, 0x0162,
	0x0302, 0x00a0, 0x0303, 0x0009, 0x0302, 0x003a, 0x0303, 0x036a, 0x0302,
	0x0003, 0x0303, 0x000f, 0x0302, 0x003a, 0x0303, 0x037f, 0x0302, 0x0097,
	0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0xc76b, 0x0302, 0x0003, 0x0303,
	0xc97f, 0x0302, 0x0097, 0x0303, 0x0162, 0x0302, 0x00a0, 0x0303, 0x000a,
	0x0302, 0x003a, 0x0303, 0x036a, 0x0302, 0x0003, 0x0303, 0x0011, 0x0302,
	0x003a, 0x0303, 0x037f, 0x0302, 0x0097, 0x0303, 0x0162, 0x0302, 0x00a0,
	0x0303, 0x000b, 0x0302, 0x00ba, 0x0303, 0x036a, 0x0302, 0x0003, 0x0303,
	0x6ac8, 0x0302, 0x001c, 0x0303, 0x038b, 0x0302, 0x0003, 0x0303, 0x778b,
	0x0302, 0x0011, 0x0303, 0x03f5, 0x0302, 0x0003, 0x0303, 0x6bc8, 0x0302,
	0x009c, 0x0303, 0x038b, 0x0302, 0x0003, 0x0303, 0xcc0f, 0x0302, 0x009f,
	0x0303, 0x03f6, 0x0302, 0x001b, 0x0303, 0x038b, 0x0302, 0x0011, 0x0303,
	0x03f6, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0xb612,
	0x0302, 0x002c, 0x0303, 0xc068, 0x0302, 0x0083, 0x0303, 0xb60f, 0x0302,
	0x001f, 0x0303, 0x03c9, 0x0302, 0x009c, 0x0303, 0x0302, 0x0302, 0x002f,
	0x0303, 0xc168, 0x0302, 0x0003, 0x0303, 0x0055, 0x0302, 0x003a, 0x0303,
	0x03b8, 0x0302, 0x001b, 0x0303, 0x0178, 0x0302, 0x0020, 0x0303, 0x1891,
	0x0302, 0x001a, 0x0303, 0x0179, 0x0302, 0x0000, 0x0303, 0x1091, 0x0302,
	0x009a, 0x0303, 0x68b5, 0x0302, 0x0003, 0x0303, 0x03b4, 0x0302, 0x0003,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0xb581, 0x0302, 0x002f, 0x0303, 0xb591, 0x0302, 0x00aa, 0x0303, 0x0000,
	0x0302, 0x00e1, 0x0303, 0x6791, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0007, 0x0302, 0x001a, 0x0303, 0x0381,
	0x0302, 0x0003, 0x0303, 0x0027, 0x0302, 0x009a, 0x0303, 0x0381, 0x0302,
	0x0003, 0x0303, 0x0067, 0x0302, 0x001a, 0x0303, 0x0381, 0x0302, 0x0003,
	0x0303, 0x00e7, 0x0302, 0x009a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303,
	0x01e7, 0x0302, 0x001a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0x7e86,
	0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x00e7, 0x0302, 0x009a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303,
	0x0067, 0x0302, 0x001a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0x0027,
	0x0302, 0x009a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0xc781, 0x0302,
	0x0083, 0x0303, 0xc081, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x00e1,
	0x0303, 0xac0a, 0x0302, 0x00af, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0xb211, 0x0302, 0x0084, 0x0303, 0x6eac, 0x0302, 0x009b, 0x0303, 0xc803,
	0x0302, 0x0097, 0x0303, 0x01b2, 0x0302, 0x00c1, 0x0303, 0xb20f, 0x0302,
	0x009f, 0x0303, 0x0372, 0x0302, 0x001e, 0x0303, 0x03b2, 0x0302, 0x0003,
	0x0303, 0x01b1, 0x0302, 0x0080, 0x0303, 0xb20f, 0x0302, 0x009f, 0x0303,
	0x03cf, 0x0302, 0x0017, 0x0303, 0x01b2, 0x0302, 0x0020, 0x0303, 0xc80f,
	0x0302, 0x001f, 0x0303, 0x03ac, 0x0302, 0x0017, 0x0303, 0x01b2, 0x0302,
	0x00c1, 0x0303, 0xb200, 0x0302, 0x0018, 0x0303, 0x03b2, 0x0302, 0x0003,
	0x0303, 0x6d95, 0x0302, 0x0083, 0x0303, 0xb275, 0x0302, 0x0083, 0x0303,
	0x73b2, 0x0302, 0x0097, 0x0303, 0x01c7, 0x0302, 0x0041, 0x0303, 0xfe09,
	0x0302, 0x000e, 0x0303, 0xfd04, 0x0302, 0x0084, 0x0303, 0xcc60, 0x0302,
	0x0017, 0x0303, 0x01c3, 0x0302, 0x00c1, 0x0303, 0x01e8, 0x0302, 0x0080,
	0x0303, 0xc2fd, 0x0302, 0x0003, 0x0303, 0x75cb, 0x0302, 0x001e, 0x0303,
	0x03b2, 0x0302, 0x0003, 0x0303, 0x01e8, 0x0302, 0x0080, 0x0303, 0xfe70,
	0x0302, 0x009c, 0x0303, 0x03fe, 0x0302, 0x0083, 0x0303, 0x75c8, 0x0302,
	0x001e, 0x0303, 0x03b2, 0x0302, 0x0003, 0x0303, 0x01e8, 0x0302, 0x0080,
	0x0303, 0x6000, 0x0302, 0x0018, 0x0303, 0x03cf, 0x0302, 0x001b, 0x0303,
	0x0360, 0x0302, 0x0003, 0x0303, 0x01e8, 0x0302, 0x0080, 0x0303, 0xcb75,
	0x0302, 0x0017, 0x0303, 0x01dd, 0x0302, 0x00c1, 0x0303, 0x740f, 0x0302,
	0x009f, 0x0303, 0xfe02, 0x0302, 0x008e, 0x0303, 0x03c9, 0x0302, 0x001e,
	0x0303, 0xb203, 0x0302, 0x0017, 0x0303, 0x01e8, 0x0302, 0x00c1, 0x0303,
	0x6005, 0x0302, 0x0084, 0x0303, 0x60c1, 0x0302, 0x009e, 0x0303, 0x03cf,
	0x0302, 0x001b, 0x0303, 0x0360, 0x0302, 0x0003, 0x0303, 0x01e8, 0x0302,
	0x0080, 0x0303, 0xfd05, 0x0302, 0x0089, 0x0303, 0xc0fd, 0x0302, 0x0083,
	0x0303, 0xcb75, 0x0302, 0x0099, 0x0303, 0x03b2, 0x0302, 0x0003, 0x0303,
	0x01e8, 0x0302, 0x0080, 0x0303, 0xfe10, 0x0302, 0x0009, 0x0303, 0xfe70,
	0x0302, 0x001d, 0x0303, 0x03fe, 0x0302, 0x0083, 0x0303, 0x75c8, 0x0302,
	0x0099, 0x0303, 0x03b2, 0x0302, 0x0003, 0x0303, 0x000e, 0x0302, 0x001a,
	0x0303, 0x03b2, 0x0302, 0x0017, 0x0303, 0x01e2, 0x0302, 0x00c1, 0x0303,
	0x6008, 0x0302, 0x0004, 0x0303, 0x01c3, 0x0302, 0x0080, 0x0303, 0x75c5,
	0x0302, 0x0097, 0x0303, 0x01e8, 0x0302, 0x00c1, 0x0303, 0x00d0, 0x0302,
	0x001a, 0x0303, 0x6003, 0x0302, 0x0017, 0x0303, 0x01cf, 0x0302, 0x00c1,
	0x0303, 0x6098, 0x0302, 0x0004, 0x0303, 0xb2ef, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0x8475, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x7572, 0x0302, 0x0083, 0x0303, 0x0016, 0x0302,
	0x00ba, 0x0303, 0x8403, 0x0302, 0x001d, 0x0303, 0x0367, 0x0302, 0x0091,
	0x0303, 0x0384, 0x0302, 0x0003, 0x0303, 0x8475, 0x0302, 0x0083, 0x0303,
	0x720f, 0x0302, 0x009f, 0x0303, 0x0375, 0x0302, 0x0097, 0x0303, 0x01ec,
	0x0302, 0x0021, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x7c02,
	0x0302, 0x0088, 0x0303, 0x0085, 0x0302, 0x0080, 0x0303, 0x7c02, 0x0302,
	0x0009, 0x0303, 0x005d, 0x0302, 0x0080, 0x0303, 0x7c02, 0x0302, 0x008d,
	0x0303, 0x004d, 0x0302, 0x0000, 0x0303, 0x7c02, 0x0302, 0x00af, 0x0303,
	0x01a6, 0x0302, 0x0080, 0x0303, 0x7c02, 0x0302, 0x002e, 0x0303, 0x0015,
	0x0302, 0x0080, 0x0303, 0xb606, 0x0302, 0x002f, 0x0303, 0x4000, 0x0302,
	0x007a, 0x0303, 0x037c, 0x0302, 0x0083, 0x0303, 0xc00f, 0x0302, 0x009f,
	0x0303, 0x01d1, 0x0302, 0x00e0, 0x0303, 0xc07c, 0x0302, 0x0083, 0x0303,
	0xea19, 0x0302, 0x0088, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xec01,
	0x0302, 0x009f, 0x0303, 0x01fc, 0x0302, 0x00e0, 0x0303, 0xc380, 0x0302,
	0x0083, 0x0303, 0xaaa0, 0x0302, 0x003a, 0x0303, 0x01d8, 0x0302, 0x00e0,
	0x0303, 0xf10f, 0x0302, 0x001f, 0x0303, 0xf403, 0x0302, 0x009e, 0x0303,
	0x03f4, 0x0302, 0x0083, 0x0303, 0x0002, 0x0302, 0x00ba, 0x0303, 0x01df,
	0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xec61, 0x0302,
	0x009f, 0x0303, 0x01fc, 0x0302, 0x00e0, 0x0303, 0x0017, 0x0302, 0x003a,
	0x0303, 0x0380, 0x0302, 0x0083, 0x0303, 0xaaa0, 0x0302, 0x003a, 0x0303,
	0x01d8, 0x0302, 0x00e0, 0x0303, 0xf10f, 0x0302, 0x001f, 0x0303, 0xf403,
	0x0302, 0x0019, 0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x0008, 0x0302,
	0x00ba, 0x0303, 0x01df, 0x0302, 0x0060, 0x0303, 0x0047, 0x0302, 0x0000,
	0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xec41, 0x0302, 0x001f, 0x0303,
	0x01fc, 0x0302, 0x00e0, 0x0303, 0x002b, 0x0302, 0x003a, 0x0303, 0x0380,
	0x0302, 0x0083, 0x0303, 0xaaa0, 0x0302, 0x003a, 0x0303, 0x01d8, 0x0302,
	0x00e0, 0x0303, 0xf10f, 0x0302, 0x001f, 0x0303, 0xf403, 0x0302, 0x0019,
	0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x0002, 0x0302, 0x00ba, 0x0303,
	0x01df, 0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xec21,
	0x0302, 0x001f, 0x0303, 0x01fc, 0x0302, 0x00e0, 0x0303, 0x003f, 0x0302,
	0x003a, 0x0303, 0x0380, 0x0302, 0x0083, 0x0303, 0xaaa0, 0x0302, 0x003a,
	0x0303, 0x01d8, 0x0302, 0x00e0, 0x0303, 0xf10f, 0x0302, 0x001f, 0x0303,
	0xf403, 0x0302, 0x009e, 0x0303, 0x03f4, 0x0302, 0x0083, 0x0303, 0x0008,
	0x0302, 0x00ba, 0x0303, 0x01df, 0x0302, 0x0060, 0x0303, 0xeb07, 0x0302,
	0x002f, 0x0303, 0x8306, 0x0302, 0x000d, 0x0303, 0xe805, 0x0302, 0x0009,
	0x0303, 0xcb7c, 0x0302, 0x0003, 0x0303, 0xc00f, 0x0302, 0x009f, 0x0303,
	0x01d1, 0x0302, 0x00e0, 0x0303, 0xc07c, 0x0302, 0x0083, 0x0303, 0xc081,
	0x0302, 0x0003, 0x0303, 0x8306, 0x0302, 0x00a4, 0x0303, 0xea6d, 0x0302,
	0x0017, 0x0303, 0x0055, 0x0302, 0x0041, 0x0303, 0xc075, 0x0302, 0x0083,
	0x0303, 0xc872, 0x0302, 0x0083, 0x0303, 0x0058, 0x0302, 0x0080, 0x0303,
	0x003f, 0x0302, 0x003a, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0xc172,
	0x0302, 0x0083, 0x0303, 0x75ea, 0x0302, 0x001b, 0x0303, 0x005d, 0x0302,
	0x00a1, 0x0303, 0xc27c, 0x0302, 0x0003, 0x0303, 0xc00f, 0x0302, 0x009f,
	0x0303, 0x01d1, 0x0302, 0x00e0, 0x0303, 0xc07c, 0x0302, 0x0083, 0x0303,
	0x003f, 0x0302, 0x003a, 0x0303, 0x03ea, 0x0302, 0x009b, 0x0303, 0x007a,
	0x0302, 0x00a1, 0x0303, 0xd16e, 0x0302, 0x0091, 0x0303, 0xeb03, 0x0302,
	0x001b, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x8503, 0x0302, 0x0086,
	0x0303, 0x035f, 0x0302, 0x001e, 0x0303, 0x0072, 0x0302, 0x0041, 0x0303,
	0x8503, 0x0302, 0x0086, 0x0303, 0x5f03, 0x0302, 0x001e, 0x0303, 0x7b03,
	0x0302, 0x0017, 0x0303, 0x007a, 0x0302, 0x00c1, 0x0303, 0xf970, 0x0302,
	0x009d, 0x0303, 0x03f9, 0x0302, 0x0003, 0x0303, 0x5f7b, 0x0302, 0x001e,
	0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0x01ef, 0x0302, 0x0060, 0x0303,
	0x75eb, 0x0302, 0x0083, 0x0303, 0x007a, 0x0302, 0x0080, 0x0303, 0x7b03,
	0x0302, 0x0017, 0x0303, 0x007a, 0x0302, 0x00c1, 0x0303, 0xf970, 0x0302,
	0x001c, 0x0303, 0x03f9, 0x0302, 0x0003, 0x0303, 0x7b5f, 0x0302, 0x0099,
	0x0303, 0x035f, 0x0302, 0x0003, 0x0303, 0x01ef, 0x0302, 0x0060, 0x0303,
	0x75eb, 0x0302, 0x0083, 0x0303, 0xe651, 0x0302, 0x002b, 0x0303, 0x00ff,
	0x0302, 0x003a, 0x0303, 0x03ea, 0x0302, 0x009b, 0x0303, 0x0101, 0x0302,
	0x0021, 0x0303, 0xc071, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0093,
	0x0303, 0xf501, 0x0302, 0x001f, 0x0303, 0x0377, 0x0302, 0x0003, 0x0303,
	0xc17c, 0x0302, 0x0003, 0x0303, 0xc00f, 0x0302, 0x009f, 0x0303, 0x01d1,
	0x0302, 0x00e0, 0x0303, 0xc07c, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x0093, 0x0303, 0xf601, 0x0302, 0x001f, 0x0303, 0x038b, 0x0302, 0x0003,
	0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xdc43, 0x0302, 0x009f, 0x0303,
	0x0375, 0x0302, 0x0083, 0x0303, 0xa96e, 0x0302, 0x009b, 0x0303, 0x7503,
	0x0302, 0x0097, 0x0303, 0x009c, 0x0302, 0x0041, 0x0303, 0xa908, 0x0302,
	0x00ab, 0x0303, 0x8b0c, 0x0302, 0x0084, 0x0303, 0x0006, 0x0302, 0x009a,
	0x0303, 0x038b, 0x0302, 0x0017, 0x0303, 0x009c, 0x0302, 0x0041, 0x0303,
	0xf670, 0x0302, 0x009e, 0x0303, 0x03f6, 0x0302, 0x0003, 0x0303, 0x009c,
	0x0302, 0x0000, 0x0303, 0x8b0f, 0x0302, 0x009f, 0x0303, 0xcc03, 0x0302,
	0x0017, 0x0303, 0x009c, 0x0302, 0x00a0, 0x0303, 0xf600, 0x0302, 0x0018,
	0x0303, 0x03f6, 0x0302, 0x0003, 0x0303, 0xcb0f, 0x0302, 0x001f, 0x0303,
	0x03ef, 0x0302, 0x0097, 0x0303, 0x00a5, 0x0302, 0x0041, 0x0303, 0x0008,
	0x0302, 0x001a, 0x0303, 0x03e9, 0x0302, 0x0097, 0x0303, 0x00af, 0x0302,
	0x0041, 0x0303, 0xe970, 0x0302, 0x001e, 0x0303, 0x03e9, 0x0302, 0x0083,
	0x0303, 0x00af, 0x0302, 0x0000, 0x0303, 0xc90f, 0x0302, 0x009f, 0x0303,
	0xef03, 0x0302, 0x0097, 0x0303, 0x00af, 0x0302, 0x0041, 0x0303, 0x000f,
	0x0302, 0x009a, 0x0303, 0x03e2, 0x0302, 0x001b, 0x0303, 0xe903, 0x0302,
	0x0097, 0x0303, 0x00af, 0x0302, 0x0041, 0x0303, 0xe900, 0x0302, 0x0098,
	0x0303, 0x03e9, 0x0302, 0x0083, 0x0303, 0x00c5, 0x0302, 0x0000, 0x0303,
	0x0000, 0x0302, 0x0093, 0x0303, 0xf563, 0x0302, 0x009f, 0x0303, 0x0368,
	0x0302, 0x0083, 0x0303, 0xed07, 0x0302, 0x0004, 0x0303, 0x0000, 0x0302,
	0x0093, 0x0303, 0xde03, 0x0302, 0x0006, 0x0303, 0x03e1, 0x0302, 0x001e,
	0x0303, 0x03e6, 0x0302, 0x0019, 0x0303, 0x0361, 0x0302, 0x0097, 0x0303,
	0x00be, 0x0302, 0x0041, 0x0303, 0xe977, 0x0302, 0x0097, 0x0303, 0x00cb,
	0x0302, 0x0020, 0x0303, 0xf570, 0x0302, 0x009e, 0x0303, 0x03f5, 0x0302,
	0x0003, 0x0303, 0x00cb, 0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0093,
	0x0303, 0xde43, 0x0302, 0x001f, 0x0303, 0x03e1, 0x0302, 0x001e, 0x0303,
	0x03e6, 0x0302, 0x0019, 0x0303, 0x6103, 0x0302, 0x0097, 0x0303, 0x00cb,
	0x0302, 0x00c1, 0x0303, 0x6807, 0x0302, 0x0005, 0x0303, 0xe20f, 0x0302,
	0x009f, 0x0303, 0x03cc, 0x0302, 0x001b, 0x0303, 0x0377, 0x0302, 0x0017,
	0x0303, 0x00cb, 0x0302, 0x0020, 0x0303, 0xf500, 0x0302, 0x0018, 0x0303,
	0x03f5, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xf141,
	0x0302, 0x001f, 0x0303, 0x0372, 0x0302, 0x0003, 0x0303, 0xeb33, 0x0302,
	0x00af, 0x0303, 0x8332, 0x0302, 0x008d, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0xe802, 0x0302, 0x00ad, 0x0303, 0x00e0, 0x0302, 0x0080, 0x0303,
	0x7971, 0x0302, 0x009e, 0x0303, 0x4c03, 0x0302, 0x0019, 0x0303, 0x0375,
	0x0302, 0x0083, 0x0303, 0x7275, 0x0302, 0x0097, 0x0303, 0x00da, 0x0302,
	0x00c1, 0x0303, 0x72fa, 0x0302, 0x0003, 0x0303, 0x00e0, 0x0302, 0x0080,
	0x0303, 0xdc6e, 0x0302, 0x001b, 0x0303, 0x7503, 0x0302, 0x0097, 0x0303,
	0x00df, 0x0302, 0x00c1, 0x0303, 0xdcfa, 0x0302, 0x0083, 0x0303, 0x00e0,
	0x0302, 0x0080, 0x0303, 0x75fa, 0x0302, 0x0083, 0x0303, 0xeb01, 0x0302,
	0x00ae, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xdd75, 0x0302, 0x0086,
	0x0303, 0xed75, 0x0302, 0x0097, 0x0303, 0x00f1, 0x0302, 0x00c1, 0x0303,
	0x68c3, 0x0302, 0x0097, 0x0303, 0x0101, 0x0302, 0x0041, 0x0303, 0x6eeb,
	0x0302, 0x009b, 0x0303, 0x03c3, 0x0302, 0x0099, 0x0303, 0xfa03, 0x0302,
	0x0017, 0x0303, 0x0101, 0x0302, 0x0041, 0x0303, 0xfa4d, 0x0302, 0x0017,
	0x0303, 0x0101, 0x0302, 0x0041, 0x0303, 0xfa00, 0x0302, 0x0018, 0x0303,
	0x03fa, 0x0302, 0x0003, 0x0303, 0x0379, 0x0302, 0x0083, 0x0303, 0x0101,
	0x0302, 0x0000, 0x0303, 0x4c10, 0x0302, 0x0005, 0x0303, 0x0000, 0x0302,
	0x0093, 0x0303, 0xdd43, 0x0302, 0x001f, 0x0303, 0x0375, 0x0302, 0x0083,
	0x0303, 0x75ed, 0x0302, 0x0097, 0x0303, 0x0101, 0x0302, 0x0041, 0x0303,
	0x6eeb, 0x0302, 0x009b, 0x0303, 0x03c2, 0x0302, 0x009e, 0x0303, 0x03fa,
	0x0302, 0x0017, 0x0303, 0x0101, 0x0302, 0x0041, 0x0303, 0xdc6e, 0x0302,
	0x001b, 0x0303, 0x03fa, 0x0302, 0x0017, 0x0303, 0x0101, 0x0302, 0x0041,
	0x0303, 0xfa70, 0x0302, 0x009e, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303,
	0x0379, 0x0302, 0x0083, 0x0303, 0x00ff, 0x0302, 0x003a, 0x0303, 0x8302,
	0x0302, 0x0024, 0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0x03ea, 0x0302,
	0x009b, 0x0303, 0x0116, 0x0302, 0x0021, 0x0303, 0xa06e, 0x0302, 0x009b,
	0x0303, 0xa002, 0x0302, 0x00ab, 0x0303, 0xc003, 0x0302, 0x001e, 0x0303,
	0x0367, 0x0302, 0x0083, 0x0303, 0xa16e, 0x0302, 0x001b, 0x0303, 0xa102,
	0x0302, 0x002b, 0x0303, 0xc003, 0x0302, 0x001e, 0x0303, 0x0367, 0x0302,
	0x0019, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x0303, 0x0302, 0x002b,
	0x0303, 0x675d, 0x0302, 0x0099, 0x0303, 0x035d, 0x0302, 0x0083, 0x0303,
	0x5dc5, 0x0302, 0x009d, 0x0303, 0x03ed, 0x0302, 0x0003, 0x0303, 0x5d03,
	0x0302, 0x009e, 0x0303, 0x035d, 0x0302, 0x0083, 0x0303, 0x8503, 0x0302,
	0x0086, 0x0303, 0x038b, 0x0302, 0x0003, 0x0303, 0x3e00, 0x0302, 0x00ba,
	0x0303, 0x038b, 0x0302, 0x0017, 0x0303, 0x0120, 0x0302, 0x0041, 0x0303,
	0x4400, 0x0302, 0x003a, 0x0303, 0x038b, 0x0302, 0x0017, 0x0303, 0x0125,
	0x0302, 0x0041, 0x0303, 0xc0e1, 0x0302, 0x0003, 0x0303, 0x0125, 0x0302,
	0x0000, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xf663, 0x0302, 0x009f,
	0x0303, 0x038b, 0x0302, 0x0003, 0x0303, 0x0600, 0x0302, 0x003a, 0x0303,
	0x03e1, 0x0302, 0x0003, 0x0303, 0x8311, 0x0302, 0x000d, 0x0303, 0xe821,
	0x0302, 0x009f, 0x0303, 0x030f, 0x0302, 0x000a, 0x0303, 0xec01, 0x0302,
	0x009f, 0x0303, 0x03cb, 0x0302, 0x009e, 0x0303, 0x03c4, 0x0302, 0x0019,
	0x0303, 0x01fc, 0x0302, 0x00e0, 0x0303, 0x780f, 0x0302, 0x009f, 0x0303,
	0x01e9, 0x0302, 0x0060, 0x0303, 0xc380, 0x0302, 0x0083, 0x0303, 0xc1ae,
	0x0302, 0x0003, 0x0303, 0xc2ae, 0x0302, 0x0003, 0x0303, 0x6dd1, 0x0302,
	0x0091, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0xc0ae, 0x0302, 0x0083,
	0x0303, 0xb100, 0x0302, 0x00af, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0xa26e, 0x0302, 0x001b, 0x0303, 0xa206, 0x0302, 0x00ab, 0x0303, 0x03c6,
	0x0302, 0x0099, 0x0303, 0x03c4, 0x0302, 0x001c, 0x0303, 0x6103, 0x0302,
	0x0019, 0x0303, 0x0361, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302, 0x0080,
	0x0303, 0xc603, 0x0302, 0x001e, 0x0303, 0x0140, 0x0302, 0x0041, 0x0303,
	0x0144, 0x0302, 0x0080, 0x0303, 0x03c4, 0x0302, 0x001c, 0x0303, 0x6103,
	0x0302, 0x0019, 0x0303, 0x0361, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302,
	0x0080, 0x0303, 0x61c6, 0x0302, 0x009d, 0x0303, 0x6103, 0x0302, 0x009e,
	0x0303, 0x0361, 0x0302, 0x0083, 0x0303, 0x06ff, 0x0302, 0x003a, 0x0303,
	0x8302, 0x0302, 0x0024, 0x0303, 0x01ff, 0x0302, 0x00ba, 0x0303, 0x03ea,
	0x0302, 0x009b, 0x0303, 0x0163, 0x0302, 0x00a1, 0x0303, 0x8302, 0x0302,
	0x0024, 0x0303, 0xed07, 0x0302, 0x0004, 0x0303, 0x0000, 0x0302, 0x0093,
	0x0303, 0xde03, 0x0302, 0x0006, 0x0303, 0x03e1, 0x0302, 0x001e, 0x0303,
	0x03e6, 0x0302, 0x0019, 0x0303, 0x0361, 0x0302, 0x0097, 0x0303, 0x015a,
	0x0302, 0x00c1, 0x0303, 0x0012, 0x0302, 0x003a, 0x0303, 0x7f03, 0x0302,
	0x0097, 0x0303, 0x0163, 0x0302, 0x00c1, 0x0303, 0x7f00, 0x0302, 0x0098,
	0x0303, 0x037f, 0x0302, 0x0083, 0x0303, 0x0163, 0x0302, 0x0080, 0x0303,
	0x0000, 0x0302, 0x0093, 0x0303, 0xde43, 0x0302, 0x001f, 0x0303, 0x03e1,
	0x0302, 0x001e, 0x0303, 0x03e6, 0x0302, 0x0019, 0x0303, 0x6103, 0x0302,
	0x0097, 0x0303, 0x0163, 0x0302, 0x00c1, 0x0303, 0x7f03, 0x0302, 0x0004,
	0x0303, 0x7f70, 0x0302, 0x001e, 0x0303, 0x037f, 0x0302, 0x0083, 0x0303,
	0xea00, 0x0302, 0x0098, 0x0303, 0x03ea, 0x0302, 0x0083, 0x0303, 0xe075,
	0x0302, 0x0006, 0x0303, 0x7503, 0x0302, 0x0097, 0x0303, 0x019b, 0x0302,
	0x0041, 0x0303, 0xe803, 0x0302, 0x000f, 0x0303, 0xd1eb, 0x0302, 0x0003,
	0x0303, 0x019b, 0x0302, 0x0000, 0x0303, 0xeb30, 0x0302, 0x002b, 0x0303,
	0x7dea, 0x0302, 0x009b, 0x0303, 0x0170, 0x0302, 0x0021, 0x0303, 0xfa00,
	0x0302, 0x0018, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303, 0x8305, 0x0302,
	0x000b, 0x0303, 0x002e, 0x0302, 0x003a, 0x0303, 0x03fa, 0x0302, 0x0017,
	0x0303, 0x0175, 0x0302, 0x00a0, 0x0303, 0x0010, 0x0302, 0x0080, 0x0303,
	0x7dea, 0x0302, 0x0097, 0x0303, 0x01cf, 0x0302, 0x00c1, 0x0303, 0xc90f,
	0x0302, 0x009f, 0x0303, 0xfa03, 0x0302, 0x009c, 0x0303, 0x03df, 0x0302,
	0x0091, 0x0303, 0xfd02, 0x0302, 0x0084, 0x0303, 0x03c1, 0x0302, 0x009e,
	0x0303, 0x03df, 0x0302, 0x0083, 0x0303, 0xc0ea, 0x0302, 0x0083, 0x0303,
	0xd170, 0x0302, 0x009d, 0x0303, 0x03d1, 0x0302, 0x0011, 0x0303, 0x03eb,
	0x0302, 0x0003, 0x0303, 0xdf6e, 0x0302, 0x001b, 0x0303, 0x0375, 0x0302,
	0x0083, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xdf43, 0x0302, 0x009f,
	0x0303, 0x0375, 0x0302, 0x009e, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303,
	0xdc6e, 0x0302, 0x001b, 0x0303, 0xfa03, 0x0302, 0x0017, 0x0303, 0x018b,
	0x0302, 0x00c1, 0x0303, 0xdcfa, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x0093, 0x0303, 0xf141, 0x0302, 0x001f, 0x0303, 0x0372, 0x0302, 0x0003,
	0x0303, 0x03fa, 0x0302, 0x0017, 0x0303, 0x0191, 0x0302, 0x0041, 0x0303,
	0x72fa, 0x0302, 0x0003, 0x0303, 0xfa0f, 0x0302, 0x009f, 0x0303, 0x0379,
	0x0302, 0x0083, 0x0303, 0x03eb, 0x0302, 0x0011, 0x0303, 0x03eb, 0x0302,
	0x0003, 0x0303, 0xd294, 0x0302, 0x0083, 0x0303, 0xd295, 0x0302, 0x0003,
	0x0303, 0xc280, 0x0302, 0x0003, 0x0303, 0x8001, 0x0302, 0x005a, 0x0303,
	0x01e9, 0x0302, 0x0060, 0x0303, 0x0010, 0x0302, 0x0080, 0x0303, 0x8334,
	0x0302, 0x008b, 0x0303, 0x3000, 0x0302, 0x003a, 0x0303, 0xe802, 0x0302,
	0x00ae, 0x0303, 0xe243, 0x0302, 0x001f, 0x0303, 0x03ea, 0x0302, 0x0097,
	0x0303, 0x01cb, 0x0302, 0x0041, 0x0303, 0x8325, 0x0302, 0x0024, 0x0303,
	0xd17c, 0x0302, 0x0083, 0x0303, 0xcb67, 0x0302, 0x0003, 0x0303, 0xc00f,
	0x0302, 0x009f, 0x0303, 0x01d1, 0x0302, 0x00e0, 0x0303, 0xc07c, 0x0302,
	0x0083, 0x0303, 0xe3dd, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0093,
	0x0303, 0xdd75, 0x0302, 0x0086, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303,
	0xdd43, 0x0302, 0x001f, 0x0303, 0x0375, 0x0302, 0x0019, 0x0303, 0x0372,
	0x0302, 0x0003, 0x0303, 0x0370, 0x0302, 0x009d, 0x0303, 0x036e, 0x0302,
	0x009b, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x03ed, 0x0302, 0x0017,
	0x0303, 0x01c6, 0x0302, 0x00c1, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303,
	0xe543, 0x0302, 0x009f, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x6775,
	0x0302, 0x0099, 0x0303, 0x03ed, 0x0302, 0x0017, 0x0303, 0x01bf, 0x0302,
	0x0041, 0x0303, 0x67dd, 0x0302, 0x0019, 0x0303, 0x03dd, 0x0302, 0x0003,
	0x0303, 0x67c9, 0x0302, 0x001c, 0x0303, 0x03dd, 0x0302, 0x0099, 0x0303,
	0x03dd, 0x0302, 0x0003, 0x0303, 0x01c6, 0x0302, 0x00c1, 0x0303, 0xed75,
	0x0302, 0x009e, 0x0303, 0x03dd, 0x0302, 0x0099, 0x0303, 0x03dd, 0x0302,
	0x0003, 0x0303, 0xed75, 0x0302, 0x009e, 0x0303, 0x03c9, 0x0302, 0x009c,
	0x0303, 0x03dd, 0x0302, 0x0099, 0x0303, 0x03dd, 0x0302, 0x0003, 0x0303,
	0xe805, 0x0302, 0x000a, 0x0303, 0xe804, 0x0302, 0x00ab, 0x0303, 0x9000,
	0x0302, 0x00fa, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0x0354, 0x0302,
	0x0083, 0x0303, 0xe804, 0x0302, 0x002a, 0x0303, 0xe803, 0x0302, 0x0024,
	0x0303, 0x0010, 0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0xee0f, 0x0302, 0x009f, 0x0303, 0x01d1, 0x0302, 0x00e0, 0x0303, 0x0391,
	0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x7886, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x7695, 0x0302, 0x0083,
	0x0303, 0x0394, 0x0302, 0x0083, 0x0303, 0x6f86, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0xc094, 0x0302, 0x0083, 0x0303, 0xc095, 0x0302, 0x0003,
	0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0386, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0x8503, 0x0302, 0x0086, 0x0303, 0x038b, 0x0302, 0x0003,
	0x0303, 0xc04c, 0x0302, 0x0083, 0x0303, 0x3213, 0x0302, 0x003a, 0x0303,
	0x8b03, 0x0302, 0x0017, 0x0303, 0x01f7, 0x0302, 0x0041, 0x0303, 0xc24c,
	0x0302, 0x0003, 0x0303, 0x01fb, 0x0302, 0x0000, 0x0303, 0x3d00, 0x0302,
	0x00ba, 0x0303, 0x8b03, 0x0302, 0x0017, 0x0303, 0x01fb, 0x0302, 0x0041,
	0x0303, 0xc14c, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303,
	0xcd03, 0x0302, 0x0019, 0x0303, 0x0360, 0x0302, 0x001e, 0x0303, 0x03f4,
	0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x6204, 0x0302, 0x0084, 0x0303, 0x6207,
	0x0302, 0x0003, 0x0303, 0xc062, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x0081, 0x0303, 0x100e, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0x5383, 0x0302, 0x0085, 0x0303,
	0x0001, 0x0302, 0x00ba, 0x0303, 0x9a4c, 0x0302, 0x001f, 0x0303, 0x039a,
	0x0302, 0x0003, 0x0303, 0x9b83, 0x0302, 0x0088, 0x0303, 0x0002, 0x0302,
	0x00ba, 0x0303, 0x9a0c, 0x0302, 0x009f, 0x0303, 0x039a, 0x0302, 0x0003,
	0x0303, 0x1000, 0x0302, 0x00ba, 0x0303, 0x0392, 0x0302, 0x0083, 0x0303,
	0xc064, 0x0302, 0x0083, 0x0303, 0x0065, 0x0302, 0x003a, 0x0303, 0x0381,
	0x0302, 0x0003, 0x0303, 0x0010, 0x0302, 0x00ba, 0x0303, 0x01d2, 0x0302,
	0x00e0, 0x0303, 0x1092, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x5053, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303,
	0x1004, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x3d3d,
	0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302,
	0x00e0, 0x0303, 0xf00f, 0x0302, 0x009f, 0x0303, 0xfa44, 0x0302, 0x009f,
	0x0303, 0x03f0, 0x0302, 0x0003, 0x0303, 0x001e, 0x0302, 0x003a, 0x0303,
	0x03fa, 0x0302, 0x0003, 0x0303, 0xe000, 0x0302, 0x007a, 0x0303, 0x039b,
	0x0302, 0x009b, 0x0303, 0x9b41, 0x0302, 0x001f, 0x0303, 0x0341, 0x0302,
	0x0003, 0x0303, 0x9b00, 0x0302, 0x0010, 0x0303, 0x0307, 0x0302, 0x008a,
	0x0303, 0xc040, 0x0302, 0x0083, 0x0303, 0xc369, 0x0302, 0x0003, 0x0303,
	0x0036, 0x0302, 0x003a, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0xc44e,
	0x0302, 0x0083, 0x0303, 0x012e, 0x0302, 0x0080, 0x0303, 0x1526, 0x0302,
	0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003,
	0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x1501, 0x0302, 0x003a, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0xc353, 0x0302, 0x0003, 0x0303, 0x01e4,
	0x0302, 0x00e0, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302,
	0x00e0, 0x0303, 0x1600, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0xc153, 0x0302, 0x0083, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303,
	0x1640, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01e4,
	0x0302, 0x00e0, 0x0303, 0xfdff, 0x0302, 0x00ba, 0x0303, 0x0382, 0x0302,
	0x001b, 0x0303, 0x0382, 0x0302, 0x0003, 0x0303, 0xc200, 0x0302, 0x0010,
	0x0303, 0x03e8, 0x0302, 0x001b, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303,
	0x1538, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053,
	0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x1537, 0x0302,
	0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302, 0x0060,
	0x0303, 0x53c2, 0x0302, 0x009b, 0x0303, 0x0058, 0x0302, 0x0020, 0x0303,
	0x53c1, 0x0302, 0x009b, 0x0303, 0x00aa, 0x0302, 0x00a0, 0x0303, 0xc165,
	0x0302, 0x0083, 0x0303, 0x0002, 0x0302, 0x00ba, 0x0303, 0x01d2, 0x0302,
	0x00e0, 0x0303, 0x3034, 0x0302, 0x00ba, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x3039, 0x0302, 0x003a, 0x0303, 0x000b, 0x0302, 0x00da, 0x0303,
	0x0368, 0x0302, 0x0083, 0x0303, 0xc3c3, 0x0302, 0x003a, 0x0303, 0x00c3,
	0x0302, 0x005a, 0x0303, 0x036a, 0x0302, 0x0003, 0x0303, 0x0068, 0x0302,
	0x00ba, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0xc44e, 0x0302, 0x0083,
	0x0303, 0xc569, 0x0302, 0x0003, 0x0303, 0x012e, 0x0302, 0x0080, 0x0303,
	0x1027, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x00dd,
	0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302,
	0x00e0, 0x0303, 0x1003, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x1011, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303,
	0x01e4, 0x0302, 0x00e0, 0x0303, 0x4a0f, 0x0302, 0x001f, 0x0303, 0x01dc,
	0x0302, 0x0060, 0x0303, 0x9b24, 0x0302, 0x0029, 0x0303, 0x1500, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003,
	0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0xc153, 0x0302, 0x0083, 0x0303,
	0x01e4, 0x0302, 0x00e0, 0x0303, 0xc175, 0x0302, 0x0003, 0x0303, 0x9b2d,
	0x0302, 0x0008, 0x0303, 0x992c, 0x0302, 0x00a8, 0x0303, 0x0081, 0x0302,
	0x003a, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0x0123, 0x0302, 0x0000,
	0x0303, 0xc04e, 0x0302, 0x0003, 0x0303, 0x000c, 0x0302, 0x003a, 0x0303,
	0x03d6, 0x0302, 0x009b, 0x0303, 0x0306, 0x0302, 0x0005, 0x0303, 0x00d0,
	0x0302, 0x00ba, 0x0303, 0x03d6, 0x0302, 0x009b, 0x0303, 0x038c, 0x0302,
	0x0004, 0x0303, 0xc575, 0x0302, 0x0083, 0x0303, 0x0112, 0x0302, 0x0080,
	0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0x0392, 0x0302, 0x0083, 0x0303,
	0x9d0f, 0x0302, 0x001f, 0x0303, 0xd661, 0x0302, 0x009f, 0x0303, 0x039d,
	0x0302, 0x0083, 0x0303, 0x151b, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302,
	0x0083, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0xc80f, 0x0302, 0x001f,
	0x0303, 0x5303, 0x0302, 0x001d, 0x0303, 0x0304, 0x0302, 0x0089, 0x0303,
	0x0020, 0x0302, 0x00ba, 0x0303, 0x9d03, 0x0302, 0x0091, 0x0303, 0x039d,
	0x0302, 0x0083, 0x0303, 0x0017, 0x0302, 0x003a, 0x0303, 0x039d, 0x0302,
	0x009b, 0x0303, 0x0308, 0x0302, 0x0004, 0x0303, 0xc366, 0x0302, 0x0003,
	0x0303, 0xc164, 0x0302, 0x0003, 0x0303, 0x00a2, 0x0302, 0x00ba, 0x0303,
	0x0362, 0x0302, 0x0083, 0x0303, 0xc84e, 0x0302, 0x0083, 0x0303, 0xc569,
	0x0302, 0x0003, 0x0303, 0x012e, 0x0302, 0x0080, 0x0303, 0x9b4f, 0x0302,
	0x009f, 0x0303, 0x0305, 0x0302, 0x002e, 0x0303, 0x1538, 0x0302, 0x003a,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc153, 0x0302, 0x0083, 0x0303,
	0x01e4, 0x0302, 0x00e0, 0x0303, 0x00b8, 0x0302, 0x0000, 0x0303, 0x0112,
	0x0302, 0x0080, 0x0303, 0xc065, 0x0302, 0x0003, 0x0303, 0x1688, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302, 0x0060,
	0x0303, 0x5383, 0x0302, 0x0004, 0x0303, 0x1689, 0x0302, 0x003a, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0x9d0f,
	0x0302, 0x001f, 0x0303, 0x5303, 0x0302, 0x009f, 0x0303, 0x039d, 0x0302,
	0x0083, 0x0303, 0x0028, 0x0302, 0x003a, 0x0303, 0x039d, 0x0302, 0x009b,
	0x0303, 0x000e, 0x0302, 0x00a1, 0x0303, 0x9b13, 0x0302, 0x008b, 0x0303,
	0x9c75, 0x0302, 0x0083, 0x0303, 0x0030, 0x0302, 0x003a, 0x0303, 0x039d,
	0x0302, 0x009b, 0x0303, 0x00bf, 0x0302, 0x0020, 0x0303, 0x75c9, 0x0302,
	0x009d, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x1522, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x7553, 0x0302, 0x0083, 0x0303,
	0x01e4, 0x0302, 0x00e0, 0x0303, 0x1523, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302,
	0x00e0, 0x0303, 0x1526, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0xc153, 0x0302, 0x0083, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303,
	0x0020, 0x0302, 0x00ba, 0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x00d1,
	0x0302, 0x00a0, 0x0303, 0x0080, 0x0302, 0x00ba, 0x0303, 0x01d2, 0x0302,
	0x00e0, 0x0303, 0x00f5, 0x0302, 0x0000, 0x0303, 0x1027, 0x0302, 0x00ba,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x0099, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x0010,
	0x0302, 0x00ba, 0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x00e2, 0x0302,
	0x0021, 0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xf041, 0x0302, 0x009f,
	0x0303, 0x03fa, 0x0302, 0x0003, 0x0303, 0x2000, 0x0302, 0x00ba, 0x0303,
	0x0392, 0x0302, 0x0083, 0x0303, 0xc369, 0x0302, 0x0003, 0x0303, 0x0160,
	0x0302, 0x00ba, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0x012e, 0x0302,
	0x0080, 0x0303, 0x1092, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x5153, 0x0302, 0x0083, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303,
	0x0040, 0x0302, 0x00ba, 0x0303, 0x01d2, 0x0302, 0x00e0, 0x0303, 0x3026,
	0x0302, 0x00ba, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x3029, 0x0302,
	0x00ba, 0x0303, 0x000e, 0x0302, 0x00da, 0x0303, 0x0368, 0x0302, 0x0083,
	0x0303, 0xb4b4, 0x0302, 0x003a, 0x0303, 0x00b4, 0x0302, 0x003a, 0x0303,
	0x036a, 0x0302, 0x0003, 0x0303, 0x00f5, 0x0302, 0x003a, 0x0303, 0x0362,
	0x0302, 0x0083, 0x0303, 0xc44e, 0x0302, 0x0083, 0x0303, 0xc569, 0x0302,
	0x0003, 0x0303, 0x012e, 0x0302, 0x0080, 0x0303, 0x1003, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0101, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x9b4f,
	0x0302, 0x009f, 0x0303, 0x030d, 0x0302, 0x002a, 0x0303, 0xc04e, 0x0302,
	0x0003, 0x0303, 0xd603, 0x0302, 0x008f, 0x0303, 0xc875, 0x0302, 0x0003,
	0x0303, 0x0112, 0x0302, 0x0080, 0x0303, 0x1482, 0x0302, 0x003a, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0x5370,
	0x0302, 0x009b, 0x0303, 0x0304, 0x0302, 0x0085, 0x0303, 0x00fc, 0x0302,
	0x003a, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0x0123, 0x0302, 0x0000,
	0x0303, 0x1480, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x7053, 0x0302, 0x0083, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0xc94e,
	0x0302, 0x0003, 0x0303, 0x00dc, 0x0302, 0x0080, 0x0303, 0x168a, 0x0302,
	0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003,
	0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x759a, 0x0302, 0x0083, 0x0303,
	0xf200, 0x0302, 0x0098, 0x0303, 0x03f2, 0x0302, 0x0083, 0x0303, 0x0001,
	0x0302, 0x00ba, 0x0303, 0x9926, 0x0302, 0x001f, 0x0303, 0x01d2, 0x0302,
	0x00e0, 0x0303, 0xc100, 0x0302, 0x0010, 0x0303, 0x9903, 0x0302, 0x001b,
	0x0303, 0x0399, 0x0302, 0x0003, 0x0303, 0x9b04, 0x0302, 0x0024, 0x0303,
	0x000e, 0x0302, 0x00ba, 0x0303, 0x0307, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0081, 0x0303, 0x9b84, 0x0302, 0x00a8, 0x0303, 0x000e, 0x0302,
	0x00ba, 0x0303, 0x0307, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0081,
	0x0303, 0xc369, 0x0302, 0x0003, 0x0303, 0xc14e, 0x0302, 0x0083, 0x0303,
	0x8302, 0x0302, 0x008b, 0x0303, 0xc24e, 0x0302, 0x0083, 0x0303, 0x012e,
	0x0302, 0x0080, 0x0303, 0xc169, 0x0302, 0x0083, 0x0303, 0x9b6f, 0x0302,
	0x001f, 0x0303, 0x0304, 0x0302, 0x002f, 0x0303, 0x0000, 0x0302, 0x003a,
	0x0303, 0x0307, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303,
	0xc2ee, 0x0302, 0x0083, 0x0303, 0x6991, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x4a0f, 0x0302, 0x001f, 0x0303,
	0x01dc, 0x0302, 0x0060, 0x0303, 0xc175, 0x0302, 0x0003, 0x0303, 0x9bab,
	0x0302, 0x0088, 0x0303, 0x99ac, 0x0302, 0x0028, 0x0303, 0x6409, 0x0302,
	0x0004, 0x0303, 0x6508, 0x0302, 0x0004, 0x0303, 0x1681, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0102, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x01e9,
	0x0302, 0x0060, 0x0303, 0x5381, 0x0302, 0x000a, 0x0303, 0x4d75, 0x0302,
	0x0083, 0x0303, 0x83b7, 0x0302, 0x000b, 0x0303, 0x9b4f, 0x0302, 0x009f,
	0x0303, 0x0306, 0x0302, 0x00aa, 0x0303, 0x1480, 0x0302, 0x00ba, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0x7d75,
	0x0302, 0x0083, 0x0303, 0x53be, 0x0302, 0x0028, 0x0303, 0x650d, 0x0302,
	0x0004, 0x0303, 0x640c, 0x0302, 0x0004, 0x0303, 0x1685, 0x0302, 0x003a,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x6667, 0x0302, 0x0083, 0x0303,
	0x01e9, 0x0302, 0x0060, 0x0303, 0x5275, 0x0302, 0x0003, 0x0303, 0x53c6,
	0x0302, 0x0009, 0x0303, 0x5270, 0x0302, 0x0099, 0x0303, 0x0352, 0x0302,
	0x0083, 0x0303, 0x6770, 0x0302, 0x001e, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x6787, 0x0302, 0x0085, 0x0303, 0x0040, 0x0302, 0x00ba, 0x0303,
	0x9a0c, 0x0302, 0x009f, 0x0303, 0x039a, 0x0302, 0x0003, 0x0303, 0x0136,
	0x0302, 0x00ba, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0x0123, 0x0302,
	0x0000, 0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0x0392, 0x0302, 0x0083,
	0x0303, 0x1526, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0xc053, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x1546,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01e9, 0x0302,
	0x0060, 0x0303, 0x1523, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x9a75, 0x0302, 0x0083, 0x0303,
	0x68db, 0x0302, 0x00a8, 0x0303, 0x68dc, 0x0302, 0x00a9, 0x0303, 0x168a,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc353, 0x0302,
	0x0003, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x0010, 0x0302, 0x00ba,
	0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x017a, 0x0302, 0x00a0, 0x0303,
	0x1538, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc153,
	0x0302, 0x0083, 0x0303, 0x01e4, 0x0302, 0x00e0, 0x0303, 0x0030, 0x0302,
	0x003a, 0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x0136, 0x0302, 0x00a1,
	0x0303, 0x4a0f, 0x0302, 0x001f, 0x0303, 0x01dc, 0x0302, 0x0060, 0x0303,
	0x9b00, 0x0302, 0x0010, 0x0303, 0x030a, 0x0302, 0x000c, 0x0303, 0x4b09,
	0x0302, 0x0028, 0x0303, 0x2000, 0x0302, 0x007a, 0x0303, 0x03d5, 0x0302,
	0x0091, 0x0303, 0x0300, 0x0302, 0x0090, 0x0303, 0x03e8, 0x0302, 0x001b,
	0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0xd100, 0x0302, 0x0090, 0x0303,
	0x03eb, 0x0302, 0x001b, 0x0303, 0x03eb, 0x0302, 0x0003, 0x0303, 0x018d,
	0x0302, 0x00ba, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0x0128, 0x0302,
	0x0080, 0x0303, 0x0004, 0x0302, 0x00ba, 0x0303, 0x9a0c, 0x0302, 0x009f,
	0x0303, 0x039a, 0x0302, 0x0003, 0x0303, 0x9d02, 0x0302, 0x008e, 0x0303,
	0x0197, 0x0302, 0x0000, 0x0303, 0x9b02, 0x0302, 0x008e, 0x0303, 0x0197,
	0x0302, 0x0000, 0x0303, 0x0002, 0x0302, 0x00ba, 0x0303, 0xe803, 0x0302,
	0x0011, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0x0199, 0x0302, 0x0080,
	0x0303, 0x010e, 0x0302, 0x0000, 0x0303, 0xc175, 0x0302, 0x0003, 0x0303,
	0x9b82, 0x0302, 0x0008, 0x0303, 0x9983, 0x0302, 0x00a8, 0x0303, 0xe802,
	0x0302, 0x0029, 0x0303, 0x01a1, 0x0302, 0x0000, 0x0303, 0xfdff, 0x0302,
	0x00ba, 0x0303, 0x0382, 0x0302, 0x001b, 0x0303, 0x0382, 0x0302, 0x0003,
	0x0303, 0x6409, 0x0302, 0x0004, 0x0303, 0x6508, 0x0302, 0x0004, 0x0303,
	0x1681, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0102,
	0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01e4, 0x0302,
	0x00e0, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303, 0x5381, 0x0302, 0x000a,
	0x0303, 0x4d75, 0x0302, 0x0083, 0x0303, 0x8393, 0x0302, 0x000b, 0x0303,
	0x9b4f, 0x0302, 0x009f, 0x0303, 0x0314, 0x0302, 0x00aa, 0x0303, 0x9b0f,
	0x0302, 0x001f, 0x0303, 0x03c8, 0x0302, 0x009d, 0x0303, 0x03cc, 0x0302,
	0x001b, 0x0303, 0x03c9, 0x0302, 0x009c, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x6507, 0x0302, 0x0004, 0x0303, 0xd70d, 0x0302, 0x00ab, 0x0303,
	0x6770, 0x0302, 0x001e, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x6783,
	0x0302, 0x0005, 0x0303, 0x7d75, 0x0302, 0x0083, 0x0303, 0x010e, 0x0302,
	0x0000, 0x0303, 0xd902, 0x0302, 0x000a, 0x0303, 0x01c1, 0x0302, 0x0000,
	0x0303, 0x6770, 0x0302, 0x001e, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303,
	0x6784, 0x0302, 0x0085, 0x0303, 0x7b75, 0x0302, 0x0083, 0x0303, 0x010e,
	0x0302, 0x0000, 0x0303, 0xe804, 0x0302, 0x0089, 0x0303, 0x0200, 0x0302,
	0x00ba, 0x0303, 0x0382, 0x0302, 0x0011, 0x0303, 0x0382, 0x0302, 0x0003,
	0x0303, 0x4b05, 0x0302, 0x0028, 0x0303, 0x4b04, 0x0302, 0x0089, 0x0303,
	0xc34b, 0x0302, 0x0003, 0x0303, 0xc175, 0x0302, 0x0003, 0x0303, 0x0112,
	0x0302, 0x0080, 0x0303, 0x0040, 0x0302, 0x00ba, 0x0303, 0x9a0c, 0x0302,
	0x009f, 0x0303, 0x039a, 0x0302, 0x0003, 0x0303, 0x3000, 0x0302, 0x003a,
	0x0303, 0x0392, 0x0302, 0x0083, 0x0303, 0x0199, 0x0302, 0x00ba, 0x0303,
	0x0362, 0x0302, 0x0083, 0x0303, 0x0128, 0x0302, 0x0080, 0x0303, 0xe802,
	0x0302, 0x008a, 0x0303, 0x0354, 0x0302, 0x0091, 0x0303, 0x03d1, 0x0302,
	0x0011, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0xb600, 0x0302, 0x00ae,
	0x0303, 0xc000, 0x0302, 0x00fa, 0x0303, 0x0300, 0x0302, 0x0090, 0x0303,
	0x03b6, 0x0302, 0x009b, 0x0303, 0x03b6, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x00e1, 0x0303, 0x0386, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x520f,
	0x0302, 0x001f, 0x0303, 0x0358, 0x0302, 0x009b, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x53b5, 0x0302, 0x0083, 0x0303, 0x01ec, 0x0302, 0x0000,
	0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0091, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x52b4, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0xb581, 0x0302, 0x002f, 0x0303, 0xb502, 0x0302,
	0x00aa, 0x0303, 0x01f3, 0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x01ec, 0x0302, 0x0000, 0x0303, 0xb553, 0x0302, 0x0086, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x4e0f, 0x0302, 0x009f, 0x0303, 0x030e, 0x0302, 0x00a8, 0x0303, 0x0328,
	0x0302, 0x00a9, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0358, 0x0302,
	0x00a8, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0307, 0x0302, 0x00a8,
	0x0303, 0x6304, 0x0302, 0x0004, 0x0303, 0x6307, 0x0302, 0x0083, 0x0303,
	0xc063, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303, 0x00cb,
	0x0302, 0x0080, 0x0303, 0x0006, 0x0302, 0x0000, 0x0303, 0xc04e, 0x0302,
	0x0003, 0x0303, 0x018c, 0x0302, 0x0000, 0x0303, 0xc0ea, 0x0302, 0x0083,
	0x0303, 0x01df, 0x0302, 0x0060, 0x0303, 0x8403, 0x0302, 0x008d, 0x0303,
	0xc067, 0x0302, 0x0083, 0x0303, 0x004e, 0x0302, 0x0060, 0x0303, 0xe805,
	0x0302, 0x000a, 0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x9000, 0x0302,
	0x005a, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0x0354, 0x0302, 0x0083,
	0x0303, 0x9000, 0x0302, 0x0018, 0x0303, 0x0390, 0x0302, 0x0003, 0x0303,
	0xe806, 0x0302, 0x0024, 0x0303, 0xe805, 0x0302, 0x00aa, 0x0303, 0xdd04,
	0x0302, 0x00a8, 0x0303, 0xc380, 0x0302, 0x0083, 0x0303, 0x800f, 0x0302,
	0x00da, 0x0303, 0x005a, 0x0302, 0x0060, 0x0303, 0xc280, 0x0302, 0x0003,
	0x0303, 0x07d0, 0x0302, 0x003a, 0x0303, 0x036f, 0x0302, 0x0011, 0x0303,
	0x005a, 0x0302, 0x0060, 0x0303, 0xc380, 0x0302, 0x0083, 0x0303, 0x9c40,
	0x0302, 0x00ba, 0x0303, 0x036f, 0x0302, 0x0011, 0x0303, 0x005a, 0x0302,
	0x0060, 0x0303, 0xe814, 0x0302, 0x00aa, 0x0303, 0x01df, 0x0302, 0x0060,
	0x0303, 0xe804, 0x0302, 0x00a4, 0x0303, 0xe802, 0x0302, 0x00ae, 0x0303,
	0x8002, 0x0302, 0x0088, 0x0303, 0x839f, 0x0302, 0x000b, 0x0303, 0xea00,
	0x0302, 0x0098, 0x0303, 0x03ea, 0x0302, 0x0083, 0x0303, 0x7d03, 0x0302,
	0x0017, 0x0303, 0x0040, 0x0302, 0x00c1, 0x0303, 0x8303, 0x0302, 0x00a4,
	0x0303, 0xcb67, 0x0302, 0x0003, 0x0303, 0x004e, 0x0302, 0x0060, 0x0303,
	0xe804, 0x0302, 0x008a, 0x0303, 0x9000, 0x0302, 0x00fa, 0x0303, 0x03b7,
	0x0302, 0x0003, 0x0303, 0x0354, 0x0302, 0x0083, 0x0303, 0xe803, 0x0302,
	0x0024, 0x0303, 0xe802, 0x0302, 0x002a, 0x0303, 0x0030, 0x0302, 0x0000,
	0x0303, 0xee67, 0x0302, 0x0086, 0x0303, 0x0046, 0x0302, 0x0080, 0x0303,
	0xc267, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x6791,
	0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x8475, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x7572,
	0x0302, 0x0083, 0x0303, 0x0016, 0x0302, 0x00ba, 0x0303, 0x8403, 0x0302,
	0x001d, 0x0303, 0x0367, 0x0302, 0x0091, 0x0303, 0x0384, 0x0302, 0x0003,
	0x0303, 0x8475, 0x0302, 0x0083, 0x0303, 0x720f, 0x0302, 0x009f, 0x0303,
	0x0375, 0x0302, 0x0097, 0x0303, 0x0050, 0x0302, 0x0021, 0x0303, 0x0000,
	0x0302, 0x00e1, 0x0303, 0x0386, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x1004,
	0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01d3, 0x0302,
	0x0060, 0x0303, 0x0808, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0011,
	0x0303, 0x0f53, 0x0302, 0x0082, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303,
	0xe802, 0x0302, 0x00ae, 0x0303, 0xc04e, 0x0302, 0x0003, 0x0303, 0x159d,
	0x0302, 0x003a, 0x0303, 0x4102, 0x0302, 0x00ad, 0x0303, 0x1596, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01d3, 0x0302, 0x0060,
	0x0303, 0x530f, 0x0302, 0x009f, 0x0303, 0x4102, 0x0302, 0x00ab, 0x0303,
	0x0300, 0x0302, 0x0090, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x001f,
	0x0302, 0x00ba, 0x0303, 0x0367, 0x0302, 0x009b, 0x0303, 0x0367, 0x0302,
	0x0083, 0x0303, 0x159e, 0x0302, 0x003a, 0x0303, 0x4102, 0x0302, 0x00ad,
	0x0303, 0x1597, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x01d3, 0x0302, 0x0060, 0x0303, 0x536a, 0x0302, 0x0003, 0x0303, 0x159f,
	0x0302, 0x00ba, 0x0303, 0x4102, 0x0302, 0x00ad, 0x0303, 0x1598, 0x0302,
	0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01d3, 0x0302, 0x0060,
	0x0303, 0x530f, 0x0302, 0x009f, 0x0303, 0x4102, 0x0302, 0x00ab, 0x0303,
	0x0300, 0x0302, 0x0090, 0x0303, 0x0369, 0x0302, 0x0003, 0x0303, 0x003f,
	0x0302, 0x003a, 0x0303, 0x0369, 0x0302, 0x001b, 0x0303, 0x0369, 0x0302,
	0x0003, 0x0303, 0x1090, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0x01d3, 0x0302, 0x0060, 0x0303, 0xe0c0, 0x0302, 0x00ba, 0x0303,
	0x0353, 0x0302, 0x001b, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x67c8,
	0x0302, 0x009c, 0x0303, 0x0353, 0x0302, 0x0011, 0x0303, 0x0369, 0x0302,
	0x0011, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0xc045, 0x0302, 0x0083,
	0x0303, 0x5345, 0x0302, 0x0091, 0x0303, 0x0345, 0x0302, 0x0083, 0x0303,
	0x01ce, 0x0302, 0x0060, 0x0303, 0x108f, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x6769, 0x0302, 0x0019, 0x0303, 0x0f6b, 0x0302,
	0x0002, 0x0303, 0x03c2, 0x0302, 0x001c, 0x0303, 0x036b, 0x0302, 0x0019,
	0x0303, 0x0f6b, 0x0302, 0x0002, 0x0303, 0x6ac3, 0x0302, 0x009c, 0x0303,
	0x036b, 0x0302, 0x0019, 0x0303, 0x0f6b, 0x0302, 0x0002, 0x0303, 0x01db,
	0x0302, 0x00ba, 0x0303, 0xe802, 0x0302, 0x000b, 0x0303, 0x0194, 0x0302,
	0x003a, 0x0303, 0x036b, 0x0302, 0x0097, 0x0303, 0x00a8, 0x0302, 0x00c1,
	0x0303, 0xc767, 0x0302, 0x0003, 0x0303, 0x00b7, 0x0302, 0x0000, 0x0303,
	0x01ae, 0x0302, 0x003a, 0x0303, 0xe802, 0x0302, 0x000b, 0x0303, 0x015c,
	0x0302, 0x00ba, 0x0303, 0x036b, 0x0302, 0x0097, 0x0303, 0x00af, 0x0302,
	0x0041, 0x0303, 0xc667, 0x0302, 0x0083, 0x0303, 0x00b7, 0x0302, 0x0000,
	0x0303, 0x0181, 0x0302, 0x00ba, 0x0303, 0xe802, 0x0302, 0x000b, 0x0303,
	0x013c, 0x0302, 0x00ba, 0x0303, 0x036b, 0x0302, 0x0097, 0x0303, 0x00b6,
	0x0302, 0x00c1, 0x0303, 0xc567, 0x0302, 0x0083, 0x0303, 0x00b7, 0x0302,
	0x0000, 0x0303, 0xc467, 0x0302, 0x0003, 0x0303, 0x67c4, 0x0302, 0x009c,
	0x0303, 0x03c8, 0x0302, 0x0091, 0x0303, 0x03c8, 0x0302, 0x001c, 0x0303,
	0x036a, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x03c9,
	0x0302, 0x009c, 0x0303, 0x0345, 0x0302, 0x0091, 0x0303, 0x0345, 0x0302,
	0x0083, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x0043, 0x0302, 0x0080,
	0x0303, 0x0046, 0x0302, 0x0080, 0x0303, 0xc067, 0x0302, 0x0083, 0x0303,
	0x8302, 0x0302, 0x008b, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0xc3ee,
	0x0302, 0x0003, 0x0303, 0x9b6f, 0x0302, 0x001f, 0x0303, 0x0386, 0x0302,
	0x002f, 0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0x0307, 0x0302, 0x0083,
	0x0303, 0x0000, 0x0302, 0x0081, 0x0303, 0x0003, 0x0302, 0x003a, 0x0303,
	0x9a0c, 0x0302, 0x009f, 0x0303, 0x039a, 0x0302, 0x0003, 0x0303, 0x1091,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x7053, 0x0302,
	0x0083, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0xc875, 0x0302, 0x0003,
	0x0303, 0x9d02, 0x0302, 0x008e, 0x0303, 0xc475, 0x0302, 0x0003, 0x0303,
	0x4bc2, 0x0302, 0x0097, 0x0303, 0x00d8, 0x0302, 0x0021, 0x0303, 0xcb75,
	0x0302, 0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0xb602, 0x0302,
	0x000a, 0x0303, 0x00df, 0x0302, 0x0080, 0x0303, 0x1091, 0x0302, 0x003a,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303,
	0x01ce, 0x0302, 0x0060, 0x0303, 0x1003, 0x0302, 0x00ba, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0x3810, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x1600, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303,
	0x01ce, 0x0302, 0x0060, 0x0303, 0x1640, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x1581, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc353, 0x0302, 0x0003,
	0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303,
	0x01ce, 0x0302, 0x0060, 0x0303, 0x15a5, 0x0302, 0x00ba, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302,
	0x0060, 0x0303, 0x1580, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303,
	0xc153, 0x0302, 0x0083, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0xffff,
	0x0302, 0x00fa, 0x0303, 0x03e2, 0x0302, 0x001b, 0x0303, 0x0101, 0x0302,
	0x0021, 0x0303, 0x0900, 0x0302, 0x00fa, 0x0303, 0x03e2, 0x0302, 0x0011,
	0x0303, 0x03e2, 0x0302, 0x0003, 0x0303, 0x9b16, 0x0302, 0x00ac, 0x0303,
	0x1023, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x3009,
	0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302,
	0x0060, 0x0303, 0x1082, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x000c, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303,
	0x01ce, 0x0302, 0x0060, 0x0303, 0x1079, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0xe1e1, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x107a, 0x0302, 0x003a,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0021, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x011c,
	0x0302, 0x0000, 0x0303, 0x1023, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302,
	0x0083, 0x0303, 0x3005, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x1033, 0x0302, 0x00ba, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0x01d3, 0x0302, 0x0060, 0x0303, 0x8000,
	0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x5853, 0x0302, 0x009b,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303,
	0x460f, 0x0302, 0x001f, 0x0303, 0x005a, 0x0302, 0x0060, 0x0303, 0x9b0f,
	0x0302, 0x001f, 0x0303, 0x03c7, 0x0302, 0x009d, 0x0303, 0x030d, 0x0302,
	0x00a8, 0x0303, 0x10ae, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x01d3, 0x0302, 0x0060, 0x0303, 0xfffe, 0x0302, 0x00ba, 0x0303,
	0x0353, 0x0302, 0x001b, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01ce,
	0x0302, 0x0060, 0x0303, 0x10a1, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302,
	0x0083, 0x0303, 0x0005, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0080, 0x0302, 0x00ba, 0x0303, 0x2000, 0x0302, 0x00da, 0x0303, 0x03e8,
	0x0302, 0x0011, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0x0140, 0x0302,
	0x003a, 0x0303, 0x0363, 0x0302, 0x0003, 0x0303, 0xc067, 0x0302, 0x0083,
	0x0303, 0x00c5, 0x0302, 0x0000, 0x0303, 0x4789, 0x0302, 0x0083, 0x0303,
	0x01aa, 0x0302, 0x00e0, 0x0303, 0x490f, 0x0302, 0x001f, 0x0303, 0x005a,
	0x0302, 0x0060, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0x0148, 0x0302,
	0x00ba, 0x0303, 0x0363, 0x0302, 0x0003, 0x0303, 0x00c5, 0x0302, 0x0000,
	0x0303, 0xd603, 0x0302, 0x00ab, 0x0303, 0x8391, 0x0302, 0x008b, 0x0303,
	0x8488, 0x0302, 0x002d, 0x0303, 0x105b, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0xce53, 0x0302, 0x0083, 0x0303, 0x01ce, 0x0302,
	0x0060, 0x0303, 0x9b00, 0x0302, 0x0010, 0x0303, 0x030f, 0x0302, 0x000a,
	0x0303, 0x4889, 0x0302, 0x0083, 0x0303, 0x0258, 0x0302, 0x003a, 0x0303,
	0x038a, 0x0302, 0x0083, 0x0303, 0xc05e, 0x0302, 0x0083, 0x0303, 0xc3ee,
	0x0302, 0x0003, 0x0303, 0x015b, 0x0302, 0x003a, 0x0303, 0x0363, 0x0302,
	0x0003, 0x0303, 0xc14e, 0x0302, 0x0083, 0x0303, 0xc667, 0x0302, 0x0083,
	0x0303, 0x0046, 0x0302, 0x0080, 0x0303, 0xd704, 0x0302, 0x00ae, 0x0303,
	0xc99a, 0x0302, 0x0003, 0x0303, 0xc168, 0x0302, 0x0003, 0x0303, 0x0045,
	0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x14e0, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x5000, 0x0302, 0x003a,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303,
	0x9d00, 0x0302, 0x0010, 0x0303, 0x0305, 0x0302, 0x008e, 0x0303, 0x1092,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x5153, 0x0302,
	0x0083, 0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0x0004, 0x0302, 0x00ba,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x9d03, 0x0302, 0x000e, 0x0303,
	0x2024, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x4bc2,
	0x0302, 0x0097, 0x0303, 0x0176, 0x0302, 0x0021, 0x0303, 0x0400, 0x0302,
	0x00ba, 0x0303, 0x0353, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x017b, 0x0302, 0x0080, 0x0303, 0x9b4f, 0x0302, 0x009f, 0x0303,
	0x0304, 0x0302, 0x002f, 0x0303, 0x0004, 0x0302, 0x00ba, 0x0303, 0x0353,
	0x0302, 0x001b, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x1003, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x01ce, 0x0302, 0x0060,
	0x0303, 0x9b4f, 0x0302, 0x009f, 0x0303, 0x030d, 0x0302, 0x002a, 0x0303,
	0xd604, 0x0302, 0x000f, 0x0303, 0xcb9a, 0x0302, 0x0083, 0x0303, 0xc268,
	0x0302, 0x0003, 0x0303, 0x0045, 0x0302, 0x0080, 0x0303, 0x0187, 0x0302,
	0x00ba, 0x0303, 0x0363, 0x0302, 0x0003, 0x0303, 0x00c2, 0x0302, 0x0080,
	0x0303, 0x14e1, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x01d3, 0x0302, 0x0060, 0x0303, 0x536d, 0x0302, 0x009b, 0x0303, 0x038b,
	0x0302, 0x0084, 0x0303, 0x6503, 0x0302, 0x0084, 0x0303, 0x6402, 0x0302,
	0x0084, 0x0303, 0x0192, 0x0302, 0x0000, 0x0303, 0xc267, 0x0302, 0x0003,
	0x0303, 0xc068, 0x0302, 0x0083, 0x0303, 0x0046, 0x0302, 0x0080, 0x0303,
	0x6667, 0x0302, 0x0083, 0x0303, 0x1681, 0x0302, 0x00ba, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0xc40f, 0x0302, 0x001f, 0x0303, 0x6703, 0x0302,
	0x001c, 0x0303, 0x03c3, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x01ce, 0x0302, 0x0060, 0x0303, 0xd604, 0x0302, 0x000f, 0x0303,
	0xcf9a, 0x0302, 0x0003, 0x0303, 0xc668, 0x0302, 0x0083, 0x0303, 0x0045,
	0x0302, 0x0080, 0x0303, 0x1681, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302,
	0x0083, 0x0303, 0x01d3, 0x0302, 0x0060, 0x0303, 0x5387, 0x0302, 0x000a,
	0x0303, 0x1683, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x01d3, 0x0302, 0x0060, 0x0303, 0x5392, 0x0302, 0x0089, 0x0303, 0x6770,
	0x0302, 0x001e, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x6795, 0x0302,
	0x0085, 0x0303, 0x018f, 0x0302, 0x0000, 0x0303, 0xc167, 0x0302, 0x0003,
	0x0303, 0xd568, 0x0302, 0x0003, 0x0303, 0x01af, 0x0302, 0x0080, 0x0303,
	0xc267, 0x0302, 0x0003, 0x0303, 0x6d68, 0x0302, 0x0003, 0x0303, 0x6700,
	0x0302, 0x0010, 0x0303, 0x0387, 0x0302, 0x001b, 0x0303, 0x0387, 0x0302,
	0x0003, 0x0303, 0x8475, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x7572, 0x0302, 0x0083, 0x0303, 0x0016, 0x0302, 0x00ba, 0x0303,
	0x8403, 0x0302, 0x001d, 0x0303, 0x0384, 0x0302, 0x0011, 0x0303, 0x0375,
	0x0302, 0x0083, 0x0303, 0x6800, 0x0302, 0x0010, 0x0303, 0x0375, 0x0302,
	0x009b, 0x0303, 0x0384, 0x0302, 0x0003, 0x0303, 0x8475, 0x0302, 0x0083,
	0x0303, 0x720f, 0x0302, 0x009f, 0x0303, 0x0375, 0x0302, 0x0097, 0x0303,
	0x01b4, 0x0302, 0x00a1, 0x0303, 0x670f, 0x0302, 0x001f, 0x0303, 0x0387,
	0x0302, 0x0011, 0x0303, 0x0387, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0x75d1, 0x0302, 0x0091, 0x0303, 0xe802, 0x0302, 0x008a,
	0x0303, 0x0354, 0x0302, 0x0091, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303,
	0xb600, 0x0302, 0x00ae, 0x0303, 0xc000, 0x0302, 0x00fa, 0x0303, 0x0300,
	0x0302, 0x0090, 0x0303, 0x03b6, 0x0302, 0x009b, 0x0303, 0x03b6, 0x0302,
	0x0083, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x520f, 0x0302, 0x001f,
	0x0303, 0x0358, 0x0302, 0x009b, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x53b5, 0x0302, 0x0083, 0x0303, 0x01d6, 0x0302, 0x0000, 0x0303, 0x8000,
	0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0091, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x52b4, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0xb581, 0x0302, 0x002f, 0x0303, 0xb502, 0x0302, 0x00aa, 0x0303,
	0x01dd, 0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x01d6,
	0x0302, 0x0000, 0x0303, 0xb553, 0x0302, 0x0086, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0xb611, 0x0302, 0x002c, 0x0303, 0xc068, 0x0302, 0x0083,
	0x0303, 0xb60f, 0x0302, 0x001f, 0x0303, 0x03c9, 0x0302, 0x009c, 0x0303,
	0x0302, 0x0302, 0x002f, 0x0303, 0xc168, 0x0302, 0x0003, 0x0303, 0x0055,
	0x0302, 0x003a, 0x0303, 0x03b8, 0x0302, 0x001b, 0x0303, 0x01ea, 0x0302,
	0x00a0, 0x0303, 0x1891, 0x0302, 0x001a, 0x0303, 0x01eb, 0x0302, 0x0080,
	0x0303, 0x1091, 0x0302, 0x009a, 0x0303, 0x68b5, 0x0302, 0x0003, 0x0303,
	0x03b4, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xb581,
	0x0302, 0x002f, 0x0303, 0xb590, 0x0302, 0x002a, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x4e0f, 0x0302, 0x009f, 0x0303,
	0x030d, 0x0302, 0x00a8, 0x0303, 0x4e00, 0x0302, 0x0014, 0x0303, 0x03cb,
	0x0302, 0x0097, 0x0303, 0x0012, 0x0302, 0x00a0, 0x0303, 0x4f07, 0x0302,
	0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0081,
	0x0303, 0xf002, 0x0302, 0x00ab, 0x0303, 0x000e, 0x0302, 0x0080, 0x0303,
	0xc0f0, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x9c81, 0x0302, 0x0088, 0x0303, 0xc868, 0x0302,
	0x0003, 0x0303, 0x019b, 0x0302, 0x0060, 0x0303, 0x1092, 0x0302, 0x003a,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0022, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x018a, 0x0302, 0x0060, 0x0303, 0xd50f,
	0x0302, 0x001f, 0x0303, 0x03e8, 0x0302, 0x0011, 0x0303, 0x03e8, 0x0302,
	0x0003, 0x0303, 0xdcfa, 0x0302, 0x0083, 0x0303, 0xc079, 0x0302, 0x0083,
	0x0303, 0xc200, 0x0302, 0x0010, 0x0303, 0x03e8, 0x0302, 0x001b, 0x0303,
	0x03e8, 0x0302, 0x0003, 0x0303, 0xc092, 0x0302, 0x0083, 0x0303, 0xc080,
	0x0302, 0x0083, 0x0303, 0x0028, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302,
	0x0002, 0x0303, 0x01ed, 0x0302, 0x00e0, 0x0303, 0x9d0b, 0x0302, 0x0009,
	0x0303, 0x2000, 0x0302, 0x007a, 0x0303, 0x0300, 0x0302, 0x0090, 0x0303,
	0x0341, 0x0302, 0x001b, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x0033,
	0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01f7, 0x0302,
	0x0060, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x9d02, 0x0302, 0x0028,
	0x0303, 0x00e7, 0x0302, 0x0000, 0x0303, 0xf000, 0x0302, 0x0018, 0x0303,
	0x03f0, 0x0302, 0x0003, 0x0303, 0xc081, 0x0302, 0x0003, 0x0303, 0xc09c,
	0x0302, 0x0003, 0x0303, 0x6b01, 0x0302, 0x003a, 0x0303, 0x003e, 0x0302,
	0x00da, 0x0303, 0x0389, 0x0302, 0x0083, 0x0303, 0x9ad2, 0x0302, 0x003a,
	0x0303, 0x0290, 0x0302, 0x00da, 0x0303, 0x038a, 0x0302, 0x0083, 0x0303,
	0x9a46, 0x0302, 0x0083, 0x0303, 0xc09b, 0x0302, 0x0083, 0x0303, 0x01bf,
	0x0302, 0x0060, 0x0303, 0x0043, 0x0302, 0x00ba, 0x0303, 0x0f4f, 0x0302,
	0x0002, 0x0303, 0x01f1, 0x0302, 0x0060, 0x0303, 0x0046, 0x0302, 0x00ba,
	0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e5, 0x0302, 0x0060, 0x0303,
	0xc04a, 0x0302, 0x0083, 0x0303, 0x01a5, 0x0302, 0x00e0, 0x0303, 0x0011,
	0x0302, 0x003a, 0x0303, 0x0399, 0x0302, 0x0003, 0x0303, 0x004d, 0x0302,
	0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e3, 0x0302, 0x0060,
	0x0303, 0x500a, 0x0302, 0x0028, 0x0303, 0x8405, 0x0302, 0x002d, 0x0303,
	0x0052, 0x0302, 0x00ba, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01f3,
	0x0302, 0x00e0, 0x0303, 0x01a5, 0x0302, 0x00e0, 0x0303, 0x0056, 0x0302,
	0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e7, 0x0302, 0x00e0,
	0x0303, 0x0048, 0x0302, 0x0000, 0x0303, 0x484a, 0x0302, 0x0083, 0x0303,
	0xc868, 0x0302, 0x0003, 0x0303, 0x019b, 0x0302, 0x0060, 0x0303, 0x005d,
	0x0302, 0x00ba, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01f1, 0x0302,
	0x0060, 0x0303, 0x0060, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002,
	0x0303, 0x01e5, 0x0302, 0x0060, 0x0303, 0x01a8, 0x0302, 0x0060, 0x0303,
	0x01a5, 0x0302, 0x00e0, 0x0303, 0x0065, 0x0302, 0x003a, 0x0303, 0x0f4f,
	0x0302, 0x0002, 0x0303, 0x01e3, 0x0302, 0x0060, 0x0303, 0x0187, 0x0302,
	0x00e0, 0x0303, 0x5008, 0x0302, 0x00a8, 0x0303, 0x8402, 0x0302, 0x00ad,
	0x0303, 0x0078, 0x0302, 0x0000, 0x0303, 0x006c, 0x0302, 0x003a, 0x0303,
	0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e7, 0x0302, 0x00e0, 0x0303, 0x4849,
	0x0302, 0x0083, 0x0303, 0x0062, 0x0302, 0x0080, 0x0303, 0x4748, 0x0302,
	0x001e, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0320, 0x0302, 0x00ab, 0x0303, 0x4b48, 0x0302, 0x0017, 0x0303,
	0x0091, 0x0302, 0x0020, 0x0303, 0x484a, 0x0302, 0x0091, 0x0303, 0x0f4a,
	0x0302, 0x0002, 0x0303, 0x01a8, 0x0302, 0x0060, 0x0303, 0x0067, 0x0302,
	0x0080, 0x0303, 0x4748, 0x0302, 0x0003, 0x0303, 0x007c, 0x0302, 0x00ba,
	0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e9, 0x0302, 0x0060, 0x0303,
	0x007f, 0x0302, 0x00ba, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e3,
	0x0302, 0x0060, 0x0303, 0x0187, 0x0302, 0x00e0, 0x0303, 0x5011, 0x0302,
	0x0028, 0x0303, 0xc968, 0x0302, 0x0083, 0x0303, 0x019b, 0x0302, 0x0060,
	0x0303, 0xb60b, 0x0302, 0x00aa, 0x0303, 0x0029, 0x0302, 0x00ba, 0x0303,
	0x0399, 0x0302, 0x0003, 0x0303, 0x0089, 0x0302, 0x00ba, 0x0303, 0x0f4f,
	0x0302, 0x0002, 0x0303, 0x01f3, 0x0302, 0x00e0, 0x0303, 0x4948, 0x0302,
	0x0083, 0x0303, 0x008d, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002,
	0x0303, 0x01e7, 0x0302, 0x00e0, 0x0303, 0x0062, 0x0302, 0x0080, 0x0303,
	0x4a46, 0x0302, 0x0003, 0x0303, 0x01bf, 0x0302, 0x0060, 0x0303, 0x0084,
	0x0302, 0x0000, 0x0303, 0xf3de, 0x0302, 0x003a, 0x0303, 0x0057, 0x0302,
	0x00da, 0x0303, 0x0389, 0x0302, 0x0083, 0x0303, 0x4847, 0x0302, 0x001b,
	0x0303, 0x009a, 0x0302, 0x0021, 0x0303, 0x4847, 0x0302, 0x0003, 0x0303,
	0x009a, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01f5,
	0x0302, 0x00e0, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x01a5, 0x0302,
	0x00e0, 0x0303, 0x01a8, 0x0302, 0x0060, 0x0303, 0x480c, 0x0302, 0x000c,
	0x0303, 0x9d06, 0x0302, 0x0029, 0x0303, 0x14e2, 0x0302, 0x003a, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0xce53, 0x0302, 0x0083, 0x0303, 0x018a,
	0x0302, 0x0060, 0x0303, 0x00ad, 0x0302, 0x0080, 0x0303, 0x15c2, 0x0302,
	0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc153, 0x0302, 0x0083,
	0x0303, 0x018a, 0x0302, 0x0060, 0x0303, 0x00ad, 0x0302, 0x0080, 0x0303,
	0x14a4, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc153,
	0x0302, 0x0083, 0x0303, 0x018a, 0x0302, 0x0060, 0x0303, 0x0031, 0x0302,
	0x00ba, 0x0303, 0x0399, 0x0302, 0x0003, 0x0303, 0x00b7, 0x0302, 0x003a,
	0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x48c7, 0x0302, 0x009b, 0x0303,
	0x00b5, 0x0302, 0x00a1, 0x0303, 0xe802, 0x0302, 0x000e, 0x0303, 0x01da,
	0x0302, 0x0060, 0x0303, 0xc24e, 0x0302, 0x0083, 0x0303, 0x01d8, 0x0302,
	0x00e0, 0x0303, 0x848a, 0x0302, 0x00ad, 0x0303, 0x9b00, 0x0302, 0x0098,
	0x0303, 0x039b, 0x0302, 0x0083, 0x0303, 0x8310, 0x0302, 0x008b, 0x0303,
	0x00bc, 0x0302, 0x0080, 0x0303, 0x4806, 0x0302, 0x000c, 0x0303, 0x9d03,
	0x0302, 0x0089, 0x0303, 0xd80c, 0x0302, 0x000a, 0x0303, 0x00c4, 0x0302,
	0x0080, 0x0303, 0xd90a, 0x0302, 0x008a, 0x0303, 0x00c4, 0x0302, 0x0080,
	0x0303, 0xda08, 0x0302, 0x000a, 0x0303, 0x00c4, 0x0302, 0x0080, 0x0303,
	0x9d14, 0x0302, 0x008f, 0x0303, 0x018f, 0x0302, 0x0060, 0x0303, 0x530f,
	0x0302, 0x009f, 0x0303, 0x03c8, 0x0302, 0x009d, 0x0303, 0x0353, 0x0302,
	0x0011, 0x0303, 0x030f, 0x0302, 0x0088, 0x0303, 0x9d04, 0x0302, 0x008d,
	0x0303, 0xc80f, 0x0302, 0x001f, 0x0303, 0x039b, 0x0302, 0x009e, 0x0303,
	0x0371, 0x0302, 0x002b, 0x0303, 0x00d1, 0x0302, 0x003a, 0x0303, 0x0f4f,
	0x0302, 0x0002, 0x0303, 0x01e7, 0x0302, 0x00e0, 0x0303, 0x00d4, 0x0302,
	0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01e3, 0x0302, 0x0060,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0187, 0x0302, 0x00e0, 0x0303,
	0x50c5, 0x0302, 0x0028, 0x0303, 0x00ca, 0x0302, 0x0000, 0x0303, 0x84ab,
	0x0302, 0x00ad, 0x0303, 0xc29c, 0x0302, 0x0083, 0x0303, 0x47c9, 0x0302,
	0x0017, 0x0303, 0x014b, 0x0302, 0x00a1, 0x0303, 0x48c9, 0x0302, 0x0017,
	0x0303, 0x014b, 0x0302, 0x00a1, 0x0303, 0x9d6d, 0x0302, 0x0009, 0x0303,
	0x15a5, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x018f,
	0x0302, 0x0060, 0x0303, 0x3b00, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302,
	0x001b, 0x0303, 0x0f53, 0x0302, 0x0082, 0x0303, 0x018a, 0x0302, 0x0060,
	0x0303, 0x01a8, 0x0302, 0x0060, 0x0303, 0x00ea, 0x0302, 0x00ba, 0x0303,
	0x0f4f, 0x0302, 0x0002, 0x0303, 0x01ef, 0x0302, 0x0060, 0x0303, 0x0187,
	0x0302, 0x00e0, 0x0303, 0xd70f, 0x0302, 0x009f, 0x0303, 0x03c7, 0x0302,
	0x001c, 0x0303, 0x0386, 0x0302, 0x00ab, 0x0303, 0x7500, 0x0302, 0x0098,
	0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x0389, 0x0302, 0x000b, 0x0303,
	0x0051, 0x0302, 0x00ba, 0x0303, 0x0399, 0x0302, 0x0003, 0x0303, 0x1580,
	0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc253, 0x0302,
	0x0083, 0x0303, 0x018a, 0x0302, 0x0060, 0x0303, 0x0001, 0x0302, 0x007a,
	0x0303, 0x0389, 0x0302, 0x0083, 0x0303, 0x0140, 0x0302, 0x003a, 0x0303,
	0x038a, 0x0302, 0x0083, 0x0303, 0xcf00, 0x0302, 0x0090, 0x0303, 0x0341,
	0x0302, 0x001b, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x9d41, 0x0302,
	0x001f, 0x0303, 0x0341, 0x0302, 0x0011, 0x0303, 0x0341, 0x0302, 0x0003,
	0x0303, 0xc040, 0x0302, 0x0083, 0x0303, 0xc05e, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0093, 0x0303, 0x2000, 0x0302, 0x00da, 0x0303, 0x0341,
	0x0302, 0x0011, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x010a, 0x0302,
	0x00ba, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x01f7, 0x0302, 0x0060,
	0x0303, 0x010d, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303,
	0x01eb, 0x0302, 0x00e0, 0x0303, 0x5e9b, 0x0302, 0x0003, 0x0303, 0xd730,
	0x0302, 0x00aa, 0x0303, 0x0110, 0x0302, 0x0000, 0x0303, 0xd100, 0x0302,
	0x0090, 0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x039d, 0x0302, 0x0083,
	0x0303, 0x9d13, 0x0302, 0x000a, 0x0303, 0xd70f, 0x0302, 0x009f, 0x0303,
	0x03c8, 0x0302, 0x001c, 0x0303, 0x0310, 0x0302, 0x002f, 0x0303, 0x9d0f,
	0x0302, 0x001f, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0307, 0x0302,
	0x0029, 0x0303, 0x15c6, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0x018f, 0x0302, 0x0060, 0x0303, 0x0800, 0x0302, 0x00ba, 0x0303,
	0x0353, 0x0302, 0x001b, 0x0303, 0x0126, 0x0302, 0x00a0, 0x0303, 0x9d0f,
	0x0302, 0x001f, 0x0303, 0x03d1, 0x0302, 0x0011, 0x0303, 0x039d, 0x0302,
	0x0083, 0x0303, 0x2424, 0x0302, 0x003a, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x0128, 0x0302, 0x0080, 0x0303, 0x0404, 0x0302, 0x003a, 0x0303,
	0x0367, 0x0302, 0x0083, 0x0303, 0x1004, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x6753, 0x0302, 0x0083, 0x0303, 0x018a, 0x0302,
	0x0060, 0x0303, 0x1003, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0x018a, 0x0302, 0x0060, 0x0303, 0x28cb, 0x0302, 0x00ba, 0x0303,
	0x000c, 0x0302, 0x005a, 0x0303, 0x0389, 0x0302, 0x0083, 0x0303, 0x01a5,
	0x0302, 0x00e0, 0x0303, 0x013b, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302,
	0x0002, 0x0303, 0x48c7, 0x0302, 0x009b, 0x0303, 0x00b5, 0x0302, 0x00a1,
	0x0303, 0xe802, 0x0302, 0x000e, 0x0303, 0x01da, 0x0302, 0x0060, 0x0303,
	0xc24e, 0x0302, 0x0083, 0x0303, 0x01d8, 0x0302, 0x00e0, 0x0303, 0x8488,
	0x0302, 0x002d, 0x0303, 0xd902, 0x0302, 0x000a, 0x0303, 0x014b, 0x0302,
	0x0080, 0x0303, 0xc49c, 0x0302, 0x0083, 0x0303, 0x4700, 0x0302, 0x0090,
	0x0303, 0x039a, 0x0302, 0x001b, 0x0303, 0x0147, 0x0302, 0x0020, 0x0303,
	0x0f9a, 0x0302, 0x0082, 0x0303, 0xc59c, 0x0302, 0x0003, 0x0303, 0x0014,
	0x0302, 0x0000, 0x0303, 0x0009, 0x0302, 0x003a, 0x0303, 0x039c, 0x0302,
	0x0003, 0x0303, 0x1000, 0x0302, 0x00ba, 0x0303, 0x0392, 0x0302, 0x0091,
	0x0303, 0x0392, 0x0302, 0x0083, 0x0303, 0x0189, 0x0302, 0x0000, 0x0303,
	0xf289, 0x0302, 0x0003, 0x0303, 0x568a, 0x0302, 0x0083, 0x0303, 0x0060,
	0x0302, 0x003a, 0x0303, 0x0399, 0x0302, 0x0003, 0x0303, 0x470f, 0x0302,
	0x009f, 0x0303, 0x03c8, 0x0302, 0x001c, 0x0303, 0x0348, 0x0302, 0x0011,
	0x0303, 0x039b, 0x0302, 0x0083, 0x0303, 0x4748, 0x0302, 0x0017, 0x0303,
	0x0147, 0x0302, 0x00a1, 0x0303, 0xd100, 0x0302, 0x0090, 0x0303, 0x03eb,
	0x0302, 0x001b, 0x0303, 0x03eb, 0x0302, 0x0003, 0x0303, 0xd500, 0x0302,
	0x0010, 0x0303, 0x03e8, 0x0302, 0x001b, 0x0303, 0x03e8, 0x0302, 0x0003,
	0x0303, 0x9c2e, 0x0302, 0x0028, 0x0303, 0x8257, 0x0302, 0x0083, 0x0303,
	0x9d09, 0x0302, 0x00ab, 0x0303, 0x9d02, 0x0302, 0x002d, 0x0303, 0x0166,
	0x0302, 0x0080, 0x0303, 0xc20f, 0x0302, 0x001f, 0x0303, 0x03e8, 0x0302,
	0x0011, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0xff00, 0x0302, 0x003a,
	0x0303, 0x0382, 0x0302, 0x0011, 0x0303, 0x0382, 0x0302, 0x0003, 0x0303,
	0x016e, 0x0302, 0x003a, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x48c7,
	0x0302, 0x009b, 0x0303, 0x016c, 0x0302, 0x00a1, 0x0303, 0xe802, 0x0302,
	0x000e, 0x0303, 0x01da, 0x0302, 0x0000, 0x0303, 0xc24e, 0x0302, 0x0083,
	0x0303, 0x01d8, 0x0302, 0x0080, 0x0303, 0x5782, 0x0302, 0x0083, 0x0303,
	0x01a8, 0x0302, 0x0060, 0x0303, 0x4803, 0x0302, 0x000c, 0x0303, 0xd904,
	0x0302, 0x000a, 0x0303, 0x015b, 0x0302, 0x0000, 0x0303, 0xda02, 0x0302,
	0x000a, 0x0303, 0x015b, 0x0302, 0x0000, 0x0303, 0xc200, 0x0302, 0x0010,
	0x0303, 0x03e8, 0x0302, 0x001b, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303,
	0x8488, 0x0302, 0x002e, 0x0303, 0xd5d9, 0x0302, 0x0003, 0x0303, 0xd5da,
	0x0302, 0x0003, 0x0303, 0xc0d9, 0x0302, 0x0083, 0x0303, 0xc0da, 0x0302,
	0x0083, 0x0303, 0x01a5, 0x0302, 0x00e0, 0x0303, 0x0180, 0x0302, 0x003a,
	0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0x8498, 0x0302, 0x00ad, 0x0303,
	0x9d05, 0x0302, 0x002a, 0x0303, 0x4703, 0x0302, 0x000c, 0x0303, 0xd9be,
	0x0302, 0x0089, 0x0303, 0x0186, 0x0302, 0x0000, 0x0303, 0xdac0, 0x0302,
	0x0089, 0x0303, 0x015b, 0x0302, 0x0000, 0x0303, 0x8402, 0x0302, 0x002a,
	0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0014, 0x0302, 0x0000, 0x0303,
	0x520f, 0x0302, 0x001f, 0x0303, 0x0358, 0x0302, 0x009b, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x53b5, 0x0302, 0x0083, 0x0303, 0x0192, 0x0302,
	0x0000, 0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0091,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x52b4, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0xb581, 0x0302, 0x002f, 0x0303, 0xb502,
	0x0302, 0x00aa, 0x0303, 0x0199, 0x0302, 0x0080, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0192, 0x0302, 0x0000, 0x0303, 0xb553, 0x0302, 0x0086,
	0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0xd100, 0x0302, 0x0090, 0x0303,
	0x03b6, 0x0302, 0x0083, 0x0303, 0x680f, 0x0302, 0x001f, 0x0303, 0x03c1,
	0x0302, 0x0091, 0x0303, 0x03d1, 0x0302, 0x0011, 0x0303, 0xe802, 0x0302,
	0x008a, 0x0303, 0x0354, 0x0302, 0x0091, 0x0303, 0x03b7, 0x0302, 0x0003,
	0x0303, 0xb600, 0x0302, 0x002f, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303,
	0xc167, 0x0302, 0x0003, 0x0303, 0xd568, 0x0302, 0x0003, 0x0303, 0x01aa,
	0x0302, 0x0080, 0x0303, 0xc267, 0x0302, 0x0003, 0x0303, 0x6d68, 0x0302,
	0x0003, 0x0303, 0x6700, 0x0302, 0x0010, 0x0303, 0x0387, 0x0302, 0x001b,
	0x0303, 0x0387, 0x0302, 0x0003, 0x0303, 0x8475, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x7572, 0x0302, 0x0083, 0x0303, 0x0016,
	0x0302, 0x00ba, 0x0303, 0x8403, 0x0302, 0x001d, 0x0303, 0x0384, 0x0302,
	0x0011, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x6800, 0x0302, 0x0010,
	0x0303, 0x0375, 0x0302, 0x009b, 0x0303, 0x0384, 0x0302, 0x0003, 0x0303,
	0x8475, 0x0302, 0x0083, 0x0303, 0x720f, 0x0302, 0x009f, 0x0303, 0x0375,
	0x0302, 0x0097, 0x0303, 0x01af, 0x0302, 0x00a1, 0x0303, 0x670f, 0x0302,
	0x001f, 0x0303, 0x0387, 0x0302, 0x0011, 0x0303, 0x0387, 0x0302, 0x0003,
	0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x460f, 0x0302, 0x001f, 0x0303,
	0x03c4, 0x0302, 0x009d, 0x0303, 0x0307, 0x0302, 0x00a8, 0x0303, 0x460f,
	0x0302, 0x001f, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0306, 0x0302,
	0x00a9, 0x0303, 0x0307, 0x0302, 0x00a8, 0x0303, 0x4608, 0x0302, 0x00a9,
	0x0303, 0x4609, 0x0302, 0x00a8, 0x0303, 0xc94b, 0x0302, 0x0003, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0xc84b, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x00e1, 0x0303, 0xc44b, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0xc24b, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x00e1,
	0x0303, 0xc14b, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303,
	0xc567, 0x0302, 0x0083, 0x0303, 0x01db, 0x0302, 0x0080, 0x0303, 0xc667,
	0x0302, 0x0083, 0x0303, 0x01db, 0x0302, 0x0080, 0x0303, 0xc767, 0x0302,
	0x0003, 0x0303, 0x01db, 0x0302, 0x0080, 0x0303, 0xc367, 0x0302, 0x0083,
	0x0303, 0x01db, 0x0302, 0x0080, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303,
	0xc4ee, 0x0302, 0x0083, 0x0303, 0x6791, 0x0302, 0x0003, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xc24e, 0x0302, 0x0083, 0x0303,
	0x01d2, 0x0302, 0x0080, 0x0303, 0xc44e, 0x0302, 0x0083, 0x0303, 0x01d2,
	0x0302, 0x0080, 0x0303, 0xc84e, 0x0302, 0x0083, 0x0303, 0x01d2, 0x0302,
	0x0080, 0x0303, 0xc94e, 0x0302, 0x0003, 0x0303, 0x01d2, 0x0302, 0x0080,
	0x0303, 0xc14e, 0x0302, 0x0083, 0x0303, 0x01d4, 0x0302, 0x0080, 0x0303,
	0xc24e, 0x0302, 0x0083, 0x0303, 0x01d4, 0x0302, 0x0080, 0x0303, 0xc84e,
	0x0302, 0x0083, 0x0303, 0x01d8, 0x0302, 0x0080, 0x0303, 0xc24e, 0x0302,
	0x0083, 0x0303, 0x01d6, 0x0302, 0x0000, 0x0303, 0xc44e, 0x0302, 0x0083,
	0x0303, 0x01d6, 0x0302, 0x0000, 0x0303, 0xc84e, 0x0302, 0x0083, 0x0303,
	0x01d6, 0x0302, 0x0000, 0x0303, 0xc44e, 0x0302, 0x0083, 0x0303, 0x01d8,
	0x0302, 0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xee13, 0x0302, 0x002f,
	0x0303, 0x4f0f, 0x0302, 0x001f, 0x0303, 0x0307, 0x0302, 0x0083, 0x0303,
	0x03c9, 0x0302, 0x001d, 0x0303, 0x034f, 0x0302, 0x0083, 0x0303, 0xee0f,
	0x0302, 0x009f, 0x0303, 0x03c9, 0x0302, 0x001d, 0x0303, 0x03c7, 0x0302,
	0x009b, 0x0303, 0x03ee, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081,
	0x0303, 0xee67, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x6791, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x4e0f, 0x0302, 0x009f, 0x0303, 0xc04e, 0x0302, 0x0003, 0x0303,
	0x0308, 0x0302, 0x00a8, 0x0303, 0x0315, 0x0302, 0x0029, 0x0303, 0x03c2,
	0x0302, 0x009d, 0x0303, 0x0303, 0x0302, 0x0028, 0x0303, 0x0303, 0x0302,
	0x00a9, 0x0303, 0x000f, 0x0302, 0x0000, 0x0303, 0x00fc, 0x0302, 0x0000,
	0x0303, 0x0130, 0x0302, 0x0080, 0x0303, 0x1027, 0x0302, 0x00ba, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x0099, 0x0302, 0x003a, 0x0303, 0x0353,
	0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0xc090, 0x0302,
	0x0003, 0x0303, 0xf299, 0x0302, 0x0004, 0x0303, 0x002c, 0x0302, 0x00ba,
	0x0303, 0x0372, 0x0302, 0x0003, 0x0303, 0x00f3, 0x0302, 0x0000, 0x0303,
	0x9a0f, 0x0302, 0x009f, 0x0303, 0xc203, 0x0302, 0x001f, 0x0303, 0x039a,
	0x0302, 0x0003, 0x0303, 0x003e, 0x0302, 0x0080, 0x0303, 0x1027, 0x0302,
	0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x00dd, 0x0302, 0x003a,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303,
	0xc090, 0x0302, 0x0003, 0x0303, 0xf2a7, 0x0302, 0x0084, 0x0303, 0x003a,
	0x0302, 0x003a, 0x0303, 0x0372, 0x0302, 0x0003, 0x0303, 0x00f3, 0x0302,
	0x0000, 0x0303, 0x9a0f, 0x0302, 0x009f, 0x0303, 0xc103, 0x0302, 0x001f,
	0x0303, 0x039a, 0x0302, 0x0003, 0x0303, 0x0046, 0x0302, 0x0080, 0x0303,
	0xf20f, 0x0302, 0x001f, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0327,
	0x0302, 0x0028, 0x0303, 0x0305, 0x0302, 0x00a9, 0x0303, 0x03c2, 0x0302,
	0x009d, 0x0303, 0x0365, 0x0302, 0x0028, 0x0303, 0x036a, 0x0302, 0x00a9,
	0x0303, 0x000f, 0x0302, 0x0000, 0x0303, 0xc0f0, 0x0302, 0x0003, 0x0303,
	0x001e, 0x0302, 0x003a, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303, 0x10ae,
	0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x0441, 0x0302,
	0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0,
	0x0303, 0x10a1, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0xc453, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a7,
	0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xdfff, 0x0302,
	0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0,
	0x0303, 0x10a6, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x000d, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144,
	0x0302, 0x00e0, 0x0303, 0x102d, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x0100, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x125f, 0x0302, 0x003a, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x070f, 0x0302, 0x00ba, 0x0303, 0x0353,
	0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x000f, 0x0302,
	0x0000, 0x0303, 0x4000, 0x0302, 0x007a, 0x0303, 0x03f0, 0x0302, 0x0003,
	0x0303, 0xc07c, 0x0302, 0x0083, 0x0303, 0x0030, 0x0302, 0x003a, 0x0303,
	0x03db, 0x0302, 0x0003, 0x0303, 0x00ff, 0x0302, 0x003a, 0x0303, 0x0030,
	0x0302, 0x005a, 0x0303, 0x03e5, 0x0302, 0x0083, 0x0303, 0x0010, 0x0302,
	0x00ba, 0x0303, 0x0030, 0x0302, 0x005a, 0x0303, 0x03e3, 0x0302, 0x0083,
	0x0303, 0x001c, 0x0302, 0x00ba, 0x0303, 0x000a, 0x0302, 0x005a, 0x0303,
	0x03dc, 0x0302, 0x0083, 0x0303, 0x000a, 0x0302, 0x003a, 0x0303, 0xf10c,
	0x0302, 0x001f, 0x0303, 0x03f1, 0x0302, 0x0083, 0x0303, 0xe80f, 0x0302,
	0x009f, 0x0303, 0x03d5, 0x0302, 0x009b, 0x0303, 0x03c9, 0x0302, 0x009c,
	0x0303, 0x03c6, 0x0302, 0x009c, 0x0303, 0x03e8, 0x0302, 0x0011, 0x0303,
	0x03e8, 0x0302, 0x0003, 0x0303, 0x0008, 0x0302, 0x00ba, 0x0303, 0xe90c,
	0x0302, 0x001f, 0x0303, 0x03e9, 0x0302, 0x0083, 0x0303, 0x9b4f, 0x0302,
	0x009f, 0x0303, 0x03fa, 0x0302, 0x0003, 0x0303, 0x10ae, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x1910, 0x0302, 0x003a, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a5,
	0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc053, 0x0302,
	0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a2, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303,
	0x0144, 0x0302, 0x00e0, 0x0303, 0x10a1, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0xc553, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302,
	0x00e0, 0x0303, 0x10a7, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0x8f00, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303,
	0x0144, 0x0302, 0x00e0, 0x0303, 0x10a6, 0x0302, 0x00ba, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x000c, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a4, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x85f0, 0x0302, 0x00ba, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x1266,
	0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xd553, 0x0302,
	0x0083, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x000f, 0x0302, 0x0000,
	0x0303, 0x1266, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x008c, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144,
	0x0302, 0x00e0, 0x0303, 0x00b3, 0x0302, 0x0080, 0x0303, 0x1266, 0x0302,
	0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x0096, 0x0302, 0x003a,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303,
	0x4000, 0x0302, 0x007a, 0x0303, 0x03f0, 0x0302, 0x0003, 0x0303, 0xc07c,
	0x0302, 0x0083, 0x0303, 0x0030, 0x0302, 0x003a, 0x0303, 0x03db, 0x0302,
	0x0003, 0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0x0005, 0x0302, 0x005a,
	0x0303, 0x03e5, 0x0302, 0x0083, 0x0303, 0x0014, 0x0302, 0x003a, 0x0303,
	0x0020, 0x0302, 0x00da, 0x0303, 0x03e3, 0x0302, 0x0083, 0x0303, 0x0024,
	0x0302, 0x003a, 0x0303, 0x000a, 0x0302, 0x005a, 0x0303, 0x03dc, 0x0302,
	0x0083, 0x0303, 0x0000, 0x0302, 0x003a, 0x0303, 0xf10c, 0x0302, 0x001f,
	0x0303, 0x03f1, 0x0302, 0x0083, 0x0303, 0x2000, 0x0302, 0x007a, 0x0303,
	0xe803, 0x0302, 0x0011, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0x0008,
	0x0302, 0x00ba, 0x0303, 0xe90c, 0x0302, 0x001f, 0x0303, 0x03e9, 0x0302,
	0x0083, 0x0303, 0x9b4f, 0x0302, 0x009f, 0x0303, 0x03fa, 0x0302, 0x0003,
	0x0303, 0x10ae, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x1910, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144,
	0x0302, 0x00e0, 0x0303, 0x10a5, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0xc753, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302, 0x00e0,
	0x0303, 0x10a2, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x0070, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144,
	0x0302, 0x00e0, 0x0303, 0x1027, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x00bb, 0x0302, 0x003a, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a1, 0x0302, 0x003a, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0xc553, 0x0302, 0x0003, 0x0303, 0x0144,
	0x0302, 0x00e0, 0x0303, 0x10a7, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x8f00, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a6, 0x0302, 0x00ba, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x000c, 0x0302, 0x003a, 0x0303, 0x0353,
	0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x10a4, 0x0302,
	0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x81f0, 0x0302, 0x003a,
	0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303,
	0x000f, 0x0302, 0x0000, 0x0303, 0xc153, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0093, 0x0303, 0x9b03, 0x0302, 0x001f, 0x0303, 0x00f9, 0x0302,
	0x00a0, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x53f5, 0x0302, 0x0003, 0x0303, 0x7207, 0x0302, 0x0083, 0x0303,
	0x0000, 0x0302, 0x0081, 0x0303, 0x10ae, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0x0149, 0x0302, 0x0060, 0x0303, 0xc153, 0x0302,
	0x0091, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0,
	0x0303, 0x10a1, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0xc453, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x1234,
	0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc053, 0x0302,
	0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0xc1f5, 0x0302, 0x0083,
	0x0303, 0x070f, 0x0302, 0x00ba, 0x0303, 0x03f6, 0x0302, 0x0003, 0x0303,
	0x9b14, 0x0302, 0x002c, 0x0303, 0x1023, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0x6853, 0x0302, 0x0083, 0x0303, 0x0144, 0x0302,
	0x00e0, 0x0303, 0x1082, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x684f, 0x0302, 0x009f, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303,
	0x0144, 0x0302, 0x00e0, 0x0303, 0x1079, 0x0302, 0x003a, 0x0303, 0x0352,
	0x0302, 0x0083, 0x0303, 0x6a53, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302,
	0x00e0, 0x0303, 0x107a, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083,
	0x0303, 0x6a4f, 0x0302, 0x001f, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303,
	0x0144, 0x0302, 0x00e0, 0x0303, 0x0125, 0x0302, 0x0000, 0x0303, 0x1023,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x6753, 0x0302,
	0x0083, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x1033, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x0149, 0x0302, 0x0060, 0x0303,
	0x8000, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0011, 0x0303, 0x0353,
	0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x5853, 0x0302,
	0x009b, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0,
	0x0303, 0x000f, 0x0302, 0x0000, 0x0303, 0x1682, 0x0302, 0x00ba, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0xffff, 0x0302, 0x003a, 0x0303, 0x039d,
	0x0302, 0x009b, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0144, 0x0302,
	0x00e0, 0x0303, 0x6667, 0x0302, 0x0083, 0x0303, 0x1681, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xc40f, 0x0302, 0x001f, 0x0303,
	0x6703, 0x0302, 0x001c, 0x0303, 0x03c1, 0x0302, 0x0091, 0x0303, 0x0353,
	0x0302, 0x0003, 0x0303, 0x0144, 0x0302, 0x00e0, 0x0303, 0x0149, 0x0302,
	0x0060, 0x0303, 0x5381, 0x0302, 0x000a, 0x0303, 0x6770, 0x0302, 0x001e,
	0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x678b, 0x0302, 0x0085, 0x0303,
	0x000f, 0x0302, 0x0000, 0x0303, 0x520f, 0x0302, 0x001f, 0x0303, 0x0358,
	0x0302, 0x009b, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x53b5, 0x0302,
	0x0083, 0x0303, 0x014c, 0x0302, 0x0000, 0x0303, 0x8000, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0091, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x52b4, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xb581,
	0x0302, 0x002f, 0x0303, 0xb502, 0x0302, 0x00aa, 0x0303, 0x0153, 0x0302,
	0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x014c, 0x0302, 0x0000,
	0x0303, 0xb553, 0x0302, 0x0086, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303,
	0xd100, 0x0302, 0x0090, 0x0303, 0x03b6, 0x0302, 0x0083, 0x0303, 0x6a0f,
	0x0302, 0x009f, 0x0303, 0x03c1, 0x0302, 0x0091, 0x0303, 0x03d1, 0x0302,
	0x0011, 0x0303, 0xe802, 0x0302, 0x008a, 0x0303, 0x0354, 0x0302, 0x0091,
	0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0xb600, 0x0302, 0x002f, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0x0386, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x4fc9, 0x0302, 0x001c, 0x0303,
	0x034f, 0x0302, 0x0083, 0x0303, 0x680f, 0x0302, 0x001f, 0x0303, 0x034f,
	0x0302, 0x0091, 0x0303, 0x0f4f, 0x0302, 0x0002, 0x0303, 0xee0f, 0x0302,
	0x009f, 0x0303, 0x03c9, 0x0302, 0x009c, 0x0303, 0x03c5, 0x0302, 0x0011,
	0x0303, 0x03d1, 0x0302, 0x0011, 0x0303, 0x03ee, 0x0302, 0x0003, 0x0303,
	0x0011, 0x0302, 0x0000, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xee0a, 0x0302, 0x00af,
	0x0303, 0x4f0f, 0x0302, 0x001f, 0x0303, 0x0307, 0x0302, 0x0083, 0x0303,
	0x03c9, 0x0302, 0x001d, 0x0303, 0x034f, 0x0302, 0x0083, 0x0303, 0xee0f,
	0x0302, 0x009f, 0x0303, 0x03c9, 0x0302, 0x001d, 0x0303, 0x03c7, 0x0302,
	0x009b, 0x0303, 0x03ee, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081,
	0x0303, 0x4e0f, 0x0302, 0x009f, 0x0303, 0x030e, 0x0302, 0x00a8, 0x0303,
	0x0302, 0x0302, 0x0029, 0x0303, 0x01c3, 0x0302, 0x0080, 0x0303, 0x0183,
	0x0302, 0x0000, 0x0303, 0xc24e, 0x0302, 0x0083, 0x0303, 0xe802, 0x0302,
	0x00ae, 0x0303, 0xc04e, 0x0302, 0x0003, 0x0303, 0xee91, 0x0302, 0x0083,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x5e00, 0x0302,
	0x0098, 0x0303, 0x035e, 0x0302, 0x0083, 0x0303, 0xe802, 0x0302, 0x00ae,
	0x0303, 0x03f0, 0x0302, 0x0003, 0x0303, 0x0026, 0x0302, 0x00ba, 0x0303,
	0x0f07, 0x0302, 0x0002, 0x0303, 0x00df, 0x0302, 0x0080, 0x0303, 0x4102,
	0x0302, 0x008c, 0x0303, 0x0033, 0x0302, 0x0000, 0x0303, 0x01d5, 0x0302,
	0x0060, 0x0303, 0xd72c, 0x0302, 0x002a, 0x0303, 0x002d, 0x0302, 0x003a,
	0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x00f3, 0x0302, 0x0000, 0x0303,
	0x0056, 0x0302, 0x0060, 0x0303, 0x0031, 0x0302, 0x00ba, 0x0303, 0x0f07,
	0x0302, 0x0002, 0x0303, 0x0112, 0x0302, 0x0080, 0x0303, 0x4188, 0x0302,
	0x0024, 0x0303, 0x8489, 0x0302, 0x00ad, 0x0303, 0xcd00, 0x0302, 0x0010,
	0x0303, 0x0341, 0x0302, 0x001b, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303,
	0x0039, 0x0302, 0x003a, 0x0303, 0x0f07, 0x0302, 0x0002, 0x0303, 0x009a,
	0x0302, 0x0000, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xd71b, 0x0302,
	0x00aa, 0x0303, 0x003e, 0x0302, 0x00ba, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x00f3, 0x0302, 0x0000, 0x0303, 0x0056, 0x0302, 0x0060, 0x0303,
	0x0042, 0x0302, 0x003a, 0x0303, 0x0f07, 0x0302, 0x0002, 0x0303, 0x0112,
	0x0302, 0x0080, 0x0303, 0x4248, 0x0302, 0x009f, 0x0303, 0x030a, 0x0302,
	0x002b, 0x0303, 0x418a, 0x0302, 0x000d, 0x0303, 0xcd41, 0x0302, 0x0091,
	0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x004a, 0x0302, 0x00ba, 0x0303,
	0x0f07, 0x0302, 0x0002, 0x0303, 0x009a, 0x0302, 0x0000, 0x0303, 0x4248,
	0x0302, 0x009f, 0x0303, 0x0302, 0x0302, 0x00ab, 0x0303, 0x001f, 0x0302,
	0x0080, 0x0303, 0x0050, 0x0302, 0x003a, 0x0303, 0x0367, 0x0302, 0x0083,
	0x0303, 0x00f3, 0x0302, 0x0000, 0x0303, 0x0053, 0x0302, 0x003a, 0x0303,
	0x0f07, 0x0302, 0x0002, 0x0303, 0x0112, 0x0302, 0x0080, 0x0303, 0xd702,
	0x0302, 0x002a, 0x0303, 0xd787, 0x0302, 0x002d, 0x0303, 0x0015, 0x0302,
	0x0080, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0xc36b, 0x0302, 0x0083,
	0x0303, 0x4102, 0x0302, 0x002f, 0x0303, 0xc26b, 0x0302, 0x0003, 0x0303,
	0x4041, 0x0302, 0x001f, 0x0303, 0x03c3, 0x0302, 0x001b, 0x0303, 0x036b,
	0x0302, 0x0097, 0x0303, 0x0064, 0x0302, 0x0020, 0x0303, 0x03c0, 0x0302,
	0x0017, 0x0303, 0x0062, 0x0302, 0x00a1, 0x0303, 0xc067, 0x0302, 0x0083,
	0x0303, 0x0065, 0x0302, 0x0000, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303,
	0x0065, 0x0302, 0x0000, 0x0303, 0xc367, 0x0302, 0x0083, 0x0303, 0x4031,
	0x0302, 0x009f, 0x0303, 0x03c3, 0x0302, 0x001b, 0x0303, 0x03c3, 0x0302,
	0x0017, 0x0303, 0x0072, 0x0302, 0x00a0, 0x0303, 0x03c0, 0x0302, 0x0017,
	0x0303, 0x0070, 0x0302, 0x00a1, 0x0303, 0x4103, 0x0302, 0x0088, 0x0303,
	0xc068, 0x0302, 0x0083, 0x0303, 0x0073, 0x0302, 0x0080, 0x0303, 0xc268,
	0x0302, 0x0003, 0x0303, 0x0073, 0x0302, 0x0080, 0x0303, 0xc168, 0x0302,
	0x0003, 0x0303, 0x0073, 0x0302, 0x0080, 0x0303, 0xc368, 0x0302, 0x0083,
	0x0303, 0x4041, 0x0302, 0x001f, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303,
	0x03c3, 0x0302, 0x001b, 0x0303, 0x036b, 0x0302, 0x0097, 0x0303, 0x0080,
	0x0302, 0x0020, 0x0303, 0x03c0, 0x0302, 0x0017, 0x0303, 0x007d, 0x0302,
	0x0021, 0x0303, 0xc069, 0x0302, 0x0003, 0x0303, 0x4108, 0x0302, 0x0029,
	0x0303, 0x0088, 0x0302, 0x0000, 0x0303, 0xc169, 0x0302, 0x0083, 0x0303,
	0x4105, 0x0302, 0x00a9, 0x0303, 0x0088, 0x0302, 0x0000, 0x0303, 0x4183,
	0x0302, 0x0029, 0x0303, 0xc369, 0x0302, 0x0003, 0x0303, 0x0088, 0x0302,
	0x0000, 0x0303, 0x0064, 0x0302, 0x00ba, 0x0303, 0x035e, 0x0302, 0x0097,
	0x0303, 0x0088, 0x0302, 0x0041, 0x0303, 0xc269, 0x0302, 0x0091, 0x0303,
	0x0369, 0x0302, 0x0003, 0x0303, 0x670f, 0x0302, 0x001f, 0x0303, 0x0368,
	0x0302, 0x0091, 0x0303, 0x0369, 0x0302, 0x0011, 0x0303, 0x0304, 0x0302,
	0x0008, 0x0303, 0xcb41, 0x0302, 0x0091, 0x0303, 0x0341, 0x0302, 0x0003,
	0x0303, 0x0092, 0x0302, 0x0080, 0x0303, 0xcb00, 0x0302, 0x0010, 0x0303,
	0x0341, 0x0302, 0x001b, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x670f,
	0x0302, 0x001f, 0x0303, 0x0368, 0x0302, 0x009b, 0x0303, 0x0369, 0x0302,
	0x001b, 0x0303, 0x0302, 0x0302, 0x0029, 0x0303, 0x0099, 0x0302, 0x0000,
	0x0303, 0xc941, 0x0302, 0x0011, 0x0303, 0x0341, 0x0302, 0x0003, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0xc042, 0x0302, 0x0003, 0x0303, 0x4104,
	0x0302, 0x008c, 0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x0342, 0x0302,
	0x0003, 0x0303, 0x00d6, 0x0302, 0x0080, 0x0303, 0x4102, 0x0302, 0x000e,
	0x0303, 0x00d6, 0x0302, 0x0080, 0x0303, 0xc142, 0x0302, 0x0083, 0x0303,
	0x4102, 0x0302, 0x002f, 0x0303, 0xc242, 0x0302, 0x0083, 0x0303, 0x4103,
	0x0302, 0x0028, 0x0303, 0xc067, 0x0302, 0x0083, 0x0303, 0x00a8, 0x0302,
	0x0080, 0x0303, 0xc467, 0x0302, 0x0003, 0x0303, 0x000c, 0x0302, 0x003a,
	0x0303, 0x0300, 0x0302, 0x0090, 0x0303, 0x0342, 0x0302, 0x001b, 0x0303,
	0x0367, 0x0302, 0x0091, 0x0303, 0x0342, 0x0302, 0x0003, 0x0303, 0xcb68,
	0x0302, 0x0003, 0x0303, 0xc969, 0x0302, 0x0003, 0x0303, 0x4103, 0x0302,
	0x00af, 0x0303, 0xc968, 0x0302, 0x0083, 0x0303, 0xcb69, 0x0302, 0x0083,
	0x0303, 0x4102, 0x0302, 0x0029, 0x0303, 0x00c6, 0x0302, 0x0000, 0x0303,
	0x4104, 0x0302, 0x008f, 0x0303, 0x01ff, 0x0302, 0x00ba, 0x0303, 0x03ea,
	0x0302, 0x009b, 0x0303, 0x00be, 0x0302, 0x0021, 0x0303, 0xf561, 0x0302,
	0x001f, 0x0303, 0x03c1, 0x0302, 0x0097, 0x0303, 0x00be, 0x0302, 0x00a0,
	0x0303, 0x03c0, 0x0302, 0x0017, 0x0303, 0x00c2, 0x0302, 0x0020, 0x0303,
	0x00c6, 0x0302, 0x0000, 0x0303, 0xd541, 0x0302, 0x0091, 0x0303, 0x0341,
	0x0302, 0x0003, 0x0303, 0xc067, 0x0302, 0x0083, 0x0303, 0x00c7, 0x0302,
	0x0080, 0x0303, 0xd541, 0x0302, 0x0091, 0x0303, 0x0341, 0x0302, 0x0003,
	0x0303, 0x6867, 0x0302, 0x0003, 0x0303, 0x00c7, 0x0302, 0x0080, 0x0303,
	0x6967, 0x0302, 0x0083, 0x0303, 0x0030, 0x0302, 0x003a, 0x0303, 0x0300,
	0x0302, 0x0090, 0x0303, 0x0342, 0x0302, 0x001b, 0x0303, 0x0367, 0x0302,
	0x0091, 0x0303, 0x0342, 0x0302, 0x0003, 0x0303, 0x4103, 0x0302, 0x000a,
	0x0303, 0x2000, 0x0302, 0x00ba, 0x0303, 0x0342, 0x0302, 0x0003, 0x0303,
	0x4103, 0x0302, 0x008b, 0x0303, 0x1000, 0x0302, 0x00ba, 0x0303, 0x0342,
	0x0302, 0x0003, 0x0303, 0x000c, 0x0302, 0x003a, 0x0303, 0x0300, 0x0302,
	0x0090, 0x0303, 0x0341, 0x0302, 0x001b, 0x0303, 0x0341, 0x0302, 0x0003,
	0x0303, 0x15a5, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x01c9, 0x0302, 0x00e0, 0x0303, 0xcf00, 0x0302, 0x003a, 0x0303, 0x0353,
	0x0302, 0x001b, 0x0303, 0x0342, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0081,
	0x0303, 0xd611, 0x0302, 0x00ab, 0x0303, 0xc067, 0x0302, 0x0083, 0x0303,
	0xc068, 0x0302, 0x0083, 0x0303, 0x15c7, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0, 0x0303, 0x000c, 0x0302,
	0x003a, 0x0303, 0x5303, 0x0302, 0x001d, 0x0303, 0x03c1, 0x0302, 0x009b,
	0x0303, 0x0367, 0x0302, 0x0019, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303,
	0x6800, 0x0302, 0x0098, 0x0303, 0x0368, 0x0302, 0x0083, 0x0303, 0x038a,
	0x0302, 0x000b, 0x0303, 0x67c4, 0x0302, 0x001e, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0303, 0x0302, 0x002b, 0x0303, 0xc941, 0x0302, 0x0011,
	0x0303, 0x0341, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303,
	0x4fc9, 0x0302, 0x001c, 0x0303, 0x034f, 0x0302, 0x0083, 0x0303, 0x670f,
	0x0302, 0x001f, 0x0303, 0x034f, 0x0302, 0x0091, 0x0303, 0x0f4f, 0x0302,
	0x0002, 0x0303, 0xee0f, 0x0302, 0x009f, 0x0303, 0x03c9, 0x0302, 0x009c,
	0x0303, 0x03c6, 0x0302, 0x0011, 0x0303, 0x03d1, 0x0302, 0x0011, 0x0303,
	0x03ee, 0x0302, 0x0003, 0x0303, 0x9d0d, 0x0302, 0x00ac, 0x0303, 0xc075,
	0x0302, 0x0083, 0x0303, 0x830c, 0x0302, 0x000b, 0x0303, 0xe80a, 0x0302,
	0x002e, 0x0303, 0x5ec2, 0x0302, 0x0017, 0x0303, 0x010a, 0x0302, 0x00c1,
	0x0303, 0x0000, 0x0302, 0x0093, 0x0303, 0xe243, 0x0302, 0x001f, 0x0303,
	0x0372, 0x0302, 0x0003, 0x0303, 0x0900, 0x0302, 0x003a, 0x0303, 0x0372,
	0x0302, 0x0099, 0x0303, 0xea03, 0x0302, 0x0097, 0x0303, 0x010b, 0x0302,
	0x0041, 0x0303, 0xc175, 0x0302, 0x0003, 0x0303, 0x7591, 0x0302, 0x0003,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x1090, 0x0302,
	0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0,
	0x0303, 0x536a, 0x0302, 0x0003, 0x0303, 0x1003, 0x0302, 0x00ba, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0, 0x0303, 0xc853,
	0x0302, 0x0091, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x0369, 0x0302,
	0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x411c, 0x0302, 0x002a,
	0x0303, 0x01d8, 0x0302, 0x00e0, 0x0303, 0x15c4, 0x0302, 0x003a, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0, 0x0303, 0x8410,
	0x0302, 0x002a, 0x0303, 0x5384, 0x0302, 0x0088, 0x0303, 0xd743, 0x0302,
	0x001f, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x40ce, 0x0302, 0x001b,
	0x0303, 0x0367, 0x0302, 0x0097, 0x0303, 0x012d, 0x0302, 0x00a1, 0x0303,
	0x67c9, 0x0302, 0x001c, 0x0303, 0x0367, 0x0302, 0x0091, 0x0303, 0x0340,
	0x0302, 0x0083, 0x0303, 0x012e, 0x0302, 0x0080, 0x0303, 0x6740, 0x0302,
	0x0006, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x400f, 0x0302, 0x001f,
	0x0303, 0x03c6, 0x0302, 0x009c, 0x0303, 0x0309, 0x0302, 0x002b, 0x0303,
	0x1003, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc800,
	0x0302, 0x0010, 0x0303, 0x0369, 0x0302, 0x001b, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x013a, 0x0302, 0x0080,
	0x0303, 0xd740, 0x0302, 0x0083, 0x0303, 0x15a0, 0x0302, 0x00ba, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0, 0x0303, 0x530f,
	0x0302, 0x009f, 0x0303, 0x4102, 0x0302, 0x00ab, 0x0303, 0x0300, 0x0302,
	0x0090, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x001f, 0x0302, 0x00ba,
	0x0303, 0x0367, 0x0302, 0x009b, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303,
	0x15a1, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9,
	0x0302, 0x00e0, 0x0303, 0x5368, 0x0302, 0x0083, 0x0303, 0x15a2, 0x0302,
	0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c9, 0x0302, 0x00e0,
	0x0303, 0x530f, 0x0302, 0x009f, 0x0303, 0x4102, 0x0302, 0x00ab, 0x0303,
	0x0300, 0x0302, 0x0090, 0x0303, 0x0369, 0x0302, 0x0003, 0x0303, 0x003f,
	0x0302, 0x003a, 0x0303, 0x0369, 0x0302, 0x001b, 0x0303, 0x0369, 0x0302,
	0x0003, 0x0303, 0x1090, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0xe0c0, 0x0302, 0x00ba, 0x0303, 0x036a, 0x0302, 0x001b, 0x0303,
	0x036a, 0x0302, 0x0003, 0x0303, 0x67c8, 0x0302, 0x009c, 0x0303, 0x036a,
	0x0302, 0x0011, 0x0303, 0x0369, 0x0302, 0x0011, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x108f, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x6769, 0x0302, 0x0019, 0x0303,
	0x0f6a, 0x0302, 0x0082, 0x0303, 0x03c2, 0x0302, 0x001c, 0x0303, 0x036a,
	0x0302, 0x0099, 0x0303, 0x0f6a, 0x0302, 0x0082, 0x0303, 0x68c3, 0x0302,
	0x001c, 0x0303, 0x036a, 0x0302, 0x0099, 0x0303, 0x0f6a, 0x0302, 0x0082,
	0x0303, 0x01db, 0x0302, 0x00ba, 0x0303, 0xe802, 0x0302, 0x000b, 0x0303,
	0x0194, 0x0302, 0x003a, 0x0303, 0x036a, 0x0302, 0x0017, 0x0303, 0x016d,
	0x0302, 0x0041, 0x0303, 0xc767, 0x0302, 0x0003, 0x0303, 0x017c, 0x0302,
	0x0000, 0x0303, 0x01ae, 0x0302, 0x003a, 0x0303, 0xe802, 0x0302, 0x000b,
	0x0303, 0x015c, 0x0302, 0x00ba, 0x0303, 0x036a, 0x0302, 0x0017, 0x0303,
	0x0174, 0x0302, 0x00c1, 0x0303, 0xc667, 0x0302, 0x0083, 0x0303, 0x017c,
	0x0302, 0x0000, 0x0303, 0x0181, 0x0302, 0x00ba, 0x0303, 0xe802, 0x0302,
	0x000b, 0x0303, 0x013c, 0x0302, 0x00ba, 0x0303, 0x036a, 0x0302, 0x0017,
	0x0303, 0x017b, 0x0302, 0x00c1, 0x0303, 0xc567, 0x0302, 0x0083, 0x0303,
	0x017c, 0x0302, 0x0000, 0x0303, 0xc467, 0x0302, 0x0003, 0x0303, 0x67c4,
	0x0302, 0x009c, 0x0303, 0x03c8, 0x0302, 0x0091, 0x0303, 0x03c8, 0x0302,
	0x001c, 0x0303, 0x0368, 0x0302, 0x0091, 0x0303, 0x0353, 0x0302, 0x0003,
	0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303,
	0x1010, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc653,
	0x0302, 0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x9a33, 0x0302,
	0x000c, 0x0303, 0x9d29, 0x0302, 0x0009, 0x0303, 0x1004, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x0808, 0x0302, 0x003a, 0x0303,
	0x0f53, 0x0302, 0x0082, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x1003,
	0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c4, 0x0302,
	0x0060, 0x0303, 0x4105, 0x0302, 0x00af, 0x0303, 0x1010, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc753, 0x0302, 0x0083, 0x0303,
	0x01c4, 0x0302, 0x0060, 0x0303, 0x1580, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x01c4, 0x0302,
	0x0060, 0x0303, 0x1581, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0xc353, 0x0302, 0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303,
	0xc053, 0x0302, 0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x1588,
	0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc353, 0x0302,
	0x0003, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x15a4, 0x0302, 0x003a,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0xc153, 0x0302, 0x0083, 0x0303,
	0x01c4, 0x0302, 0x0060, 0x0303, 0x15a5, 0x0302, 0x00ba, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0xc053, 0x0302, 0x0003, 0x0303, 0x0f00, 0x0302,
	0x003a, 0x0303, 0x039d, 0x0302, 0x009b, 0x0303, 0x0353, 0x0302, 0x0011,
	0x0303, 0x0f53, 0x0302, 0x0082, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303,
	0x0015, 0x0302, 0x0080, 0x0303, 0x1004, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x0404, 0x0302, 0x003a, 0x0303, 0x0f53, 0x0302,
	0x0082, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x1003, 0x0302, 0x00ba,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303,
	0x0015, 0x0302, 0x0080, 0x0303, 0x1004, 0x0302, 0x003a, 0x0303, 0x0f52,
	0x0302, 0x0002, 0x0303, 0x0202, 0x0302, 0x003a, 0x0303, 0x0f53, 0x0302,
	0x0082, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303, 0x1003, 0x0302, 0x00ba,
	0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x01c4, 0x0302, 0x0060, 0x0303,
	0x0015, 0x0302, 0x0080, 0x0303, 0x01c3, 0x0302, 0x0080, 0x0303, 0x520f,
	0x0302, 0x001f, 0x0303, 0x0358, 0x0302, 0x009b, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x53b5, 0x0302, 0x0083, 0x0303, 0x01cc, 0x0302, 0x0080,
	0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0091, 0x0303,
	0x0f52, 0x0302, 0x0002, 0x0303, 0x52b4, 0x0302, 0x0083, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0xb581, 0x0302, 0x002f, 0x0303, 0xb502, 0x0302,
	0x00aa, 0x0303, 0x01d3, 0x0302, 0x0000, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x01cc, 0x0302, 0x0080, 0x0303, 0xb553, 0x0302, 0x0086, 0x0303,
	0x0000, 0x0302, 0x00e1, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303, 0xd568,
	0x0302, 0x0003, 0x0303, 0x01da, 0x0302, 0x0000, 0x0303, 0xc267, 0x0302,
	0x0003, 0x0303, 0x6d68, 0x0302, 0x0003, 0x0303, 0x6700, 0x0302, 0x0010,
	0x0303, 0x0387, 0x0302, 0x001b, 0x0303, 0x0387, 0x0302, 0x0003, 0x0303,
	0x8475, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x7572,
	0x0302, 0x0083, 0x0303, 0x0016, 0x0302, 0x00ba, 0x0303, 0x8403, 0x0302,
	0x001d, 0x0303, 0x0384, 0x0302, 0x0011, 0x0303, 0x0375, 0x0302, 0x0083,
	0x0303, 0x6800, 0x0302, 0x0010, 0x0303, 0x0375, 0x0302, 0x009b, 0x0303,
	0x0384, 0x0302, 0x0003, 0x0303, 0x8475, 0x0302, 0x0083, 0x0303, 0x720f,
	0x0302, 0x009f, 0x0303, 0x0375, 0x0302, 0x0097, 0x0303, 0x01df, 0x0302,
	0x0021, 0x0303, 0x670f, 0x0302, 0x001f, 0x0303, 0x0387, 0x0302, 0x0011,
	0x0303, 0x0387, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x6204, 0x0302, 0x0084, 0x0303, 0x6207, 0x0302, 0x0003,
	0x0303, 0xc062, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303,
	0xc04e, 0x0302, 0x0003, 0x0303, 0x9917, 0x0302, 0x0028, 0x0303, 0xb603,
	0x0302, 0x00ab, 0x0303, 0x9982, 0x0302, 0x002b, 0x0303, 0x010f, 0x0302,
	0x0080, 0x0303, 0xb60f, 0x0302, 0x001f, 0x0303, 0x0023, 0x0302, 0x0080,
	0x0303, 0x000b, 0x0302, 0x0080, 0x0303, 0x9b03, 0x0302, 0x002f, 0x0303,
	0x9d07, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303, 0x9d0f,
	0x0302, 0x001f, 0x0303, 0xe801, 0x0302, 0x001f, 0x0303, 0x039d, 0x0302,
	0x0083, 0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x1000, 0x0302, 0x00da,
	0x0303, 0x03b6, 0x0302, 0x009b, 0x0303, 0x03b6, 0x0302, 0x0083, 0x0303,
	0x9d0f, 0x0302, 0x001f, 0x0303, 0xe802, 0x0302, 0x008a, 0x0303, 0x0354,
	0x0302, 0x0091, 0x0303, 0x03d1, 0x0302, 0x0011, 0x0303, 0x03b7, 0x0302,
	0x0003, 0x0303, 0x000b, 0x0302, 0x0080, 0x0303, 0x00d0, 0x0302, 0x0080,
	0x0303, 0x030c, 0x0302, 0x0028, 0x0303, 0x030d, 0x0302, 0x0029, 0x0303,
	0x03c2, 0x0302, 0x009d, 0x0303, 0x030f, 0x0302, 0x0028, 0x0303, 0x030f,
	0x0302, 0x00a9, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x0307, 0x0302,
	0x00a8, 0x0303, 0x030a, 0x0302, 0x00a9, 0x0303, 0x03c2, 0x0302, 0x009d,
	0x0303, 0x0307, 0x0302, 0x00a8, 0x0303, 0x0305, 0x0302, 0x00a9, 0x0303,
	0x000d, 0x0302, 0x0080, 0x0303, 0x00cb, 0x0302, 0x0080, 0x0303, 0x0093,
	0x0302, 0x0000, 0x0303, 0x0056, 0x0302, 0x0000, 0x0303, 0x00c9, 0x0302,
	0x0000, 0x0303, 0x00b6, 0x0302, 0x0080, 0x0303, 0x00a6, 0x0302, 0x0000,
	0x0303, 0x0071, 0x0302, 0x0000, 0x0303, 0x0075, 0x0302, 0x0080, 0x0303,
	0x6a03, 0x0302, 0x0028, 0x0303, 0xc280, 0x0302, 0x0003, 0x0303, 0x003b,
	0x0302, 0x0080, 0x0303, 0xc380, 0x0302, 0x0083, 0x0303, 0x6753, 0x0302,
	0x0083, 0x0303, 0x9b02, 0x0302, 0x00ac, 0x0303, 0x6853, 0x0302, 0x0083,
	0x0303, 0x1023, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x00df, 0x0302, 0x00e0, 0x0303, 0x9b0f, 0x0302, 0x002c, 0x0303, 0x1082,
	0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x684f, 0x0302,
	0x009f, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x00df, 0x0302, 0x00e0,
	0x0303, 0x1079, 0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303,
	0x6953, 0x0302, 0x0003, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x107a,
	0x0302, 0x003a, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x694f, 0x0302,
	0x001f, 0x0303, 0x0353, 0x0302, 0x0003, 0x0303, 0x00df, 0x0302, 0x00e0,
	0x0303, 0x6b04, 0x0302, 0x0028, 0x0303, 0x0054, 0x0302, 0x00ba, 0x0303,
	0x0307, 0x0302, 0x0083, 0x0303, 0x0101, 0x0302, 0x0000, 0x0303, 0x7507,
	0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303, 0x005c, 0x0302,
	0x003a, 0x0303, 0x0362, 0x0302, 0x0083, 0x0303, 0xc24e, 0x0302, 0x0083,
	0x0303, 0xc567, 0x0302, 0x0083, 0x0303, 0xc7ee, 0x0302, 0x0083, 0x0303,
	0x0128, 0x0302, 0x0080, 0x0303, 0xc06a, 0x0302, 0x0003, 0x0303, 0x9c6b,
	0x0302, 0x0083, 0x0303, 0x3034, 0x0302, 0x00ba, 0x0303, 0x0367, 0x0302,
	0x0083, 0x0303, 0x3039, 0x0302, 0x003a, 0x0303, 0x000b, 0x0302, 0x00da,
	0x0303, 0x0368, 0x0302, 0x0083, 0x0303, 0xc3c3, 0x0302, 0x003a, 0x0303,
	0x00c3, 0x0302, 0x005a, 0x0303, 0x0369, 0x0302, 0x0003, 0x0303, 0x0069,
	0x0302, 0x003a, 0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x0037, 0x0302,
	0x0080, 0x0303, 0xc19c, 0x0302, 0x0083, 0x0303, 0xc099, 0x0302, 0x0003,
	0x0303, 0x1804, 0x0302, 0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0xc153, 0x0302, 0x0083, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x8300,
	0x0302, 0x000b, 0x0303, 0x0012, 0x0302, 0x0000, 0x0303, 0x1892, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x5053, 0x0302, 0x0003,
	0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x007b, 0x0302, 0x003a, 0x0303,
	0x0362, 0x0302, 0x0083, 0x0303, 0xc14e, 0x0302, 0x0083, 0x0303, 0xc567,
	0x0302, 0x0083, 0x0303, 0xc7ee, 0x0302, 0x0083, 0x0303, 0x0128, 0x0302,
	0x0080, 0x0303, 0xf099, 0x0302, 0x0003, 0x0303, 0xc16a, 0x0302, 0x0083,
	0x0303, 0x9c0f, 0x0302, 0x009f, 0x0303, 0x03c1, 0x0302, 0x009d, 0x0303,
	0x036b, 0x0302, 0x0083, 0x0303, 0xc29c, 0x0302, 0x0083, 0x0303, 0x3005,
	0x0302, 0x003a, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303, 0x3009, 0x0302,
	0x003a, 0x0303, 0x000c, 0x0302, 0x005a, 0x0303, 0x0368, 0x0302, 0x0083,
	0x0303, 0xe1e1, 0x0302, 0x003a, 0x0303, 0x0021, 0x0302, 0x005a, 0x0303,
	0x0369, 0x0302, 0x0003, 0x0303, 0x008c, 0x0302, 0x00ba, 0x0303, 0x0375,
	0x0302, 0x0083, 0x0303, 0x0037, 0x0302, 0x0080, 0x0303, 0xffff, 0x0302,
	0x00fa, 0x0303, 0x03e2, 0x0302, 0x001b, 0x0303, 0x0092, 0x0302, 0x00a1,
	0x0303, 0x0d80, 0x0302, 0x00fa, 0x0303, 0x03e2, 0x0302, 0x0011, 0x0303,
	0x03e2, 0x0302, 0x0003, 0x0303, 0x0012, 0x0302, 0x0000, 0x0303, 0x0065,
	0x0302, 0x003a, 0x0303, 0x0381, 0x0302, 0x0003, 0x0303, 0x1892, 0x0302,
	0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x5153, 0x0302, 0x0083,
	0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x1821, 0x0302, 0x003a, 0x0303,
	0x0352, 0x0302, 0x0083, 0x0303, 0x00e4, 0x0302, 0x0060, 0x0303, 0xd500,
	0x0302, 0x0010, 0x0303, 0x0353, 0x0302, 0x001b, 0x0303, 0x0353, 0x0302,
	0x0003, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x182a, 0x0302, 0x00ba,
	0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0x9a4f, 0x0302, 0x001f, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x0012,
	0x0302, 0x0000, 0x0303, 0xc099, 0x0302, 0x0003, 0x0303, 0x9b0b, 0x0302,
	0x002d, 0x0303, 0x1821, 0x0302, 0x003a, 0x0303, 0x0f52, 0x0302, 0x0002,
	0x0303, 0x00e4, 0x0302, 0x0060, 0x0303, 0xd553, 0x0302, 0x0091, 0x0303,
	0x0353, 0x0302, 0x0003, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x182a,
	0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0083, 0x0303, 0xcb53, 0x0302,
	0x0083, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0xc06a, 0x0302, 0x0003,
	0x0303, 0xc06b, 0x0302, 0x0083, 0x0303, 0xc89c, 0x0302, 0x0083, 0x0303,
	0x0081, 0x0302, 0x0000, 0x0303, 0x9a0f, 0x0302, 0x009f, 0x0303, 0x03c2,
	0x0302, 0x009d, 0x0303, 0x036a, 0x0302, 0x0003, 0x0303, 0x9c0f, 0x0302,
	0x009f, 0x0303, 0x03c2, 0x0302, 0x009d, 0x0303, 0x036b, 0x0302, 0x0083,
	0x0303, 0x3026, 0x0302, 0x00ba, 0x0303, 0x0367, 0x0302, 0x0083, 0x0303,
	0x3029, 0x0302, 0x00ba, 0x0303, 0x000e, 0x0302, 0x00da, 0x0303, 0x0368,
	0x0302, 0x0083, 0x0303, 0xb4b4, 0x0302, 0x003a, 0x0303, 0x00b4, 0x0302,
	0x003a, 0x0303, 0x0369, 0x0302, 0x0003, 0x0303, 0x00c7, 0x0302, 0x00ba,
	0x0303, 0x0375, 0x0302, 0x0083, 0x0303, 0x0037, 0x0302, 0x0080, 0x0303,
	0xc49c, 0x0302, 0x0083, 0x0303, 0x0012, 0x0302, 0x0000, 0x0303, 0x9a6a,
	0x0302, 0x0003, 0x0303, 0x005d, 0x0302, 0x0080, 0x0303, 0xd199, 0x0302,
	0x0003, 0x0303, 0xf20f, 0x0302, 0x001f, 0x0303, 0xb663, 0x0302, 0x001f,
	0x0303, 0x03f2, 0x0302, 0x0083, 0x0303, 0x0012, 0x0302, 0x0000, 0x0303,
	0xf20f, 0x0302, 0x001f, 0x0303, 0x9903, 0x0302, 0x009f, 0x0303, 0x03f2,
	0x0302, 0x0083, 0x0303, 0xc099, 0x0302, 0x0003, 0x0303, 0xf204, 0x0302,
	0x002b, 0x0303, 0x00de, 0x0302, 0x003a, 0x0303, 0x039d, 0x0302, 0x0083,
	0x0303, 0x0075, 0x0302, 0x0080, 0x0303, 0x00db, 0x0302, 0x003a, 0x0303,
	0x039d, 0x0302, 0x0083, 0x0303, 0x0056, 0x0302, 0x0000, 0x0303, 0x00de,
	0x0302, 0x003a, 0x0303, 0x039d, 0x0302, 0x0083, 0x0303, 0x00c9, 0x0302,
	0x0000, 0x0303, 0x000b, 0x0302, 0x0080, 0x0303, 0x520f, 0x0302, 0x001f,
	0x0303, 0x0358, 0x0302, 0x009b, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303,
	0x53b5, 0x0302, 0x0083, 0x0303, 0x00e7, 0x0302, 0x0000, 0x0303, 0x8000,
	0x0302, 0x00ba, 0x0303, 0x0352, 0x0302, 0x0091, 0x0303, 0x0f52, 0x0302,
	0x0002, 0x0303, 0x52b4, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0xb581, 0x0302, 0x002f, 0x0303, 0xb502, 0x0302, 0x00aa, 0x0303,
	0x00ee, 0x0302, 0x0000, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x00e7,
	0x0302, 0x0000, 0x0303, 0xb553, 0x0302, 0x0086, 0x0303, 0x0000, 0x0302,
	0x00e1, 0x0303, 0xd100, 0x0302, 0x0090, 0x0303, 0x03b6, 0x0302, 0x0083,
	0x0303, 0x6a0f, 0x0302, 0x009f, 0x0303, 0x03c1, 0x0302, 0x0091, 0x0303,
	0x03d1, 0x0302, 0x0011, 0x0303, 0xe802, 0x0302, 0x008a, 0x0303, 0x0354,
	0x0302, 0x0091, 0x0303, 0x03b7, 0x0302, 0x0003, 0x0303, 0xb600, 0x0302,
	0x002f, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x0386, 0x0302, 0x0083,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x00e1, 0x0303, 0x1033, 0x0302,
	0x00ba, 0x0303, 0x0f52, 0x0302, 0x0002, 0x0303, 0x00e4, 0x0302, 0x0060,
	0x0303, 0x8000, 0x0302, 0x00ba, 0x0303, 0x0353, 0x0302, 0x0011, 0x0303,
	0x0f53, 0x0302, 0x0082, 0x0303, 0x00df, 0x0302, 0x00e0, 0x0303, 0x5853,
	0x0302, 0x009b, 0x0303, 0x0f53, 0x0302, 0x0082, 0x0303, 0x00df, 0x0302,
	0x00e0, 0x0303, 0xf906, 0x0302, 0x003a, 0x0303, 0x03d1, 0x0302, 0x0011,
	0x0303, 0x00fa, 0x0302, 0x0060, 0x0303, 0x0000, 0x0302, 0x0081, 0x0303,
	0xc7ee, 0x0302, 0x0083, 0x0303, 0x9905, 0x0302, 0x00aa, 0x0303, 0x9947,
	0x0302, 0x001b, 0x0303, 0x011c, 0x0302, 0x00a0, 0x0303, 0x03c7, 0x0302,
	0x009b, 0x0303, 0x011c, 0x0302, 0x0021, 0x0303, 0xc067, 0x0302, 0x0083,
	0x0303, 0x8302, 0x0302, 0x008b, 0x0303, 0xc167, 0x0302, 0x0003, 0x0303,
	0xc100, 0x0302, 0x0010, 0x0303, 0x03e8, 0x0302, 0x001b, 0x0303, 0x03e8,
	0x0302, 0x0003, 0x0303, 0x0128, 0x0302, 0x0080, 0x0303, 0xc24e, 0x0302,
	0x0083, 0x0303, 0x8004, 0x0302, 0x0088, 0x0303, 0x8302, 0x0302, 0x008b,
	0x0303, 0x0121, 0x0302, 0x0080, 0x0303, 0xc14e, 0x0302, 0x0083, 0x0303,
	0xc367, 0x0302, 0x0083, 0x0303, 0xe80f, 0x0302, 0x009f, 0x0303, 0x03c1,
	0x0302, 0x0091, 0x0303, 0x03e8, 0x0302, 0x0003, 0x0303, 0x0128, 0x0302,
	0x0080, 0x0303, 0xee67, 0x0302, 0x0083, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x6791, 0x0302, 0x0003, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303,
	0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000,
	0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302,
	0x0042, 0x0303, 0x0000, 0x0302, 0x0042, 0x0303, 0x0000, 0x0302, 0x0042,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8000, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8001, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8002,
	0x1222, 0x0005, 0x1221, 0x4818, 0x1220, 0x8003, 0x1222, 0x000b, 0x1221,
	0x4818, 0x1220, 0x8004, 0x1222, 0x0017, 0x1221, 0x4820, 0x1220, 0x8005,
	0x1222, 0x001f, 0x1221, 0x0000, 0x1220, 0x8006, 0x1222, 0x0010, 0x1221,
	0x0000, 0x1220, 0x8007, 0x1222, 0x0010, 0x1221, 0x0000, 0x1220, 0x8008,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8009, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x800a, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x800b,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x800c, 0x1222, 0x0028, 0x1221,
	0x0000, 0x1220, 0x800d, 0x1222, 0x0001, 0x1221, 0x0000, 0x1220, 0x800e,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x800f, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8010, 0x1222, 0x0a60, 0x1221, 0x8000, 0x1220, 0x8011,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8012, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8013, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8014,
	0x1222, 0x001e, 0x1221, 0x0000, 0x1220, 0x8015, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8016, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8017,
	0x1222, 0x7fff, 0x1221, 0x0000, 0x1220, 0x8018, 0x1222, 0x0003, 0x1221,
	0x0000, 0x1220, 0x8019, 0x1222, 0x0003, 0x1221, 0x0000, 0x1220, 0x801a,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x801b, 0x1222, 0x0003, 0x1221,
	0x0000, 0x1220, 0x801c, 0x1222, 0x0400, 0x1221, 0x0000, 0x1220, 0x801d,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x801e, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x801f, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8020,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8021, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8022, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8023,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8024, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8025, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8026,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8027, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8028, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8029,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x802a, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x802b, 0x1222, 0x0f70, 0x1221, 0x8000, 0x1220, 0x802c,
	0x1222, 0x0100, 0x1221, 0x0000, 0x1220, 0x802d, 0x1222, 0x007f, 0x1221,
	0x0000, 0x1220, 0x802e, 0x1222, 0x0040, 0x1221, 0x8000, 0x1220, 0x802f,
	0x1222, 0x0001, 0x1221, 0x0000, 0x1220, 0x8030, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8031, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8032,
	0x1222, 0x00fa, 0x1221, 0x0000, 0x1220, 0x8033, 0x1222, 0x00d2, 0x1221,
	0x0000, 0x1220, 0x8034, 0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8035,
	0x1222, 0x02aa, 0x1221, 0x0000, 0x1220, 0x8036, 0x1222, 0x0000, 0x1221,
	0x0000, 0x1220, 0x8037, 0x1222, 0x0014, 0x1221, 0x8000, 0x1220, 0x8038,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x8039, 0x1222, 0x47ab, 0x1221,
	0x0000, 0x1220, 0x803a, 0x1222, 0x0554, 0x1221, 0x0000, 0x1220, 0x803b,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x803c, 0x1222, 0x03ff, 0x1221,
	0x0000, 0x1220, 0x803d, 0x1222, 0x1fff, 0x1221, 0x8000, 0x1220, 0x803e,
	0x1222, 0x0000, 0x1221, 0x0000, 0x1220, 0x803f, 0x102f, 0xc0f0, 0x1000,
	0x0000, 0x100b, 0xf182, 0x1001, 0x0000, 0x1007, 0x0004, 0x1228, 0x0000,
	0x1235, 0x0000, 0x1a35, 0x0000, 0x1327, 0x0000, 0x132b, 0x0000, 0x0003,
	0x0001, 0x0004, 0x0001, 0x0005, 0x0001, 0x0006, 0x0001, 0x0007, 0x0001,
	0x0301, 0x0000, 0x1320, 0x0000, 0x1324, 0x0000, 0x1328, 0x0000, 0x0017,
	0x0040, 0x1238, 0x0003, 0x1a38, 0x0003, 0x1249, 0x0001, 0x1343, 0x0006,
	0x1262, 0x007f, 0x1263, 0x007f, 0x1264, 0x007f, 0x1248, 0x8100, 0x1247,
	0x8010, 0x1268, 0x0002, 0x1282, 0x0005, 0x1288, 0x3000, 0x124c, 0x8001,
	0x1298, 0x0003, 0x1245, 0x8c0f, 0x10aa, 0xff5d, 0x1265, 0x001e, 0x1027,
	0x00dd, 0x12b1, 0x0000, 0x1a98, 0x0003, 0x18ae, 0x0441, 0x18a1, 0x0004,
	0x18a7, 0xdfff, 0x18a6, 0x000d, 0x1a45, 0x8c0f, 0x18aa, 0xff5d, 0x1a34,
	0x0002, 0x1a5e, 0x0000, 0x1a5f, 0x0700, 0x1a49, 0x0001, 0x1b43, 0x0006,
	0x1a65, 0x001e, 0x1a62, 0x007f, 0x1a63, 0x007f, 0x1a64, 0x007f, 0x1a48,
	0x8100, 0x1a47, 0x8010, 0x1a68, 0x0002, 0x1a82, 0x0014, 0x1a4c, 0x8001,
	0x1827, 0x00dd, 0x1a35, 0x01f7, 0x1ab1, 0x0000, 0x0017, 0x0000, 0x000a,
	0x0946, 0x0009, 0x2016, 0x0008, 0x0314, 0x0005, 0x0003, 0x0007, 0x0000,
	0x0011, 0x0000, 0x0014, 0x12cd, 0x0017, 0x0000,
};

unsigned int cs4224_kr_ucode_image_length = sizeof(cs4224_kr_ucode_image)
	/ sizeof(cs4224_kr_ucode_image[0]);

#endif /* !CS_SKIP_UCODE_DOWNLOAD */

/* extern unsigned short g_cs4224_simplex_valid_mate_slice[]; */
/* unsigned char cs4224_diags_register_can_read(unsigned short addr); */

static enum e_cs4224_hardware_id cs4224_sku_hw_id = CS4224_HW_UNDEF;

int g_cs4224_mseq_stall_delay_us = 100;

void cs4224_reset_static_state_for_slice(unsigned int slice)
{
	CS_TRACE(("resetting slice %d\n", slice));
}

void cs4224_reset_static_state(void)
{
	cs4224_reset_die_static_state(0);
}

void cs4224_reset_die_static_state(unsigned int die)
{
	unsigned int upper_bits = die & 0xFFFFFF00;
	unsigned int slice;

	for (slice = upper_bits; (slice & 0xFF) < CS4224_MAX_NUM_SLICES(die);
		slice++) {
		cs4224_reset_static_state_for_slice(slice);
	}

	/* this forces the cached hw_id to be re-initialized */
	cs4224_sku_hw_id = CS4224_HW_UNDEF;
}

int cs4224_init_api_version(unsigned int slice)
{
	int status = CS_OK;
	unsigned short api_version;

	/* Program the API version number */
	api_version = (CS_API_VERSION_MAJOR & 0xF);
	api_version <<= 4;
	api_version |= (CS_API_VERSION_MINOR & 0xF);
	api_version <<= 8;
	api_version |= CS_API_VERSION_UPDATE;
	status |= cs4224_reg_set_channel(slice, CS4224_GLOBAL_SCRATCH7,
						api_version);

	return status;
}

unsigned int cs4224_max_num_ports;
unsigned int cs4224_max_num_dies;

unsigned int CS4224_MAX_NUM_PORTS(void)
{
	return CS4224_MAX_NUM_SLICES(0);
}

unsigned int CS4224_MAX_NUM_SLICES(unsigned int slice)
{
	/* Determine the h/w ID in order to
	 * initialize the number of ports
	 */
	cs4224_hw_id(slice);

	/* Now that the number of ports has been identified
	 * just return the initialized value
	 */
	return cs4224_max_num_ports;
}

unsigned int CS4224_MAX_NUM_DIES(unsigned int slice)
{
	cs4224_hw_id(slice);
	return cs4224_max_num_dies;
}

cs4224_callback_lock g_cs4224_callback_lock = NULL;
cs4224_callback_unlock g_cs4224_callback_unlock;

void cs4224_set_callback_for_lock(cs4224_callback_lock callback)
{
	g_cs4224_callback_lock = callback;
}

void cs4224_set_callback_for_unlock(cs4224_callback_unlock callback)
{
	g_cs4224_callback_unlock = callback;
}

int cs4224_lock(unsigned int slice)
{
	if (!g_cs4224_callback_lock)
		return CS_OK;

	return g_cs4224_callback_lock(slice);
}

int cs4224_unlock(unsigned int slice)
{
	if (!g_cs4224_callback_unlock)
		return CS_OK;

	return g_cs4224_callback_unlock(slice);
}

enum e_cs4224_hardware_id cs4224_hw_id(unsigned int slice)
{
	if (cs4224_sku_hw_id == CS4224_HW_UNDEF) {
		unsigned short sku;
		int status = CS_OK;

		cs4224_lock(slice);

		status |= cs4224_reg_get(0 | (slice & 0xffffff00),
						CS4224_EFUSE_PDF_SKU, &sku);

		cs4224_unlock(slice);

		if (status != CS_OK) {
			CS_TRACE(("WARNING: Could not read SKU\n"));
			sku = 0x0000;
		}

		/* clear SKU bits [15:5] and [3] */
		sku &= 0x0017;

		/* If the EFUSE is not programmed then default to CS4343.
		 * Bit 5 will be set to 1 if the EFUSE is programmed. If it
		 * is not programmed it will 0.
		 */
		if ((sku & 0x10) != 0x10) {
			unsigned short chip_id;

			/* Try reading the ASIC ID from the second die. If it
			 * returns a non valid value then assume this is a 4D
			 * part. If it is a 4D part then the ASIC ID should
			 * return 0xffff. A value of 0 probably means that we're
			 * running CS Explorer in offline mode.
			 */
			status |= cs4224_reg_get(
				1 | (slice & 0xffffff00),
				CS4224_GLOBAL_CHIP_ID_MSB, &chip_id);

			if (chip_id == 0 || chip_id == 0x3e5) {
				CS_TRACE(("WARNING: EFUSE not programmed\n"));
				CS_TRACE(("WARNING: Defaulting to CS4243\n"));

				cs4224_sku_hw_id = CS4224_HW_CS4343;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4343_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4343_DIES;
			} else {
				CS_TRACE(("WARNING: EFUSE not programmed\n"));
				CS_TRACE(("WARNING: Defaulting to CS4223\n"));

				cs4224_sku_hw_id = CS4224_HW_CS4223;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4223_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4223_DIES;
			}
		}
		/* Otherwise check what SKU is programmed */
		else {
			switch (sku) {
			case CS4224_HW_CS4223:
				/* 4 port duplex */
				cs4224_sku_hw_id = CS4224_HW_CS4223;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4223_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4223_DIES;
				break;
			case CS4224_HW_CS4224:
				/* 16 port simplex */
				cs4224_sku_hw_id = CS4224_HW_CS4224;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4224_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4224_DIES;
				break;
			case CS4224_HW_CS4221:
				/* 10 port simplex */
				cs4224_sku_hw_id = CS4224_HW_CS4221;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4221_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4221_DIES;
				break;
			case CS4224_HW_CS4227:
				/* 2 port duplex */
				cs4224_sku_hw_id = CS4224_HW_CS4227;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4227_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4227_DIES;
				break;
			case CS4224_HW_CS4210:
				/* 16 port simplex */
				cs4224_sku_hw_id = CS4224_HW_CS4210;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4210_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4210_DIES;
				break;
			case CS4224_HW_CS4341:
				/* 8 port duplex */
				cs4224_sku_hw_id = CS4224_HW_CS4341;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4341_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4341_DIES;
				break;
			default:
				/* SKU must be cs4343 or if undefined
				 * we default to 8 port duplex
				 */
				cs4224_sku_hw_id = CS4224_HW_CS4343;
				cs4224_max_num_ports =
					CS4224_MAX_NUM_CS4343_PORTS;
				cs4224_max_num_dies =
					CS4224_MAX_NUM_CS4343_DIES;
				break;
			}
		}
	}

	return cs4224_sku_hw_id;
}

unsigned char cs4224_is_hw_simplex(unsigned int slice)
{
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	if ((hw_id == CS4224_HW_CS4224) ||
	    (hw_id == CS4224_HW_CS4221) ||
	    (hw_id == CS4224_HW_CS4210)) {
		return CS_TRUE;
	}
	return CS_FALSE;
}

unsigned char cs4224_is_hw_duplex(unsigned int slice)
{
	/* only keep the 'list' of simplex/duplex slices in one place */
	if (cs4224_is_hw_simplex(slice))
		return CS_FALSE;

	return CS_TRUE;
}

unsigned short cs4224_pp_sku_cs4343_offset[8] = {
	/* slice 0, 1, 2, 3, 4, 5, 6, 7 */
	0x0000,	0x1000, 0x2000, 0x3000, 0x3000,	0x2000, 0x1000, 0x0000
};

unsigned short cs4224_mseq_sku_cs4343_offset[8] = {
	/* slice 0, 1, 2, 3, 4, 5, 6, 7 */
	0x0000,	0x0000, 0x100, 0x100, 0x0100, 0x0100, 0x0000, 0x0000
};

/* CS4224_HW_CS4224  16 port simplex
 * CS4224_HW_CS4210  16 port simplex, 15G only
 */
unsigned short cs4224_pp_sku_cs4224_offset[16] = {
	/* slice 0, 1, 2, 3, 4, 5, 6, 7 */
	0x0000,	0x1000, 0x3000, 0x1000, 0x2000,	0x0000, 0x3000, 0x2000,
	/* slice 8, 9, 10, 11, 12, 13, 14, 15 */
	0x0000,	0x2000, 0x1000, 0x3000, 0x2000,	0x0000, 0x1000, 0x3000
};

unsigned short cs4224_mseq_sku_cs4224_offset[16] = {
	/* slice 0, 1, 2, 3, 4, 5, 6, 7 */
	0x0000,	0x0000, 0x0100, 0x0000, 0x0100,	0x0000, 0x0100, 0x0100,
	/* slice 8, 9, 10, 11, 12, 13, 14, 15 */
	0x0000,	0x0100, 0x0000, 0x0100, 0x0100,	0x0000, 0x0000, 0x0100
};

/* CS4224_HW_CS4221  10 port simplex
 * Relative to CS4224, CS4221 is mapped:
 *    CS4221   CS4224
 *     Port     Port
 *       0        1
 *       1        0
 *       2        3
 *       3        5
 *       4        6
 *       5       10
 *       6        9
 *       7       11
 *       8       13
 *       9       14
 */
unsigned short cs4224_pp_sku_cs4221_offset[10] = {
	/* slice 0, 1, 2, 3, 4, 5, 6, 7 */
	0x1000, 0x0000, 0x1000, 0x0000, 0x3000, 0x1000, 0x2000, 0x3000,
	/* slice 8, 9, */
	0x0000,	0x1000
};

unsigned short cs4224_mseq_sku_cs4221_offset[10] = {
	/* slice     0,    1,    2,    3,    4,    5,    6,   7 */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0100, 0x0000, 0x0100, 0x0100,
	/* slice     8,    9,  */
	0x0000,	0x0000
};

unsigned int cs4224_adj_pp(unsigned int slice, unsigned short addr)
{
	unsigned short adj_addr = addr;
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	switch (hw_id) {
	case CS4224_HW_CS4223: /*  4 port duplex   */

		if ((slice & 0xFF) > 3) {
			CS_TRACE(("ERROR: Duplex slice arg > 3, slice=0x%x\n",
				  slice));
		}
		/* same mapping as cs4343 */
		adj_addr = (addr + cs4224_pp_sku_cs4343_offset[slice & 0x7]);
		break;

	case CS4224_HW_CS4224: /* 16 port simplex  */

		if ((slice & 0xFF) > 15) {
			CS_TRACE(("ERROR: Simplex slice arg > 15, slice=0x%x\n",
				  slice));
		}
		adj_addr = (addr + cs4224_pp_sku_cs4224_offset[slice & 0xF]);

		break;

	case CS4224_HW_CS4343: /*  8 port duplex   */

		if ((slice & 0xFF) > 7) {
			CS_TRACE(("ERROR: Duplex slice arg > 7, slice=0x%x\n",
				  slice));
		}
		adj_addr = (addr + cs4224_pp_sku_cs4343_offset[slice & 0x7]);

		break;

	case CS4224_HW_CS4221: /* 10 port simplex */

		if ((slice & 0xFF) > 9) {
			CS_TRACE(("ERROR: Simplex slice arg > 9, slice=0x%x\n",
				  slice));
			adj_addr = (addr + cs4224_pp_sku_cs4221_offset[0]);
		} else {
			adj_addr = (addr
				+ cs4224_pp_sku_cs4221_offset[slice & 0xF]);
		}

		break;

	case CS4224_HW_CS4227: /*  2 port duplex   */

		if ((slice & 0xFF) > 1) {
			CS_TRACE(("ERROR: Duplex slice arg > 1, slice=0x%x\n",
				  slice));
		}
		/* same mapping as cs4343 */
		adj_addr = (addr + cs4224_pp_sku_cs4343_offset[slice & 0x7]);
		break;

	case CS4224_HW_CS4210: /* 16 port simplex  */

		if ((slice & 0xFF) > 15) {
			CS_TRACE(("ERROR: Simplex slice arg > 15, slice=0x%x\n",
				  slice));
		}
		adj_addr = (addr + cs4224_pp_sku_cs4224_offset[slice & 0xF]);

		break;

	case CS4224_HW_CS4341: /*  8 port duplex   */
		if ((slice & 0xFF) > 7) {
			CS_TRACE(("ERROR: Duplex slice arg > 7, slice=0x%x\n",
				  slice));
		}
		adj_addr = (addr + cs4224_pp_sku_cs4343_offset[slice & 0x7]);

		break;

	default:
		CS_TRACE(("ERROR: invalid hw_id: %x\n", hw_id));
		break;
	}

	return adj_addr;
}

unsigned int cs4224_adj_mseq(unsigned int slice, unsigned short addr)
{
	unsigned short adj_addr = addr;
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	switch (hw_id) {
	case CS4224_HW_CS4223: /*  4 port duplex   */
		if ((slice & 0xFF) > 3) {
			CS_TRACE(("ERROR: Duplex slice arg > 3, slice=%d\n",
				  slice));
		}
		/* same mapping as cs4343 */
		adj_addr = (addr + cs4224_mseq_sku_cs4343_offset[slice & 0x7]);
		break;

	case CS4224_HW_CS4224: /* 16 port simplex  */
		if ((slice & 0xFF) > 15) {
			CS_TRACE(("ERROR: Simplex slice arg > 15, slice=%d\n",
				  slice));
		}
		adj_addr = (addr + cs4224_mseq_sku_cs4224_offset[slice & 0xF]);

		break;

	case CS4224_HW_CS4343: /*  8 port duplex   */
		if ((slice & 0xFF) > 7) {
			CS_TRACE(("ERROR: Duplex slice arg > 7, slice=%d\n",
				  slice));
		}
		adj_addr = (addr + cs4224_mseq_sku_cs4343_offset[slice & 0x7]);

		break;

	case CS4224_HW_CS4221: /* 10 port simplex */
		if ((slice & 0xFF) > 9) {
			CS_TRACE(("ERROR: Simplex slice arg > 9, slice=%d\n",
				  slice));
			adj_addr = (addr + cs4224_mseq_sku_cs4221_offset[0]);
		} else {
			adj_addr = (addr
				+ cs4224_mseq_sku_cs4221_offset[slice & 0xF]);
		}

		break;

	case CS4224_HW_CS4227: /*  2 port duplex   */
		if ((slice & 0xFF) > 1) {
			CS_TRACE(("ERROR: Duplex slice arg > 1, slice=%d\n",
				  slice));
		}
		/* same mapping as cs4343 */
		adj_addr = (addr + cs4224_mseq_sku_cs4343_offset[slice & 0x7]);
		break;

	case CS4224_HW_CS4210: /* 16 port simplex  */
		if ((slice & 0xFF) > 15) {
			CS_TRACE(("ERROR: Simplex slice arg > 15, slice=%d\n",
				  slice));
		}
		adj_addr = (addr + cs4224_mseq_sku_cs4224_offset[slice & 0xF]);

		break;

	case CS4224_HW_CS4341: /*  8 port duplex   */
		if ((slice & 0xFF) > 7) {
			CS_TRACE(("ERROR: Duplex slice arg > 7, slice=%d\n",
				  slice));
		}
		adj_addr = (addr + cs4224_mseq_sku_cs4343_offset[slice & 0x7]);

		break;

	default:
		CS_TRACE(("ERROR: invalid hw_id: %x\n", hw_id));
		break;
	}

	return adj_addr;
}

/* simplex mate slices (slices with the same PP number on the die) */
unsigned short cs4224_mate_slice_sku_cs4224[16] = {
	/* slice    0, 1, 2, 3, 4, 5, 6, 7 */
	5, 3, 6, 1, 7, 0, 2, 4,
	/* slice    8,  9, 10, 11, 12, 13, 14, 15 */
	13, 12, 14, 15, 9, 8, 10, 11 };

/* CS4221 has some 'empty' slices that don't have a mate slice, those must be
 * handled manually, as there is no way to address them with reg_Xet_channel
 */
unsigned short cs4224_mate_slice_sku_cs4221[10] = {
	/* slice    0, 1, 2, 3, 4 */
	2, 3, 0, 1, 0xff,
	/* slice    5, 6, 7, 8, 9 */
	9, 0xff, 0xff, 0xff, 5
};

int cs4224_simplex_mate_slice(unsigned int slice, unsigned int *mate)
{
	int status = CS_OK;
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);
	unsigned char base_slice = (slice & 0xFF);
	unsigned int upper_bits = (slice & 0xFFFFFF00);

	*mate = 0xFF;

	switch (hw_id) {
	case CS4224_HW_CS4224:
	case CS4224_HW_CS4210: /*16 port simplex*/
		if ((slice & 0xFF) > 15) {
			CS_TRACE(("ERROR: slice parameter > 15\n"));
			return CS_ERROR;
		}
		*mate = cs4224_mate_slice_sku_cs4224[base_slice];

		break;
	case CS4224_HW_CS4221: /*10 port simplex*/
		if ((slice & 0xFF) > 10) {
			CS_TRACE(("ERROR: slice parameter > 10\n"));
			return CS_ERROR;
		}
		*mate = cs4224_mate_slice_sku_cs4221[base_slice];

		break;
	default:
		/* if duplex then just return the same slice,
		 * since that's technically correct
		 */
		*mate = base_slice;

		break;
	}

	if (*mate == 0xFF) {
		CS_TRACE(("ERROR: base_slice %d simplex mate missing\n",
			  base_slice));
		status |= CS_ERROR;
	}

	*mate |= upper_bits;

	return status;
}

unsigned char cs4224_line_rx_to_host_tx_dir(unsigned int slice)
{
	unsigned char lrx_to_htx = CS_TRUE;
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	/* Mask off the upper bits in case they are set
	 * by the user. Otherwise the switch statement
	 * below won't work properly.
	 */
	slice = slice & 0xFF;

	if ((hw_id == CS4224_HW_CS4224) || (hw_id == CS4224_HW_CS4210)) {
		/* 16 port simplex */
		switch (slice) {
		case 0:
		case 1:
		case 2:
			lrx_to_htx = CS_TRUE;
			break;

		case 3:
			lrx_to_htx = CS_FALSE;
			break;

		case 4:
			lrx_to_htx = CS_TRUE;
			break;

		case 5:
		case 6:
		case 7:
			lrx_to_htx = CS_FALSE;
			break;

		case 8:
		case 9:
		case 10:
		case 11:
			lrx_to_htx = CS_TRUE;
			break;

		case 12:
		case 13:
		case 14:
		case 15:
			lrx_to_htx = CS_FALSE;
			break;

		default:
			lrx_to_htx = CS_TRUE;
			break;
		}
	} else if (hw_id == CS4224_HW_CS4221) {
		/* 10 port simplex */
		switch (slice) {
		case 0:
		case 1:
			lrx_to_htx = CS_TRUE;
			break;

		case 2:
		case 3:
		case 4:
			lrx_to_htx = CS_FALSE;
			break;

		case 5:
		case 6:
		case 7:
			lrx_to_htx = CS_TRUE;
			break;

		case 8:
		case 9:
			lrx_to_htx = CS_FALSE;
			break;

		default:
			lrx_to_htx = CS_TRUE;
			break;
		}
	}

	return lrx_to_htx;
}

int cs4224_get_mseq_id(unsigned int slice, enum e_cs4224_mseq_id *mseq_id)
{
	int status = CS_OK;

	/* if mseq_id isn't simplex then just assume the user
	 * knows what they are doing
	 */
	if (cs4224_is_hw_simplex(slice) && *mseq_id == CS4224_SPLX_MSEQ) {
		if (cs4224_line_rx_to_host_tx_dir(slice))
			*mseq_id = CS4224_DPLX_LINE_MSEQ;
		else
			*mseq_id = CS4224_DPLX_HOST_MSEQ;
	} else {
		if (*mseq_id == CS4224_SPLX_MSEQ) {
			CS_TRACE(("ERROR: Duplex slice, invalid mseq_id\n"));
			status |= CS_ERROR;
		}
	}

	return status;
}

unsigned int cs4224_mseq_get_addr_offset(unsigned int slice,
					 enum e_cs4224_mseq_id mseq)
{
	unsigned int offset = 0; /* 0 is line side, 0x800 is host side */

	if (cs4224_is_hw_simplex(slice) && mseq == CS4224_SPLX_MSEQ) {
		if (!cs4224_line_rx_to_host_tx_dir(slice))
			offset = 0x800;
	} else if (mseq == CS4224_DPLX_HOST_MSEQ) {
		offset = 0x800;
	}

	return offset;
}

int cs4224_get_cfg_side(unsigned int slice, enum e_cs4224_cfg_sides_t *side)
{
	int status = CS_OK;

	/* a bit different than get_mseq_id, if you specify a non-simplex side
	 * then default to using that, otherwise use the Rx side of the simplex
	 * slice
	 */
	if (cs4224_is_hw_simplex(slice) && (*side == CS4224_CFG_SIMPLEX)) {
		if (cs4224_line_rx_to_host_tx_dir(slice))
			*side = CS4224_CFG_LINE_SIDE;
		else
			*side = CS4224_CFG_HOST_SIDE;
	} else {
		if (*side == CS4224_CFG_SIMPLEX) {
			CS_TRACE(("ERROR: Duplex slice, not simplex\n"));
			status |= CS_ERROR;
		}
	}

	return status;
}

int cs4224_mux_enable(unsigned int slice,
		      enum e_cs4224_cfg_sides_t side,
		      unsigned char enable)
{
	int status = CS_OK;
	unsigned short stx0_misc_addr;
	unsigned short data;

	if (side == CS4224_CFG_SIMPLEX) {
		CS_TRACE(("ERROR: CS4224_CFG_SIMPLEX arg not supported\n"));
		return CS_ERROR;
	}

	if (side == CS4224_CFG_LINE_SIDE)
		stx0_misc_addr = CS4224_PP_LINE_SDS_COMMON_STX0_MISC;
	else
		stx0_misc_addr = CS4224_PP_HOST_SDS_COMMON_STX0_MISC;

	cs4224_lock(slice);

	status |= cs4224_reg_get_channel(slice, stx0_misc_addr, &data);
	if (enable)
		data &= ~0x11; /* power-up mux, STX_EYEMODE_EN=0 */
	else
		data |= 0x11; /* power-down mux, STX_EYEMODE_EN=1 */

	status |= cs4224_reg_set_channel(slice, stx0_misc_addr, data);

	cs4224_unlock(slice);

	return status;
}

int cs4224_demux_enable(unsigned int slice, enum e_cs4224_cfg_sides_t side,
			unsigned char enable)
{
	int status = CS_OK;
	unsigned short mseq_power_down_addr;
	unsigned short agc_config1_addr;
	unsigned short data;

	if (side == CS4224_CFG_SIMPLEX) {
		CS_TRACE(("ERROR: CS4224_CFG_SIMPLEX arg not supported\n"));
		return CS_ERROR;
	}

	if (side == CS4224_CFG_LINE_SIDE) {
		mseq_power_down_addr =
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		agc_config1_addr = CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1;
	} else {
		mseq_power_down_addr =
			CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		agc_config1_addr = CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG1;
	}

	cs4224_lock(slice);

	cs4224_reg_get_channel(slice, agc_config1_addr, &data);

	/* Don't touch the power down registers if we're in CX1 mode */
	if (data != 0x5) {
		if (enable) {
			status |= cs4224_reg_get_channel(
				slice, mseq_power_down_addr, &data);
			data |= 0x001f; /* partial power-up */
			status |= cs4224_reg_set_channel(
				slice, mseq_power_down_addr, data);
			data &= ~0x001f; /* clear the partial power-up */
			data &= ~0x01c0; /* full power-up demux */
			status |= cs4224_reg_set_channel(
				slice, mseq_power_down_addr, data);
		} else {
			status |= cs4224_reg_get_channel(
				slice, mseq_power_down_addr, &data);
			data |= 0x01c0; /* power-down demux */
			status |= cs4224_reg_set_channel(
				slice, mseq_power_down_addr, data);
		}
	}

	cs4224_unlock(slice);

	return status;
}

#define CS4224_SENSE_POINT_LOCAL_TEMPERATURE  1 << 0
#define CS4224_SENSE_POINT_REMOTE_TEMPERATURE 1 << 1
#define CS4224_SENSE_POINT_0p9_RX_VOLTAGE     1 << 2
#define CS4224_SENSE_POINT_0p9_TX_VOLTAGE     1 << 3
#define CS4224_SENSE_POINT_0p9_DIG_RX_VOLTAGE 1 << 4
#define CS4224_SENSE_POINT_0p9_DIG_TX_VOLTAGE 1 << 5
#define CS4224_SENSE_POINT_1p8_RX_VOLTAGE     1 << 6
#define CS4224_SENSE_POINT_1p8_TX_VOLTAGE     1 << 7
#define CS4224_SENSE_POINT_1p5_RX_VOLTAGE     1 << 8
#define CS4224_SENSE_POINT_1p5_TX_VOLTAGE     1 << 9

int cs4224_enable_monitor_sense_points(unsigned int die,
				       unsigned char wait_till_stabilized)
{
	unsigned short reg_data;
	int status = CS_OK;

	status |= cs4224_reg_get(die, CS4224_MONITOR_CONTROL0, &reg_data);

	if (reg_data != 0x630F) {
		status |= cs4224_reg_set(die, CS4224_MONITOR_CONFIG_MASK,
		CS4224_SENSE_POINT_LOCAL_TEMPERATURE |
		CS4224_SENSE_POINT_0p9_RX_VOLTAGE |
		CS4224_SENSE_POINT_1p8_RX_VOLTAGE);

		status |= cs4224_reg_set(die, CS4224_MONITOR_CONTROL0, 0x630F);

		if (wait_till_stabilized) {
			unsigned short reg_data;
			int last = 0;
			int i;

			/* Waiting for the monitors to stabilize */
			cs4224_reg_get(die, CS4224_MONITOR_STATUS_FINAL6,
				       &reg_data);

			for (i = 0; i < 100; i++) {
				CS_MDELAY(10);

				if (last > (int)reg_data)
					break;

				last = (int)reg_data;
				cs4224_reg_get(die,
					       CS4224_MONITOR_STATUS_FINAL6,
					       &reg_data);
			}
		}
	}

	return status;
}

int cs4224_version(char *buffer, unsigned int buffer_len)
{
	CS_STRNCPY(buffer,
		   "CS4224 API Version " CS_API_VERSION "\n"
		   "Built on " CS_API_BUILD_DATE "\n",
		   buffer_len);

	return CS_OK;
}

int cs4224_init_vco(unsigned int slice, unsigned int addr)
{
	int status = CS_OK;
	unsigned short offset = 0;
	unsigned short data = 0;
	unsigned short i = 0;

	if (addr & CS4224_LINE_TO_HOST_OFFSET)
		offset = CS4224_LINE_TO_HOST_OFFSET;
	else
		offset = 0;

	cs4224_lock(slice);

	/* re-initialize the VCO ALT coarse tuning algorithm
	 * to start searching for the correct frequency
	 */
	status |= cs4224_reg_set_channel(slice, addr, 0x2001);
	status |= cs4224_reg_set_channel(slice, addr, 0xa001);
	status |= cs4224_reg_set_channel(slice, addr, 0x2001);

	/* the worst-case time out value is where F(ref_clk) > F(vco_clk) so a
	 * timeout of 3.3ms should suffice for the completion of coarse tuning.
	 */
	for (i = 1500; i > 0; i--) {
		/* check for CTVALID */
		status |= cs4224_reg_get_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_RXVCO0_ALTCT_STATUS + offset,
			&data);
		if ((data & CS_BIT15) || (status != CS_OK)) {
			/* tuning done or error, exit */
			break;
		}
		CS_UDELAY(10);
	}
	cs4224_unlock(slice);

	if (i == 0) {
		if (offset == 0)
			CS_TRACE(("ERROR: Timeout slice %d Line VCO\n", slice));
		else
			CS_TRACE(("ERROR: Timeout slice %d Host VCO\n", slice));

		CS_TRACE(("ERROR: check CS_UDELAY implementation\n"));
		status |= CS_ERROR;
	}

	return status;
}

int cs4224_gbl_clock_monitor(unsigned int slice,
			     struct cs4224_gbl_clk_mon_cfg_t *clk_mon_cfg)
{
	int status = CS_OK;
	unsigned short data;

	CS_PRINTF(("Configuring the GBL clkmon\n"));

	data = clk_mon_cfg->clksel_divider | clk_mon_cfg->clksel_src;
	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_CLKSEL, data);

	data = 0;
	if (clk_mon_cfg->go)
		data = 0x0002;

	if (clk_mon_cfg->free_run)
		data |= 0x0010;

	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_CTRL, data);

	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_DURATION,
						clk_mon_cfg->duration);

	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_MINTHRESH0,
						clk_mon_cfg->min_thresh_0);
	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_MINTHRESH1,
						clk_mon_cfg->min_thresh_1);
	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_MAXTHRESH0,
						clk_mon_cfg->max_thresh_0);
	status |= cs4224_reg_set_channel(slice, CS4224_CLKMON_GBL_MAXTHRESH1,
						clk_mon_cfg->max_thresh_1);

	return status;
}

void cs4224_pp_clk_mon_cfg_init(struct cs4224_pp_clk_mon_cfg_t *clk_mon_cfg)
{
	clk_mon_cfg->clksel_divider = CS4224_CLK_MON_DIV1;
	clk_mon_cfg->clksel_src = CS4224_CLK_MON_PP_SRX;
	clk_mon_cfg->go = CS_TRUE;
	clk_mon_cfg->free_run = CS_TRUE;
	clk_mon_cfg->duration = 0x4000;
	clk_mon_cfg->min_thresh = 0;
	clk_mon_cfg->max_thresh = 0;
}

int cs4224_pp_clock_monitor(unsigned int slice,
			    enum e_cs4224_cfg_sides_t side,
			    struct cs4224_pp_clk_mon_cfg_t *clk_mon_cfg)
{
	int status = CS_OK;
	unsigned short clksel_addr;
	unsigned short ctrl_addr;
	unsigned short duration_addr;
	unsigned short min_thresh0_addr;
	unsigned short min_thresh1_addr;
	unsigned short max_thresh0_addr;
	unsigned short max_thresh1_addr;
	unsigned short data;

	status |= cs4224_get_cfg_side(slice, &side);

	if (side == CS4224_CFG_HOST_SIDE) {
		clksel_addr = CS4224_PP_CLKMON_EGR_CLKSEL;
		ctrl_addr = CS4224_PP_CLKMON_EGR_CTRL;
		duration_addr = CS4224_PP_CLKMON_EGR_DURATION;
		min_thresh0_addr = CS4224_PP_CLKMON_EGR_MINTHRESH0;
		min_thresh1_addr = CS4224_PP_CLKMON_EGR_MINTHRESH1;
		max_thresh0_addr = CS4224_PP_CLKMON_EGR_MAXTHRESH0;
		max_thresh1_addr = CS4224_PP_CLKMON_EGR_MAXTHRESH1;
	} else {
		clksel_addr = CS4224_PP_CLKMON_ING_CLKSEL;
		ctrl_addr = CS4224_PP_CLKMON_ING_CTRL;
		duration_addr = CS4224_PP_CLKMON_ING_DURATION;
		min_thresh0_addr = CS4224_PP_CLKMON_ING_MINTHRESH0;
		min_thresh1_addr = CS4224_PP_CLKMON_ING_MINTHRESH1;
		max_thresh0_addr = CS4224_PP_CLKMON_ING_MAXTHRESH0;
		max_thresh1_addr = CS4224_PP_CLKMON_ING_MAXTHRESH1;
	}

	data = clk_mon_cfg->clksel_divider | clk_mon_cfg->clksel_src;
	status |= cs4224_reg_set_channel(slice, clksel_addr, data);

	status |= cs4224_reg_set_channel(
			slice,
			duration_addr,
			clk_mon_cfg->duration);

	status |= cs4224_reg_set_channel(
			slice,
			min_thresh0_addr,
			(unsigned short)clk_mon_cfg->min_thresh);

	status |= cs4224_reg_set_channel(
			slice,
			min_thresh1_addr,
			(clk_mon_cfg->min_thresh >> 16));

	status |= cs4224_reg_set_channel(
			slice,
			max_thresh0_addr,
			(unsigned short)clk_mon_cfg->max_thresh);

	status |= cs4224_reg_set_channel(
			slice,
			max_thresh1_addr,
			(clk_mon_cfg->max_thresh >> 16));

	data = 0x0100; /* config */
	if (clk_mon_cfg->go)
		data |= CS_BIT1; /* go */

	if (clk_mon_cfg->free_run)
		data |= CS_BIT4; /* free-run */

	status |= cs4224_reg_set_channel(slice, ctrl_addr, data);
	data &= ~CS_BIT1; /* clear go */
	status |= cs4224_reg_set_channel(slice, ctrl_addr, data);

	return status;
}

int cs4224_pp_clock_monitor_freq_fixp(
	unsigned int slice, enum e_cs4224_cfg_sides_t side,
	struct cs4224_pp_clk_mon_cfg_t *pp_clk_mon_cfg,
	unsigned int ref_clk_rate,
	unsigned short *freq)
{
	int status = CS_OK;
	signed char i = 0;
	unsigned short data = 0;
	unsigned int count = 0;
	unsigned int ticks_per_sec = 0;
	unsigned short des_freq = *freq;
	unsigned char stalled = CS_FALSE;
	unsigned short power_down = 0;

	unsigned short status_addr;
	unsigned short count0_addr;
	unsigned short count1_addr;
	unsigned short power_down_addr;
	enum e_cs4224_mseq_id mseq_id;

	status |= cs4224_get_cfg_side(slice, &side);

	if (side == CS4224_CFG_HOST_SIDE) {
		status_addr = CS4224_PP_CLKMON_EGR_STATUS;
		count0_addr = CS4224_PP_CLKMON_EGR_COUNT0;
		count1_addr = CS4224_PP_CLKMON_EGR_COUNT1;
		power_down_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		mseq_id = CS4224_DPLX_HOST_MSEQ;
	} else {
		status_addr = CS4224_PP_CLKMON_ING_STATUS;
		count0_addr = CS4224_PP_CLKMON_ING_COUNT0;
		count1_addr = CS4224_PP_CLKMON_ING_COUNT1;
		power_down_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		mseq_id = CS4224_DPLX_LINE_MSEQ;
	}

	status |= cs4224_query_mseq_is_stalled(slice, mseq_id, &stalled);
	if (!stalled)
		cs4224_mseq_stall(slice, mseq_id, CS_TRUE);

	status |= cs4224_reg_get_channel(slice, power_down_addr, &power_down);
	if (power_down != 0x0000) {
		status |= cs4224_reg_set_channel(slice, power_down_addr, 0x1f);
		status |= cs4224_reg_set_channel(slice, power_down_addr, 0x00);
	}

	/* setup clock mon */
	pp_clk_mon_cfg->clksel_divider = CS4224_CLK_MON_DIV1;
	pp_clk_mon_cfg->go = CS_TRUE;
	pp_clk_mon_cfg->free_run = CS_TRUE;
	pp_clk_mon_cfg->duration = 0x4000;

	status |= cs4224_pp_clock_monitor(slice, side, pp_clk_mon_cfg);

	for (i = 10; i > 0; i--) {
		status |= cs4224_reg_get_channel(slice, status_addr, &data);
		if (data & 0x1)
			break;
		CS_UDELAY(200); /* 1/156.25MHz * 0x4000 = 154us */
	}
	if (i <= 0) {
		CS_TRACE(("ERROR: clock monitor count never completed\n"));
		status |= CS_ERROR;
		return status;
	}

	status |= cs4224_reg_get_channel(slice, count1_addr, &data);
	count = (unsigned int)data << 16;
	status |= cs4224_reg_get_channel(slice, count0_addr, &data);
	count |= (unsigned int)data;

	/* freq = count * ref * div / duration */
	/* control the truncating; don't loose precision unintentionally */
	ticks_per_sec = ((long long int)count * ref_clk_rate) / 1000;

	*freq = (ticks_per_sec * 1) / 0x4000;

	if (des_freq != 0) {
		unsigned short temp = des_freq * 9;

		if ((*freq * 10) < temp)
			CS_TRACE(("WARNING: freq < des_freq -10%%: %u < %u\n",
				  *freq, temp / 9));
		temp = des_freq * 11;
		if ((*freq * 10) > temp)
			CS_TRACE(("WARNING: freq > des_freq +10%%: %u > %u\n",
				  *freq, temp / 11));
	}

	if (power_down != 0x0000)
		status |= cs4224_reg_set_channel(slice, power_down_addr,
							power_down);

	if (!stalled)
		cs4224_mseq_stall(slice, mseq_id, CS_FALSE);

	return status;
}

int cs4224_init_ac_decoupling_caps(unsigned int slice,
				   struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned short reg_data;

	if ((cs4224_is_hw_simplex(slice) &&
	     cs4224_line_rx_to_host_tx_dir(slice)) ||
	     (cs4224_is_hw_duplex(slice))) {
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG4,
			&reg_data);

		if (rules->enable_ac_decoupling_caps)
			reg_data |= 0x0080;
		else
			reg_data &= ~0x0080;

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG4,
			reg_data);
	}
	if ((cs4224_is_hw_simplex(slice) &&
	     !cs4224_line_rx_to_host_tx_dir(slice)) ||
	     (cs4224_is_hw_duplex(slice))) {
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG4,
			&reg_data);

		if (rules->enable_ac_decoupling_caps)
			reg_data |= 0x0080;
		else
			reg_data &= ~0x0080;

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG4,
			reg_data);
	}

	return status;
}

int cs4224_rules_set_default(enum e_cs4224_target_application application,
			     struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	rules->application = application;

	rules->rx_if.splx_eq.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->rx_if.splx_edc_mode = CS_HSIO_EDC_MODE_SR;

	rules->tx_if.splx_driver.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->tx_if.splx_disable = CS_FALSE;

	rules->rx_if.dplx_line_eq.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->rx_if.dplx_host_eq.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->rx_if.dplx_line_edc_mode = CS_HSIO_EDC_MODE_SR;
	rules->rx_if.dplx_host_edc_mode = CS_HSIO_EDC_MODE_SR;

	rules->tx_if.dplx_line_driver.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->tx_if.dplx_host_driver.traceloss = CS_HSIO_TRACE_LOSS_2dB;
	rules->tx_if.dplx_line_disable = CS_FALSE;
	rules->tx_if.dplx_host_disable = CS_FALSE;

	rules->unsquelch_driver = CS_TRUE;
	rules->tx_auto_squelch = CS_FALSE;
	rules->enable_ac_decoupling_caps = CS_TRUE;
	rules->enable_power_savings = CS_TRUE;
	rules->enable_die_broadcast = CS_TRUE;

	/* By default leave debugging info disabled. Customers
	 * can turn this on if desired.
	 */
	rules->show_debug_info = CS_TRUE;

	/* By default turn off the ability of the microsequencer to
	 * perform a dynamic reconfig
	 */
	rules->mseq_dyn_reconfig = CS_FALSE;

	switch (application) {
	case CS4224_TARGET_APPLICATION_OC12:
	case CS4224_TARGET_APPLICATION_1G:
	case CS4224_TARGET_APPLICATION_5G:
	case CS4224_TARGET_APPLICATION_7p5G:
	case CS4224_TARGET_APPLICATION_8G:
	case CS4224_TARGET_APPLICATION_8p5G:
	case CS4224_TARGET_APPLICATION_15G:
	case CS4224_TARGET_APPLICATION_16G_FC:
	case CS4224_TARGET_APPLICATION_10G_FC:
	case CS4224_TARGET_APPLICATION_8G_FC:
	case CS4224_TARGET_APPLICATION_4G_FC:
	case CS4224_TARGET_APPLICATION_2G_FC:
	case CS4224_TARGET_APPLICATION_1G_FC:
	case CS4224_TARGET_APPLICATION_10G_KR:
	case CS4224_TARGET_APPLICATION_40G_KR:
	case CS4224_TARGET_APPLICATION_KRAN:
	case CS4224_TARGET_APPLICATION_FCAN:
		status = CS_ERROR;
		printf("Application not yet supported\n");
		break;
	case CS4224_TARGET_APPLICATION_10G:
		/* 156.25MHz ref clock, 10.3125GHz */
		rules->ref_clk_rate = CS4224_REF_CLK_156p25;

		rules->clkdiv.enable = CS_TRUE;
		rules->clkdiv.fastdiv = 0x3; /* div by 40 (default) */
		rules->clkdiv.ddiv = CS4224_DDIV_DIV1;
		rules->clkdiv.rdiv = CS4224_RDIV_DIV66;

		break;

	default: {
		CS_TRACE(("ERROR: Invalid application=%d\n", application));
		status = CS_ERROR;
		break;
	}
	}

	return status;
}

int cs4224_mseq_squelch_ctrl(unsigned int slice,
			     enum e_cs4224_datapath_dir_t dir,
				   unsigned char enable)
{
	int status = CS_OK;
	unsigned short offset = 0x0;
	unsigned short spare12;
	enum e_cs4224_mseq_id mate_mseq;
	unsigned char stalled = CS_FALSE;
	unsigned short data;

	status |= cs4224_get_cfg_side(slice, (enum e_cs4224_cfg_sides_t *)&dir);
	if (status != CS_OK)
		return status;

	if (dir == CS4224_LINE_RX_TO_HOST_TX_DIR) {
		offset = 0x0;
		mate_mseq = CS4224_DPLX_HOST_MSEQ;
	} else {
		offset = CS4224_LINE_TO_HOST_OFFSET;
		mate_mseq = CS4224_DPLX_LINE_MSEQ;
	}

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + offset,
		&spare12);

	if (enable) {
		if (cs4224_is_hw_simplex(slice)) {
			status |= cs4224_query_mseq_is_stalled(slice, mate_mseq,
								&stalled);
			if (stalled)
				status |= cs4224_mseq_stall(slice, mate_mseq,
								CS_FALSE);
		}
		if (CS_IF_CLR(spare12, CS_BIT2)) {
			/* enable MSEQ control of squelch */
			spare12 = CS_SET(spare12, CS_BIT2);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB +
				    offset,
				    spare12);
		}
	} else {
		if (spare12 & CS_BIT2) {
			/* disable MSEQ control of squelch */
			spare12 = CS_CLR(spare12, CS_BIT2);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB +
				    offset,
				    spare12);

			/* delay for mseq to catch up */
			CS_UDELAY(30);

			/* disable this interfaces mail-out */
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_LSB +
				    offset,
				    0);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_MSB +
				    offset,
				    0);
			/* disable the connected interface's mail in */
			status |= cs4224_reg_get_channel(
				    slice,
				    CS4224_PP_HOST_SDS_DSP_MSEQ_MAIL_IN_MSB -
				    offset,
				    &data);
			data = CS_CLR(data, CS_BIT15 | CS_BIT12);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_HOST_SDS_DSP_MSEQ_MAIL_IN_MSB -
				    offset,
				    data);
		}
	}

	return status;
}

int cs4224_send_squelch_request(unsigned int slice,
				enum e_cs4224_datapath_dir_t dir,
					unsigned char squelch)
{
	int status = CS_OK;
	unsigned short offset = 0;

	status |= cs4224_get_cfg_side(slice, (enum e_cs4224_cfg_sides_t *)&dir);
	if (status != CS_OK)
		return status;

	if (dir == CS4224_LINE_RX_TO_HOST_TX_DIR)
		offset = 0;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	if (squelch)
		/* enable squelch */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_LSB + offset,
			0x8000);
	else
		/* disable squelch */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_LSB + offset,
			0x0000);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_MSB + offset,
		0x9000);

	return status;
}

int cs4224_squelch_driver(unsigned int slice, enum e_cs4224_cfg_sides_t side,
			  unsigned char squelch)
{
	int status = CS_OK;
	unsigned short squelch_addr;
	unsigned short ctrla_addr;
	unsigned short ctrla_storage_addr;
	unsigned short spare12_addr;
	unsigned short ctrla_storage_dft;
	unsigned short ctrla;
	unsigned short spare12;

	/* Cannot use cs4224_get_cfg_side, that is for the Rx side, not Tx */
	if (cs4224_is_hw_simplex(slice) && (side == CS4224_CFG_SIMPLEX)) {
		if (cs4224_line_rx_to_host_tx_dir(slice))
			side = CS4224_CFG_HOST_SIDE;
		else
			side = CS4224_CFG_LINE_SIDE;
	} else {
		if (side == CS4224_CFG_SIMPLEX) {
			CS_TRACE(("ERROR: Duplex slice, not simplex\n"));
			status |= CS_ERROR;
		}
	}

	if (side == CS4224_CFG_LINE_SIDE) {
		squelch_addr = CS4224_PP_LINE_SDS_COMMON_STX0_SQUELCH;
		ctrla_addr = CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLA;
		ctrla_storage_addr =
			CS4224_PP_LINE_SDS_COMMON_COREFRAC0_STAGE2PRELOAD0;
		spare12_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_SPARE12_LSB;
	} else {
		squelch_addr = CS4224_PP_HOST_SDS_COMMON_STX0_SQUELCH;
		ctrla_addr = CS4224_PP_HOST_SDS_COMMON_STX0_TX_OUTPUT_CTRLA;
		ctrla_storage_addr =
			CS4224_PP_HOST_SDS_COMMON_COREFRAC0_STAGE2PRELOAD0;
		spare12_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB;
	}
	ctrla_storage_dft =
		CS4224_PP_LINE_SDS_COMMON_COREFRAC0_STAGE2PRELOAD0_dft;

	status |= cs4224_reg_get_channel(slice, spare12_addr, &spare12);
	if (spare12 & 4) {
		CS_TRACE(("ERROR: Squelching controlled by microsequencer,"));
		CS_TRACE(("call cs4224_mseq_squelch_ctrl to disable auto"));
		CS_TRACE((" Tx squelch control on slice, or use "));
		CS_TRACE(("cs4224_send_squelch_request instead.\n"));
		return CS_ERROR;
	}

	if (squelch) {
		/* enable squelch */

		status |= cs4224_reg_get_channel(slice, ctrla_addr, &ctrla);
		if (ctrla != 0x0000) {
			/* save then clear ctrla */
			status |= cs4224_reg_set_channel(
				slice, ctrla_storage_addr, ctrla);
			status |= cs4224_reg_set_channel(slice, ctrla_addr,
								0x0000);
		}
		status |= cs4224_reg_set_channel(slice, squelch_addr, 0x0001);
	} else {
		/* disable squelch */

		status |= cs4224_reg_get_channel(slice, ctrla_storage_addr,
							&ctrla);
		if (ctrla != ctrla_storage_dft) {
			/* restore cltra */
			status |= cs4224_reg_set_channel(slice, ctrla_addr,
								ctrla);
			/* set cltra storage back to 0xcccc (default) */
			status |= cs4224_reg_set_channel(
				slice, ctrla_storage_addr, ctrla_storage_dft);
		}
		status |= cs4224_reg_set_channel(slice, squelch_addr, 0x0000);
	}

	return status;
}

/* This method performs a hard reset of a single die. See the
 * header file for details
 */
int cs4224_hard_reset_die(unsigned int die)
{
	int status = CS_OK;
	unsigned int stride = 0;
	unsigned short data = 0x0;

	/* Wait for EEPROM download (which will be running after reset) */
	status |= cs4224_wait_for_eeprom_finished(die, 2500, 1);

	cs4224_lock(die);

	/* Fix the MDIO_CONFIG register so accesses to MSEQ registers
	 * don't time out (bug 35875)
	 */
	if (cs4224_reg_set(die, CS4224_GLOBAL_MDIO_CONFIG,
			   CS4224_GLOBAL_MDIO_CONFIG_pref) != CS_OK) {
		/* If there is something wrong with the comms then quit */
		CS_TRACE(("ERROR: Something wrong with reg get/set methods\n"));
		return CS_ERROR;
	}

	/* Blindly stall all the mseqs */
	for (stride = 0; stride < 0x4000; stride += 0x1000) {
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_MSEQCLKCTRL + stride,
			0x8004);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS + stride,
			0x0008);
		status |= cs4224_reg_set(
			die, CS4224_PP_HOST_HOSTMISC_MSEQCLKCTRL + stride,
			0x8004);
		status |= cs4224_reg_set(
			die, CS4224_PP_HOST_SDS_DSP_MSEQ_OPTIONS + stride,
			0x0008);
	}

	/* Use broadcast to write to all the ports at once,
	 * which can't be used to stall the mseqs
	 */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x80);
	{
		/* Reset all port-pairs (PP) on the die */
		status |= cs4224_reg_set(die, CS4224_PP_LINE_LINEMISC_CLKEN,
						0xFFFF);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			0x0000);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_SOFT_RESET, 0x0001);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_MPIF_RESET_DOTREG, 0x0007);
		/* Fix common_tx_sr default (ACJTAG) (bug 37448) */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_SOFT_RESET, 0x0000);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_MPIF_RESET_DOTREG, 0x0000);
		status |= cs4224_reg_set(die, CS4224_PP_LINE_LINEMISC_CLKEN,
						0x0000);
	}
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x0);

#ifndef CS_DONT_USE_MPIF_SOFT_RESET
	/* Reset a die of the ASIC. This write is self clearing */
	cs4224_reg_set(die, CS4224_GLOBAL_MPIF_SOFT_RESET, 0xDEAD);
#endif

	/* Wait for the EEPROM to finish downloading the die */
	status |= cs4224_wait_for_eeprom_finished(die, 2500, 1);

	/* Squelch all transmitters out of reset */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x80);

	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_STX0_SQUELCH, 0x1);

	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x0);

	if (status == CS_OK) {
		/* if everything worked, then set the scratch to say this die
		 * was successfully reset
		 */
		status |= cs4224_reg_get(die, CS4224_GLOBAL_SCRATCH6, &data);
		status |= cs4224_reg_set(die, CS4224_GLOBAL_SCRATCH6,
						CS_SET(data, CS_BIT1));
	}

	cs4224_unlock(die);

	/* reset the API static-state */
	cs4224_reset_die_static_state(die);

	return status;
}

/* This method performs a hard reset on the K2 package. See
 * the header file for details
 */
int cs4224_hard_reset(unsigned int slice)
{
	int status = CS_OK;
	unsigned int die = slice & 0xFFFFFF00;

	/* reset die 0, which is always present */
	status |= cs4224_hard_reset_die(die);

	if (CS4224_MAX_NUM_DIES(die) > 1) {
		/* reset die 1, which is chip-dependent */
		status |= cs4224_hard_reset_die(die | 0x1);
	}

	return status;
}

int cs4224_mseq_enable_power_savings(unsigned int slice,
				     enum e_cs4224_mseq_id mseq_id,
						unsigned char enable)
{
	int status = CS_OK;
	unsigned short reg_data = 0;
	unsigned short spare12 = 0;
	unsigned short clkdiv_ctrl = 0;
	enum e_cs4224_edc_mode edc_mode = CS_HSIO_EDC_MODE_DISABLED;
	unsigned char stalled;
	unsigned short offset = 0;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ)
		offset = 0;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	/* get the edc mode */
	status |= cs4224_query_edc_mode(slice, mseq_id, &edc_mode);
	/* see if the mseq is stalled */
	status |= cs4224_query_mseq_is_stalled(slice, mseq_id, &stalled);
	/* check the power-savings state */
	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + offset,
		&spare12);

	if (enable) {
		/* ring oscillator can't have power savings enabled; allow one
		 * to turn it off but not on
		 */
		status |= cs4224_reg_get_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG + offset,
			&reg_data);
		if (reg_data & CS_BIT7)
			return status;

		/* clkdiv helps us figure out the config of this Rx intf */
		status |= cs4224_reg_get_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKDIV_CTRL + offset,
			&clkdiv_ctrl);

		/* if the mseq is stalled or in SR mode then you need to control
		 * power savings manually
		 */
		if (stalled || (edc_mode == CS_HSIO_EDC_MODE_SR)) {
			/* if the fracn is needed we need to make sure we don't
			 * turn it off!
			 */
			if ((clkdiv_ctrl & 0x000F) == 0x0009)
				/*fracn used*/
				reg_data = 0x1e7;
			else
				/*fracn not used*/
				reg_data = 0x1f7;

			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB +
				    offset,
				    reg_data);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_MSB +
				    offset,
				    0x0);
		}

		/* power savings enable bit */
		spare12 |= CS_BIT1;
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + offset,
			spare12);
	} else { /* disable */
		/* if the mseq is stalled or in SR mode then you need to control
		 * power savings manually
		 */
		if (stalled || (edc_mode == CS_HSIO_EDC_MODE_SR)) {
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB +
				    offset,
				    0x65);
			status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_MSB +
				    offset,
				    0x0);
		}

		/* power savings bit */
		spare12 = CS_CLR(spare12, CS_BIT1);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + offset,
			spare12);

		/* Delay while the mseq changes the power down reg */
		CS_UDELAY(50);
	}

	return status;
}

int cs4224_query_mseq_power_savings(unsigned int slice,
				    enum e_cs4224_mseq_id mseq_id,
						unsigned char *enabled)
{
	int status = CS_OK;
	unsigned short reg_data = 0;
	unsigned int spare12_addr;

	*enabled = CS_FALSE;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ)
		spare12_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB;
	else
		spare12_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_SPARE12_LSB;

	status |= cs4224_reg_get_channel(slice, spare12_addr, &reg_data);

	if ((reg_data & 0x2) == 0x2)
		*enabled = CS_TRUE;
	else
		*enabled = CS_FALSE;

	return status;
}

int cs4224_query_mseq_is_stalled(unsigned int slice,
				 enum e_cs4224_mseq_id mseq_id,
					unsigned char *stalled)
{
	int status = CS_OK;
	unsigned short options_shadow_addr;
	unsigned short data;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ)
		options_shadow_addr =
			CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS_SHADOW;
	else
		options_shadow_addr =
			CS4224_PP_HOST_SDS_DSP_MSEQ_OPTIONS_SHADOW;

	status |= cs4224_reg_get_channel(slice, options_shadow_addr, &data);

	if (data == 0xbada) {
		/* some kind of read issue with power savings */
		CS_TRACE(("ERROR: SDS_DSP_MSEQ_OPTIONS_SHADOW == 0xbada\n"));
		return CS_ERROR;
	}

	if ((data & 0x8) == 0x8)
		*stalled = CS_TRUE;
	else
		*stalled = CS_FALSE;

	return status;
}

int cs4224_clear_mailbox(unsigned int slice, enum e_cs4224_mseq_id mseq_id)
{
	int status = CS_OK;
	unsigned short mailbox_in_addr, mailbox_out_addr;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ) {
		mailbox_in_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_MAIL_IN_LSB;
		mailbox_out_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_OUT_LSB;
	} else {
		mailbox_in_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_MAIL_IN_LSB;
		mailbox_out_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_MAIL_OUT_LSB;
	}

	cs4224_lock(slice);

	/* clear the out mail box queues in case they have a squelch/un-squelch
	 * request in them
	 */
	status |= cs4224_reg_set_channel(slice, mailbox_out_addr, 0);
	status |= cs4224_reg_set_channel(slice, mailbox_out_addr + 1, 0);
	/* clear the in mail box queue request */
	status |= cs4224_reg_set_channel(slice, mailbox_in_addr, 0);
	status |= cs4224_reg_set_channel(slice, mailbox_in_addr + 1, 0);

	cs4224_unlock(slice);

	return status;
}

void cs4224_mseq_stall_set_delay(int us)
{
	g_cs4224_mseq_stall_delay_us = us;
}

int cs4224_mseq_stall_get_delay(void)
{
	return g_cs4224_mseq_stall_delay_us;
}

int cs4224_mseq_stall(unsigned int slice, enum e_cs4224_mseq_id mseq_id,
		      unsigned char stall)
{
	int status = CS_OK;
	unsigned short reg_offset = 0x0;
	unsigned short reg_data = 0;
	unsigned short bank, spare12 = 0;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ)
		reg_offset = 0x0;
	else
		reg_offset = CS4224_LINE_TO_HOST_OFFSET;

	cs4224_lock(slice);

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + reg_offset,
		&spare12);

	if (stall) {
		/* this enables the clocks even if the mseq is powered-down */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_MSEQCLKCTRL + reg_offset,
			0x8004);

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS + reg_offset,
			0x0008);

		/* Delay after stalling the microsequencer to ensure any timers
		 * have expired
		 */
		CS_UDELAY(cs4224_mseq_stall_get_delay());

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_ENABLE + reg_offset,
			0x0000);

		/* When stalling the microsequencer need to revert MSEQ_SERDES
		 * to it's default value or it may get set temporarily to 0x3b
		 * by the slicer calibration
		 */
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES + reg_offset,
			&reg_data);
		reg_data &= ~0xf; /* don't overwrite custom filtr */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES + reg_offset,
			reg_data);
	} else { /* un-stall */
		/* clear the SNR control reg, see youtrack cs-170 */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SNR_CTRL + reg_offset,
			0x0000);

		/* save the current bank */
		cs4224_reg_get_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
			&bank);

		/* force a bank change to trigger the clearing of the
		 * micro-sequencer power-down timer. See MSEQBANKSWAP reg for
		 * details
		 */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
			0x0007);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
			0x0000);

		/* if slice configured for FC-AN. Note that KR-AN does not use
		 * this stall/un-stall method
		 */
		if (spare12 & 0x0020) {
			if (mseq_id == CS4224_DPLX_LINE_MSEQ) {
				/* In FC-AN mode the the LINE side bank select
				 * is initialized to 4
				 */
				status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR,
				    0x0004);
			} else {
				/* In FC-AN mode the the HOST side bank select
				 * is initialized to 7
				 */
				status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_HOST_SDS_DSP_MSEQ_BANK_SELECTOR,
				    0x0007);
			}
		} else { /* not FC-AN */
			/* Revert the bank-select in DFE mode. This will not
			 * work for protocol modes like KR-AN and FC-AN.
			 */
			if (bank == 3) {
				/* In SR mode the bank select is fixed at 3. */
				status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				    reg_offset,
				    0x0003);
			} else {
				/* DFE modes */
				status |= cs4224_reg_set_channel(
				    slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				    reg_offset,
				    0x0000);
			}
		}

		/* if the microsequencer triggered dynamic reconfig is enabled,
		 * then force the bank to 7
		 */
		status |= cs4224_reg_get_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE26_MSB + reg_offset,
			&reg_data);
		if (reg_data & 0x8000) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				reg_offset,
				0x0007);
		}

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_MSEQCLKCTRL + reg_offset,
			0x0004);

		/* Restore the datastore registers before un-stalling */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + 1 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + 2 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + reg_offset,
			0x803c);

		/* reset the program counter and unstall */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_PC + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS + reg_offset,
			0x0007);
	}

	cs4224_unlock(slice);

	return status;
}

int cs4224_mseqs_stall(struct cs4224_interface_t interfaces[],
		       unsigned short length,
			     unsigned char stall)
{
	int status = CS_OK;
	unsigned short reg_data = 0;
	unsigned short bank, spare12_lsb = 0;
	unsigned int i = 0;

	for (i = 0; i < length; i++) {
		unsigned short reg_offset = 0x0;
		unsigned int slice = interfaces[i].slice;
		enum e_cs4224_mseq_id mseq_id;

		/* first convert mseq_id to be either line/host, so we don't
		 * have to have this check in the next loop
		 */
		status |= cs4224_get_mseq_id(interfaces[i].slice,
						&interfaces[i].mseq_id);
		if (status != CS_OK) {
			CS_TRACE(("ERROR: Converting interfaces\n"));
			return status;
		}
		mseq_id = interfaces[i].mseq_id;

		if (mseq_id == CS4224_DPLX_LINE_MSEQ)
			reg_offset = 0x0;
		else
			reg_offset = CS4224_LINE_TO_HOST_OFFSET;

		cs4224_lock(slice);
		if (stall) {
			/* this enables the clocks even if the microsequencer is
			 * powered-down
			 */
			status |=
				cs4224_reg_set_channel(
					slice,
					CS4224_PP_LINE_LINEMISC_MSEQCLKCTRL +
					reg_offset,
					0x8004);

			status |=
				cs4224_reg_set_channel(
					slice,
					CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS +
					reg_offset,
					0x0008);

			/* rest of stall code is after the loop */
		} else { /* un-stall */
			/* If the mseq is in control of squelch, set squelch
			 * right off the bat because the mseq starts off in a
			 * non EDC converged state.
			 */
			status |=
				cs4224_reg_get_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB +
				   reg_offset,
				   &spare12_lsb);
			if (spare12_lsb & CS_BIT2) {
				status |=
				   cs4224_reg_set_channel(
					slice,
					CS4224_PP_HOST_SDS_COMMON_STX0_SQUELCH -
					reg_offset,
					0x1); /* Tx side,inverse the mseq side*/
			}

			/* save the current bank */
			status |=
				cs4224_reg_get_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				   reg_offset,
				   &bank);

			/* force a bank change to trigger the clearing of the
			 * micro-sequencer power-down timer. See MSEQBANKSWAP
			 * reg for details
			 */
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				   reg_offset,
				   0x0007);
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				   reg_offset,
				   0x0000);

			/* if slice configured for FC-AN. Note that KR-AN does
			 * not use this stall/un-stall method
			 */
			if (spare12_lsb & 0x0020) {
				if (mseq_id == CS4224_DPLX_LINE_MSEQ) {
					/* In FC-AN mode the the LINE side bank
					 * select is initialized to 4
					 */
					status |= cs4224_reg_set_channel(
						slice,
				      CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR,
						0x0004);
				} else {
					/* In FC-AN mode the the HOST side bank
					 * select is initialized to 7
					 */
					status |= cs4224_reg_set_channel(
						slice,
				      CS4224_PP_HOST_SDS_DSP_MSEQ_BANK_SELECTOR,
						0x0007);
				}
			} else {
				/* not FC-AN: Revert the bank-select in DFE
				 * mode. This will not work for protocol modes
				 * like KR-AN and FC-AN
				 */
				if (bank == 3) {
					/* In SR mode the bank select is fixed
					 * at 3.
					 */
					status |= cs4224_reg_set_channel(
						slice,
				     CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
						reg_offset,
						0x0003);
			} else {
				/* DFE modes */
				status |= cs4224_reg_set_channel(
				     slice,
				     CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				     reg_offset,
				     0x0000);
				}
			}

			/* if the microsequencer triggered dynamic reconfig is
			 * enabled, then force the bank to 7
			 */
			status |=
				cs4224_reg_get_channel(
					slice,
					CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE26_MSB
					+ reg_offset,
					&reg_data);
			if (reg_data & 0x8000) {
				status |= cs4224_reg_set_channel(
				     slice,
				     CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR +
				     reg_offset,
				     0x0007);
			}
			/* rest of unstall code after the loop */
		}
		cs4224_unlock(slice);
	}

	if (stall) {
		/* Delay after stalling the microsequencer to ensure any timers
		 * have expired
		 */
		CS_UDELAY(100);
	}

	for (i = 0; i < length; i++) {
		unsigned short reg_offset = 0x0;
		unsigned int slice = interfaces[i].slice;
		enum e_cs4224_mseq_id mseq_id = interfaces[i].mseq_id;

		if (mseq_id == CS4224_DPLX_LINE_MSEQ)
			reg_offset = 0x0;
		else
			reg_offset = CS4224_LINE_TO_HOST_OFFSET;

		cs4224_lock(slice);
		if (stall) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_ENABLE + reg_offset,
				0x0000);

			/* When stalling the microsequencer need to revert
			 * MSEQ_SERDES to it's default value or it may get set
			 * temporarily to 0x3b by the slicer calibration
			 */
			status |= cs4224_reg_get_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES + reg_offset,
				&reg_data);
			reg_data &= ~0xf; /* don't overwrite custom filtr */
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES + reg_offset,
				reg_data);
		} else {
			status |=
				cs4224_reg_set_channel(
					slice,
					CS4224_PP_LINE_LINEMISC_MSEQCLKCTRL +
					reg_offset,
					0x0004);

			/* Restore the datastore registers before un-stalling */
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + 1
				+ reg_offset,
				0x0000);
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + 2
				+ reg_offset,
				0x0000);
			status |=
				cs4224_reg_set_channel(
					slice,
					CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR +
					reg_offset,
					0x803c);

			/* reset the program counter and unstall */
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_PC + reg_offset,
				0x0000);
			status |=
				cs4224_reg_set_channel(
					slice,
					CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS +
					reg_offset,
					0x0007);
		}
		cs4224_unlock(slice);
	}

	return status;
}

int cs4224_init_global_timer_fixp(unsigned int slice, unsigned int ref_clk_freq)
{
	int status = CS_OK;
	unsigned int ref_clk_per;
	unsigned int num_clks;

	ref_clk_per = 1000000000L / ref_clk_freq;
	num_clks = 100000000L / ref_clk_per;

	cs4224_lock(slice);

	status |= cs4224_reg_set_channel(
		slice, CS4224_GLOBAL_GT_10KHZ_REF_CLK_CNT0, num_clks);
	status |= cs4224_reg_set_channel(
		slice, CS4224_GLOBAL_GT_10KHZ_REF_CLK_CNT1, num_clks >> 16);

	cs4224_unlock(slice);

	return status;
}

int cs4224_init_lc_vco_tmp_thresh(unsigned int slice,
				  enum e_cs4224_cfg_sides_t intf)
{
	int status = CS_OK;
	unsigned short offset;

	if (intf == CS4224_CFG_LINE_SIDE)
		offset = 0x0000;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	/* These thresholds valid for LC VCO only */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES00 + offset,
		0x13a);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES01 + offset,
		0x127);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES02 + offset,
		0x117);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES03 + offset,
		0x0);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES04 + offset,
		0x0);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES10 + offset,
		0x13f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES11 + offset,
		0x12f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES12 + offset,
		0x127);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES13 + offset,
		0x0);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES14 + offset,
		0x0);

	return status;
}

int cs4224_init_ro_vco_tmp_thresh(unsigned int slice,
				  enum e_cs4224_cfg_sides_t intf)
{
	int status = CS_OK;
	unsigned short offset;

	if (intf == CS4224_CFG_LINE_SIDE)
		offset = 0x0000;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	/* These thresholds valid for Ring Oscillator VCO only */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES00 + offset,
		0x018f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES01 + offset,
		0x015f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES02 + offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES03 + offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES04 + offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES10 + offset,
		0x018f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES11 + offset,
		0x015f);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES12 + offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES13 + offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPTHRES14 + offset,
		0x0000);

	return status;
}

int cs4224_init_10g(unsigned int slice, enum e_cs4224_cfg_sides_t intf,
		    struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned short offset = 0;

	if (intf == CS4224_CFG_LINE_SIDE)
		offset = 0x0000;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	cs4224_lock(slice);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_DLY_CTRL2 + offset,
		0x1000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_DLY_CTRL1 + offset,
		0x0008);

	cs4224_unlock(slice);

	return status;
}

int cs4224_config_target_application_intf(unsigned int slice,
					  enum e_cs4224_cfg_sides_t intf,
						struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	status |= cs4224_get_cfg_side(slice, &intf);
	if (status != CS_OK)
		return status;

	switch (rules->application) {
	case CS4224_TARGET_APPLICATION_5G:
	case CS4224_TARGET_APPLICATION_7p5G:
	case CS4224_TARGET_APPLICATION_8G:
	case CS4224_TARGET_APPLICATION_8p5G:
	case CS4224_TARGET_APPLICATION_10G_FC:
	case CS4224_TARGET_APPLICATION_10G_KR:
	case CS4224_TARGET_APPLICATION_40G_KR:
	case CS4224_TARGET_APPLICATION_KRAN:
	case CS4224_TARGET_APPLICATION_15G:
	case CS4224_TARGET_APPLICATION_16G_FC:
	case CS4224_TARGET_APPLICATION_8G_FC:
	case CS4224_TARGET_APPLICATION_1G_FC:
	case CS4224_TARGET_APPLICATION_2G_FC:
	case CS4224_TARGET_APPLICATION_4G_FC:
	case CS4224_TARGET_APPLICATION_FCAN:
	case CS4224_TARGET_APPLICATION_OC12:
	case CS4224_TARGET_APPLICATION_1G:
		status = CS_ERROR;
		break;
	case CS4224_TARGET_APPLICATION_10G:
		status |= cs4224_init_10g(slice, intf, rules);
		break;

	default:
		status = CS_ERROR;
		break;
	}

	return status;
}

int cs4224_config_target_application(unsigned int slice,
				     struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice))
		status |= cs4224_config_target_application_intf(
			slice, CS4224_CFG_LINE_SIDE, rules);

	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice))
		status |= cs4224_config_target_application_intf(
			slice, CS4224_CFG_HOST_SIDE, rules);

	return status;
}

int cs4224_pgm_reg_from_efuse(unsigned int slice,
			      enum e_cs4224_cfg_sides_t dir)
{
	int status = CS_OK;
	unsigned short efuse_data0, efuse_data1, efuse_data2;
	unsigned short cml_itune, cml_rtune;
	unsigned short vco_itune;
	unsigned short driver_rtune, driver_itune;
	unsigned short agc_itune, agc_rtune;
	unsigned short drivercml_rtune, drivercml_itune;
	unsigned short reg_data, old_data;
	unsigned int die = cs4224_get_die_from_slice(slice);
	unsigned short reg_offset = 0x0000;

	cs4224_lock(slice);

	status |= cs4224_reg_get(die, CS4224_EFUSE_PDF_POLY_RES_CAL_DATA0,
					&efuse_data0);
	status |= cs4224_reg_get(die, CS4224_EFUSE_PDF_POLY_RES_CAL_DATA1,
					&efuse_data1);
	status |= cs4224_reg_get(die, CS4224_EFUSE_PDF_POLY_RES_CAL_DATA2,
					&efuse_data2);

	if (dir == CS4224_CFG_LINE_SIDE)
		reg_offset = 0;
	else
		reg_offset = CS4224_LINE_TO_HOST_OFFSET;

	/* assign efuse cml_itune to
	 * PP_LINE_SDS_COMMON_SRX0_RX_IBIAS_TUNE[2:0]
	 */
	cml_itune = efuse_data0 & 0x0070;

	status |= cs4224_reg_get_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_IBIAS_TUNE + reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0x0007) | (cml_itune >> 4);
	if (old_data != reg_data) {
		status |=
			cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_SRX0_RX_IBIAS_TUNE +
				reg_offset,
				reg_data);
	}

	/* assign vco_itune to PP_LINE_SDS_COMMON_SRX0_RX_RBIAS_TUNE[6:4]
	 * and cml_rtune to PP_LINE_SDS_COMMON_SRX0_RX_RBIAS_TUNE[2:0]
	 */
	vco_itune = efuse_data0 & 0x7000;
	cml_rtune = efuse_data1 & 0x7000;

	status |= cs4224_reg_get_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_RBIAS_TUNE + reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0x0077) | (cml_rtune >> 12) | (vco_itune >> 8);
	if (old_data != reg_data) {
		status |=
			cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_SRX0_RX_RBIAS_TUNE +
				reg_offset,
				reg_data);
	}

	/* assign efuse agc_itune to
	 * CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG2[6:4]
	 */
	agc_itune = efuse_data0 & 0x0007;

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG2 + reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0xFF8F) | (agc_itune << 4);
	if (old_data != reg_data) {
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG2 + reg_offset,
			reg_data);
	}

	/* assign driver_rtune to
	 * CS4224_PP_LINE_SDS_COMMON_STX0_DRIVER_TUNE[10:8]   and
	 * driver_itune to CS4224_PP_LINE_SDS_COMMON_STX0_DRIVER_TUNE[2:0]
	 */
	driver_rtune = efuse_data2 & 0x0070;
	driver_itune = efuse_data1 & 0x0007;

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_TUNE - reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0x0707) | (driver_rtune << 4)
			| (driver_itune >> 0);
	if (old_data != reg_data) {
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_TUNE - reg_offset,
			reg_data);
	}

	/* assign drivercml_rtune to
	 * CS4224_PP_LINE_SDS_COMMON_STX0_DRIVERCML_TUNE[10:8]
	 * and drivercml_itune to
	 * CS4224_PP_LINE_SDS_COMMON_STX0_DRIVERCML_TUNE[2:0]
	 */
	drivercml_rtune = efuse_data2 & 0x0700;
	drivercml_itune = efuse_data1 & 0x0070;

	status |= cs4224_reg_get_channel(
		slice,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVERCML_TUNE - reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0x0707) | (drivercml_rtune << 0)
			| (drivercml_itune >> 4);
	if (old_data != reg_data) {
		status |=
			cs4224_reg_set_channel(
				slice,
				CS4224_PP_HOST_SDS_COMMON_STX0_DRIVERCML_TUNE -
				reg_offset,
				reg_data);
	}

	/* assign efuse agc_itune to
	 * CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_RTUNE[2:0]
	 */
	agc_rtune = efuse_data1 & 0x0700;

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_RTUNE + reg_offset,
		&reg_data);
	old_data = reg_data;
	reg_data = (reg_data & ~0x0007) | (agc_rtune >> 8);
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_RTUNE + reg_offset,
			reg_data);

	/* the value for the dfe i/rtune should not be based on the
	 * E-Fuse setting
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG + reg_offset,
		0x1910);

	/* over-ride efuse data with registers */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_LINEMISC_OVERRIDE_EN + reg_offset,
		0x0006);

	cs4224_unlock(slice);

	return status;
}

int cs4224_update_pre_equalizer(unsigned int slice, unsigned short reg_offset,
				enum e_cs4224_trace_loss traceloss)
{
	int status = CS_OK;
	unsigned short lim_eqadj;
	unsigned short pass_eqadj1;
	unsigned short pass_eqadj2;

	switch (traceloss) {
	case CS_HSIO_TRACE_LOSS_0dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0001;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_1dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0000;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_2dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0001;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_3dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0004;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_4dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0005;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_5dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0006;
		pass_eqadj2 = 0x070F;
		break;
	}
	case CS_HSIO_TRACE_LOSS_6dB: {
		lim_eqadj = 0x0100;
		pass_eqadj1 = 0x0007;
		pass_eqadj2 = 0x070F;
		break;
	}
	default: {
		return CS_OK;
	}
	}
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_MISC + reg_offset,
		lim_eqadj);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_EQADJ1 + reg_offset,
		pass_eqadj1);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_EQADJ2 + reg_offset,
		pass_eqadj2);

	return status;
}

int cs4224_save_edc_mode(unsigned int slice, unsigned short reg_offset,
			 enum e_cs4224_edc_mode edc_mode)
{
	int status = CS_OK;
	unsigned short saved_edc_mode;

	switch (edc_mode) {
	case CS_HSIO_EDC_MODE_DISABLED:
		saved_edc_mode = 0;
		break;
	case CS_HSIO_EDC_MODE_CX1:
		saved_edc_mode = 1;
		break;
	case CS_HSIO_EDC_MODE_SR:
		saved_edc_mode = 2;
		break;
	case CS_HSIO_EDC_MODE_ZR:
		saved_edc_mode = 3;
		break;
	case CS_HSIO_EDC_MODE_DWDM:
		saved_edc_mode = 4;
		break;
	case CS_HSIO_EDC_MODE_10G_BP:
		saved_edc_mode = 5;
		break;
	case CS_HSIO_EDC_MODE_15G_BP:
		saved_edc_mode = 6;
		break;
	case CS_HSIO_EDC_MODE_5G_BP:
		saved_edc_mode = 7;
		break;
	case CS_HSIO_EDC_MODE_7p5G_BP:
		saved_edc_mode = 8;
		break;
	case CS_HSIO_EDC_MODE_8p5G_BP:
		saved_edc_mode = 9;
		break;
	case CS_HSIO_EDC_MODE_FCAN:
		saved_edc_mode = 10;
		break;
	case CS_HSIO_EDC_MODE_15G_BP_27dB:
		saved_edc_mode = 11;
		break;
	case CS_HSIO_EDC_MODE_SMLRM:
		saved_edc_mode = 12;
		break;

	default:
		CS_TRACE(("ERROR: %s: slice = %d, ", __func__, slice));
		CS_TRACE(("unknown EDC mode = %d\n", edc_mode));
		status |= CS_ERROR;
		saved_edc_mode = 1;
		break;
	}
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_STAGE2PRELOAD0 + reg_offset,
		saved_edc_mode);

	return status;
}

int cs4224_query_edc_mode(unsigned int slice, enum e_cs4224_mseq_id mseq_id,
			  enum e_cs4224_edc_mode *edc_mode)
{
	int status = CS_OK;
	unsigned short saved_edc_mode = 0;
	/* offset used for accessing host addresses over line ones */
	unsigned short reg_offset = 0x0000;

	/* init to disabled */
	*edc_mode = CS_HSIO_EDC_MODE_DISABLED;

	status |= cs4224_get_mseq_id(slice, &mseq_id);
	if (status != CS_OK)
		return status;

	if (mseq_id == CS4224_DPLX_LINE_MSEQ)
		reg_offset = 0x0000;
	else
		reg_offset = CS4224_LINE_TO_HOST_OFFSET;

	status |= cs4224_reg_get_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_STAGE2PRELOAD0 + reg_offset,
		&saved_edc_mode);

	switch (saved_edc_mode) {
	/* Note, edc mode enum uses too many bits, hense the custom mapping here
	 * If the EDC version has never been initialized then the preloader will
	 * have the h/w reset value of 0xCCCC. Report this as EDC disabled
	 */
	case 0:
	case 0xCCCC:
		*edc_mode = CS_HSIO_EDC_MODE_DISABLED;
		break;
	case 1:
		*edc_mode = CS_HSIO_EDC_MODE_CX1;
		break;
	case 2:
		*edc_mode = CS_HSIO_EDC_MODE_SR;
		break;
	case 3:
		*edc_mode = CS_HSIO_EDC_MODE_ZR;
		break;
	case 4:
		*edc_mode = CS_HSIO_EDC_MODE_DWDM;
		break;
	case 5:
		*edc_mode = CS_HSIO_EDC_MODE_10G_BP;
		break;
	case 6:
		*edc_mode = CS_HSIO_EDC_MODE_15G_BP;
		break;
	case 7:
		*edc_mode = CS_HSIO_EDC_MODE_5G_BP;
		break;
	case 8:
		*edc_mode = CS_HSIO_EDC_MODE_7p5G_BP;
		break;
	case 9:
		*edc_mode = CS_HSIO_EDC_MODE_8p5G_BP;
		break;
	case 10:
		*edc_mode = CS_HSIO_EDC_MODE_FCAN;
		break;
	case 11:
		*edc_mode = CS_HSIO_EDC_MODE_15G_BP_27dB;
		break;
	case 12:
		*edc_mode = CS_HSIO_EDC_MODE_SMLRM;
		break;

	default:
		CS_TRACE(("ERROR: slice = %x, unknown saved EDC mode = %x\n",
			  slice, saved_edc_mode));
		status |= CS_ERROR;
		break;
	}
	return status;
}

int cs4224_init_edc_mode_intf(unsigned int slice,
			      struct cs4224_rules_t *rules,
				    enum e_cs4224_mseq_id intf)
{
	int status = CS_OK;
	enum e_cs4224_edc_mode mode;
	enum e_cs4224_trace_loss traceloss;
	/* offset used for accessing host addresses over line ones */
	unsigned short reg_offset = 0x0000;
	/* flags for fc or 8g apps */
	unsigned short reg_data = 0x0000;

	cs4224_lock(slice);

	if (cs4224_is_hw_simplex(slice)) {
		mode = rules->rx_if.splx_edc_mode;
		traceloss = rules->rx_if.splx_eq.traceloss;

		if (cs4224_line_rx_to_host_tx_dir(slice)) {
			intf = CS4224_DPLX_LINE_MSEQ;
			reg_offset = 0x0000;
		} else {
			intf = CS4224_DPLX_HOST_MSEQ;
			reg_offset = CS4224_LINE_TO_HOST_OFFSET;
		}
	} else if (cs4224_is_hw_duplex(slice)) {
		if (intf == CS4224_DPLX_LINE_MSEQ) {
			reg_offset = 0x0000;

			mode = rules->rx_if.dplx_line_edc_mode;
			traceloss = rules->rx_if.dplx_line_eq.traceloss;
		} else if (intf == CS4224_DPLX_HOST_MSEQ) {
			reg_offset = CS4224_LINE_TO_HOST_OFFSET;

			mode = rules->rx_if.dplx_host_edc_mode;
			traceloss = rules->rx_if.dplx_host_eq.traceloss;
		} else {
			CS_TRACE(("ERROR: Invalid interface. intf=%d\n", intf));
			return CS_ERROR;
		}
	} else {
		/* error in the API checks for SKU... */
		CS_TRACE(("ERROR: unknown hardware id\n"));
		return CS_ERROR;
	}

	/* reset reset count */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_RESET_COUNT_LSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_RESET_COUNT_MSB + reg_offset,
		0x0000);

	/* Reset MSEQ datastore register between mode switches */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D0 + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D1 + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + reg_offset,
		0x803c);

	/* function number */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D0 + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D1 + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + reg_offset,
		0x800E);

	/* Reset some MSEQ registers to defaults between mode switches
	 * (may be overwritten below)
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG4 + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG4_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_MSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE4_LSB + reg_offset,
		0x3000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES + reg_offset,
		CS4224_PP_LINE_SDS_DSP_MSEQ_SERDES_dft);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_RXLOCKD0_RESOLUTION + reg_offset,
		0x0001);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE21_LSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE21_MSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE13_LSB + reg_offset,
		0x0008);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE13_MSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_BIAS1 + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_BIAS1_dft);

	status |= cs4224_save_edc_mode(slice, reg_offset, mode);

	switch (mode) {
	case CS_HSIO_EDC_MODE_SR: {
		/* only set the SR bit, everything else is off by default */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + reg_offset,
			0x0001);

		/* set spare1[0] = 1 to speed up locking in SR mode */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE1_LSB + reg_offset,
			0x0001);

		/* power down is set differently from the scripts since spare12
		 * is also different
		 */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
			0x0065);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_MSB + reg_offset,
			0x0000);

		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
			0x0003);

		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA + reg_offset,
			0x0099);

		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1 + reg_offset,
			0x0004);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG + reg_offset,
			0x0441);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB + reg_offset,
			0xDFFF);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB + reg_offset,
			0x000D);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_PHSEL + reg_offset,
			0x001E);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_OVERRIDE_EN + reg_offset,
			0x0000);

		/* function number set to 1 for SR only */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D0 + reg_offset,
			0x0001);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_D1 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_GRAM_CR + reg_offset,
			0x800E);

		/* apply pre-equalizer settings */
		status |= cs4224_update_pre_equalizer(slice, reg_offset,
							traceloss);

		break;
	}

	case CS_HSIO_EDC_MODE_CX1:
	case CS_HSIO_EDC_MODE_SMLRM: {
		/* differs from script, don't enable power savings or
		 * auto squelch by default
		 */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_LSB + reg_offset,
			0x0B00);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_MSB + reg_offset,
			0x0C09);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
			0x001F);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA + reg_offset,
			0x0099);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1 + reg_offset,
			0x0005);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB + reg_offset,
			0x8F00);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB + reg_offset,
			0x000c);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_EQADJ1 + reg_offset,
			0x0008);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_EQADJ2 + reg_offset,
			0x000B);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_BANK_SELECTOR + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE9_LSB + reg_offset,
			0x00FF);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE9_MSB + reg_offset,
			0x0030);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE6_LSB + reg_offset,
			0x000F);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE6_MSB + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE2_LSB + reg_offset,
			0x2350);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE2_MSB + reg_offset,
			0x1650);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE18_MSB + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE18_LSB + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE11_MSB + reg_offset,
			0x07FF);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE11_LSB + reg_offset,
			0xFFFE);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE7_LSB + reg_offset,
			0x0010);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE7_MSB + reg_offset,
			0x0030);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_COEF_DSP_FLOAT + reg_offset,
			0x8001);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE0_LSB + reg_offset,
			0x001D);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE0_MSB + reg_offset,
			0x000A);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE4_LSB + reg_offset,
			0x3000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE4_MSB + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG4 + reg_offset,
			0x85f0);

		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE21_LSB + reg_offset,
			0x000A);

		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_IC_DFE0_1 + reg_offset,
			0x00F0);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_IC_DFE1_1 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_IC_DFE2_1 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_IC_DFE3_1 + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_IC_DFE4_1 + reg_offset,
			0x0000);

		/* Read efuse data (POLY_RES_CAL_DATAx) and configure
		 * itune/rtune settings
		 */
		if (intf == CS4224_DPLX_LINE_MSEQ)
			status |= cs4224_pgm_reg_from_efuse(
				slice, CS4224_CFG_LINE_SIDE);
		else
			status |= cs4224_pgm_reg_from_efuse(
				slice, CS4224_CFG_HOST_SIDE);

		if (mode == CS_HSIO_EDC_MODE_CX1) {
			/* the following sets the data slicer offset:
			 * 0x80 for CX1, 0x90 for ZR, DWDM TBD
			 */
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_SLICER +
				   reg_offset,
				   0x0080);
			/* max PHASE_SELECT+1 */
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE21_MSB +
				   reg_offset,
				   0x0023);
		} else {
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_SLICER +
				   reg_offset,
				   0x0040);
			/* max PHASE_SELECT+1 */
			status |=
				cs4224_reg_set_channel(
				   slice,
				   CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE21_MSB +
				   reg_offset,
				   0x0020);
		}
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_RTUNE + reg_offset,
			0x0000);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG2 + reg_offset,
			0x0000);

		break;
	}

	case CS_HSIO_EDC_MODE_DWDM:
	case CS_HSIO_EDC_MODE_ZR:
	case CS_HSIO_EDC_MODE_10G_BP:
	case CS_HSIO_EDC_MODE_15G_BP:
	case CS_HSIO_EDC_MODE_15G_BP_27dB:
	case CS_HSIO_EDC_MODE_5G_BP:
	case CS_HSIO_EDC_MODE_7p5G_BP:
	case CS_HSIO_EDC_MODE_8p5G_BP:
		printf("EDC Mode not supported\n");
		break;
	case CS_HSIO_EDC_MODE_DISABLED: {
		/* un-rail the driver charge pump current */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA + reg_offset,
			0x0099);

		/* power everything up */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
			0x001f);
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
			0x0000);

		/* un-squelch the driver */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_HOST_SDS_COMMON_STX0_SQUELCH - reg_offset,
			0x0000);

		/* apply pre-equalizer settings */
		status |= cs4224_update_pre_equalizer(slice, reg_offset,
							traceloss);

		break;
	}
	default: {
		status |= cs4224_save_edc_mode(slice, reg_offset,
						CS_HSIO_EDC_MODE_DISABLED);

		CS_TRACE(
			("ERROR: Invalid application mode %d.\n", mode));
		status |= CS_ERROR;
		return status;
	}
	}

	/* if the ring oscillator VCO is used, override some of these values */
	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG + reg_offset,
		&reg_data);
	if (reg_data & CS_BIT7) {
		/* the ring oscillator requires the charge pump = 0x44 */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA + reg_offset,
			0x0044);

		/* when using the ring oscillator, override srx_cml_rtune=0,
		 * srx_vco_itune=2 and srx_cml_itune=0
		 */
		status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_SRX0_RX_RBIAS_TUNE + reg_offset,
		   0x0020);
		status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_SRX0_RX_IBIAS_TUNE + reg_offset,
		   0);
	}

	cs4224_unlock(slice);

	return status;
}

int cs4224_init_edc_mode_set(unsigned int slice,
			     struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	if (cs4224_is_hw_simplex(slice)) {
		status |= cs4224_init_edc_mode_intf(slice, rules,
							CS4224_SPLX_MSEQ);
	} else if (cs4224_is_hw_duplex(slice)) {
		if (rules->application != CS4224_TARGET_APPLICATION_FCAN) {
			/* the FC-AN ucode configures the line side DFE mode,
			 * don't do it here...
			 */
			status |= cs4224_init_edc_mode_intf(
				slice, rules, CS4224_DPLX_LINE_MSEQ);
		}
		status |= cs4224_init_edc_mode_intf(slice, rules,
							CS4224_DPLX_HOST_MSEQ);
	} else {
		/* error in the API checks for SKU... */
		CS_TRACE(("ERROR: init_edc_mode_set: unknown hardware id"));
		return CS_ERROR;
	}

	return status;
}

unsigned int cs4224_query_vco_rate(const struct cs4224_rules_t *rules)
{
	/* ref_clk_rate is in KHz */
	unsigned int ref_clk_rate = 0;
	/* rdiv from clkdiv rules */
	unsigned int rdiv = 0;
	/* calculated (approx) vco rate */
	unsigned int vco_rate = 0;

#if defined(CS_HAS_FLOATING_POINT)
	/* if we're using floating point then the ref_clk is in MHz,
	 * multiply up to get KHz and scrap the decimal
	 */
	ref_clk_rate = rules->ref_clk_rate * 1000;
#else
	/* if we're not using floating point, then ref_clk_rate is in KHz */
	ref_clk_rate = rules->ref_clk_rate;
#endif

	/* decode clkdiv.rdiv for use in the calc */
	switch (rules->clkdiv.rdiv) {
	case CS4224_RDIV_DIV64:
		rdiv = 64;
		break;
	case CS4224_RDIV_DIV66:
		rdiv = 66;
		break;
	case CS4224_RDIV_DIV80:
		rdiv = 80;
		break;
	case CS4224_RDIV_DIV100:
		rdiv = 100;
		break;
	case CS4224_RDIV_DIV128:
		rdiv = 128;
		break;
	default:
		/* doesn't matter,we might be out of range or using the FRACn */
		rdiv = 0;
		break;
	}

	if (rules->clkdiv.enable) {
		/* we're just trying to find the VCO rate, not the data-rate,
		 * so ignore ddiv
		 */
		vco_rate = ref_clk_rate * rdiv;
	}

	/* if using the clkdiv (for ddiv) AND the fracn, then this if statement
	 * must come last
	 */
	if (rules->fracdiv.enable) {
		unsigned long long frac_part =
					 (unsigned long long)(ref_clk_rate) *
					 (rules->fracdiv.numerator);
		/* decimate the 'fractional' part, left with int KHz */
		frac_part = frac_part >> 21;
		vco_rate = (ref_clk_rate * 8 * rules->fracdiv.divisor)
			+ (unsigned int)frac_part;
	}

	return vco_rate;
}

unsigned int cs4224_query_data_rate(const struct cs4224_rules_t *rules)
{
	unsigned int vco_rate = cs4224_query_vco_rate(rules);
	unsigned int data_rate = vco_rate;

	if (rules->clkdiv.enable) {
		switch (rules->clkdiv.ddiv) {
		case CS4224_RULE_DISABLED:
			/* not sure how this happened, but ignore it */
			break;
		default:
			/* shift by the enum, since the divide
			 * amount == 2**(enum)
			 */
			data_rate =
			       vco_rate >> (unsigned short)(rules->clkdiv.ddiv);
			break;
		}
	}

	return data_rate;
}

unsigned char cs4224_need_low_lc_vco(const struct cs4224_rules_t *rules)
{
	unsigned int vco_rate = cs4224_query_vco_rate(rules);

	/* the 'real' range is 11.3G to 12.5G, but we'll fudge it a bit in case
	 * our fracn calcs have some rounding errors
	 */
	if ((vco_rate >= 11280000) && (vco_rate <= 12520000))
		return CS_TRUE;

	return CS_FALSE;
}

int cs4224_force_low_lc_vco_intf(unsigned int slice,
				 enum e_cs4224_datapath_dir_t intf,
					unsigned char force_enable)
{
	int status = CS_OK;
	unsigned short offset = 0;

	if (intf == CS4224_LINE_RX_TO_HOST_TX_DIR)
		offset = 0;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	if (force_enable)
		status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_ALTCT_LIMIT_HI + offset,
		   0x40);
	else
		status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_ALTCT_LIMIT_HI + offset,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_ALTCT_LIMIT_HI_dft);

	return status;
}

int cs4224_force_and_init_low_lc_vco(unsigned int slice,
				     unsigned char force_enable)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_force_low_lc_vco_intf(
			slice, CS4224_LINE_RX_TO_HOST_TX_DIR, force_enable);
		status |= cs4224_init_vco(
			slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_CONTROL);
	}

	if (cs4224_is_hw_duplex(slice) ||
	    (!cs4224_line_rx_to_host_tx_dir(slice))) {
		status |= cs4224_force_low_lc_vco_intf(
			slice, CS4224_HOST_RX_TO_LINE_TX_DIR, force_enable);
		status |= cs4224_init_vco(
			slice, CS4224_PP_HOST_SDS_COMMON_RXVCO0_CONTROL);
	}

	return status;
}

int cs4224_force_low_lc_vco(unsigned int slice, unsigned char force_enable)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_force_low_lc_vco_intf(
			slice, CS4224_LINE_RX_TO_HOST_TX_DIR, force_enable);
	}

	if (cs4224_is_hw_duplex(slice) ||
	    (!cs4224_line_rx_to_host_tx_dir(slice))) {
		status |= cs4224_force_low_lc_vco_intf(
			slice, CS4224_HOST_RX_TO_LINE_TX_DIR, force_enable);
	}

	return status;
}

int cs4224_init_alt_coarse_tuning_intf(unsigned int slice,
				       enum e_cs4224_cfg_sides_t intf,
					     struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned short reg_data = 0;
	unsigned short offset = 0x0000;
	/* flag to say whether we should force the low lc vco or not */
	unsigned char force_low_lc_vco = CS_FALSE;

	status |= cs4224_get_cfg_side(slice, &intf);
	if (status != CS_OK)
		return status;

	cs4224_lock(slice);

	if (intf == CS4224_CFG_LINE_SIDE) {
		offset = 0x0000;
	} else if (intf == CS4224_CFG_HOST_SIDE) {
		offset = CS4224_LINE_TO_HOST_OFFSET;
	} else {
		CS_TRACE(("ERROR: Invalid interface. intf=%d\n", intf));
		return CS_ERROR;
	}

	/* level 4d00 is approx 77 deg C */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPLVL3 + offset,
		0x4d00);
	/* level 4100 is approx 65 deg C */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPLVL2 + offset,
		0x4100);
	/* level 3900 is approx 30 deg C */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPLVL1 + offset,
		0x3900);
	/* level 2d00 is approx 0 deg C */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXVCO0_TMPLVL0 + offset,
		0x2d00);

	/* ring oscillator config reg */
	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG + offset,
		&reg_data);

	if (reg_data & CS_BIT7) {
		/* if the 8G ring oscillator VCO is used */
		status |= cs4224_init_ro_vco_tmp_thresh(slice, intf);
	} else {
		/* the LC VCO */
		status |= cs4224_init_lc_vco_tmp_thresh(slice, intf);

		/* determine if we need to force the low LC VCO */
		force_low_lc_vco = cs4224_need_low_lc_vco(rules);
	}

	status |= cs4224_force_low_lc_vco(slice, force_low_lc_vco);

	cs4224_unlock(slice);

	return status;
}

int cs4224_init_alt_coarse_tuning(unsigned int slice,
				  struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) || cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_init_alt_coarse_tuning_intf(
			slice, CS4224_CFG_LINE_SIDE, rules);
	}
	if (cs4224_is_hw_duplex(slice) || !cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_init_alt_coarse_tuning_intf(
			slice, CS4224_CFG_HOST_SIDE, rules);
	}

	return status;
}

int cs4224_disable_tx_driver_if_req(unsigned int slice,
				    struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	cs4224_lock(slice);

	/* Disable the transmitters if requested */
	if (cs4224_is_hw_simplex(slice)) {
		if (cs4224_line_rx_to_host_tx_dir(slice) &&
		    (rules->tx_if.splx_disable == CS_TRUE)) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_HOST_SDS_COMMON_STX0_TX_OUTPUT_CTRLA,
				0x0800);
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_HOST_SDS_COMMON_STX0_TX_OUTPUT_CTRLB,
				0);
			}

		if (!cs4224_line_rx_to_host_tx_dir(slice) &&
		    (rules->tx_if.splx_disable == CS_TRUE)) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLA,
				0x0800);
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLB,
				0);
		}
	} else { /* duplex */
		if (rules->tx_if.dplx_line_disable == CS_TRUE) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLA,
				0x0800);
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLB,
				0);
		}
		if (rules->tx_if.dplx_host_disable == CS_TRUE) {
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_HOST_SDS_COMMON_STX0_TX_OUTPUT_CTRLA,
				0x0800);
			status |= cs4224_reg_set_channel(
				slice,
				CS4224_PP_HOST_SDS_COMMON_STX0_TX_OUTPUT_CTRLB,
				0);
		}
	}

	cs4224_unlock(slice);

	return status;
}

int cs4224_init_driver_trace_loss_intf(
		unsigned int slice,
		enum e_cs4224_datapath_dir_t intf,
		enum e_cs4224_trace_loss trace_loss)
{
	int status = CS_OK;
	unsigned short ctrla = 0;
	unsigned short ctrlb = 0;
	enum e_cs4224_tx_driver_interface driver;

	if (intf == CS4224_LINE_RX_TO_HOST_TX_DIR) {
		driver = CS4224_TX_HOST_INTERFACE;
	} else if (intf == CS4224_HOST_RX_TO_LINE_TX_DIR) {
		driver = CS4224_TX_LINE_INTERFACE;
	} else {
		CS_TRACE(("ERROR: intf must be line or host, not simplex\n"));
		return CS_ERROR;
	}

	switch (trace_loss) {
	case CS_HSIO_TRACE_LOSS_0dB:
	case CS_HSIO_TRACE_LOSS_1dB: {
		ctrla = 0x814;
		ctrlb = 0x4;
		break;
	}
	case CS_HSIO_TRACE_LOSS_2dB: {
		ctrla = 0x3810;
		ctrlb = 0x5;
		break;
	}
	case CS_HSIO_TRACE_LOSS_3dB: {
		ctrla = 0x4818;
		ctrlb = 0x9;
		break;
	}
	case CS_HSIO_TRACE_LOSS_4dB: {
		ctrla = 0x4818;
		ctrlb = 0xb;
		break;
	}
	case CS_HSIO_TRACE_LOSS_5dB: {
		ctrla = 0x4818;
		ctrlb = 0xd;
		break;
	}
	case CS_HSIO_TRACE_LOSS_6dB: {
		ctrla = 0x4818;
		ctrlb = 0xe;
		break;
	}
	case CS_HSIO_TRACE_LOSS_15dB: {
		ctrla = 0x782B;
		ctrlb = 0x4;
		break;
	}
	case CS_HSIO_TRACE_LOSS_27dB: {
		ctrla = 0x7828;
		ctrlb = 0x80A;
		break;
	}
	case CS_HSIO_TRACE_LOSS_CR4: {
		ctrla = 0x5827;
		ctrlb = 0x6;
		break;
	}
	case CS_HSIO_TRACE_LOSS_LAB: {
		ctrla = 0x0813;
		ctrlb = 0x1;
		break;
	}
	default: {
		CS_TRACE(("ERROR: Unsupported traceloss setting %d\n",
			  trace_loss));
		return CS_ERROR;
	}
	}

	status |= cs4224_init_driver_settings(slice, driver, ctrla, ctrlb);

	return status;
}

int cs4224_init_mseq_dyn_reconfig_intf(unsigned int slice,
				       unsigned short offset)
{
	int status = CS_OK;
	unsigned short reg_data;

	/* SPARE12[30] tells the DFE mission mode code mseq_dyn_reconfig is
	 * being used and to back to the mseq_dyn_reconfig code once each
	 * mission mode loop is done.
	 */
	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_MSB + offset,
		&reg_data);
	reg_data |= 0x4000;
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE12_MSB + offset,
		reg_data);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE20_MSB + offset,
		0x4000);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE24_MSB + offset,
		0x4000);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE26_MSB + offset,
		0x801e);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_SPARE6_MSB + offset, 0x0d80);

	return status;
}

int cs4224_init_mseq_dyn_reconfig(unsigned int slice,
				  struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned short reg_data;

	if ((rules->mseq_dyn_reconfig == CS_TRUE) &&
	    (rules->application != CS4224_TARGET_APPLICATION_KRAN) &&
	    (rules->application != CS4224_TARGET_APPLICATION_FCAN)) {
		/* The mseq_dyn_reconfig feature is not compatible with
		 * the FC microcode
		 */
		status |= cs4224_reg_get_channel(
			slice, CS4224_GLOBAL_UCODE_VERSION_KR, &reg_data);
		if (reg_data == 0) {
			CS_TRACE(("ERROR: mseq_dyn_reconfig requires KR\n"));
			return CS_ERROR;
		}

		status |= cs4224_init_mseq_dyn_reconfig_intf(slice, 0);
		status |= cs4224_init_mseq_dyn_reconfig_intf(
			slice, CS4224_LINE_TO_HOST_OFFSET);
	}

	return status;
}

int cs4224_init_driver_trace_loss(unsigned int slice,
				  struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	enum e_cs4224_trace_loss host_traceloss, line_traceloss;

	if (cs4224_is_hw_simplex(slice)) {
		host_traceloss = rules->tx_if.splx_driver.traceloss;
		line_traceloss = rules->tx_if.splx_driver.traceloss;
	} else {
		host_traceloss = rules->tx_if.dplx_host_driver.traceloss;
		line_traceloss = rules->tx_if.dplx_line_driver.traceloss;
	}

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_init_driver_trace_loss_intf(
			slice, CS4224_LINE_RX_TO_HOST_TX_DIR, host_traceloss);
	}

	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_init_driver_trace_loss_intf(
			slice, CS4224_HOST_RX_TO_LINE_TX_DIR, line_traceloss);
	}

	return status;
}

int cs4224_init_driver_settings(unsigned int slice,
				enum e_cs4224_tx_driver_interface intf,
				unsigned short ctrla,
				unsigned short ctrlb)
{
	int status = CS_OK;
	unsigned short offset = 0;

	if (cs4224_is_hw_simplex(slice) &&
	    intf == CS4224_TX_SIMPLEX_INTERFACE) {
		/* figure out which driver we need to change */
		if (cs4224_line_rx_to_host_tx_dir(slice))
			intf = CS4224_TX_HOST_INTERFACE;
		else
			intf = CS4224_TX_LINE_INTERFACE;
	} else if (intf == CS4224_TX_SIMPLEX_INTERFACE) {
		CS_TRACE(("ERROR: duplex slice cannot use intf = "));
		CS_TRACE(("CS4224_TX_SIMPLEX_INTERFACE\n"));
		return CS_ERROR;
	}

	if (intf == CS4224_TX_HOST_INTERFACE)
		offset = CS4224_LINE_TO_HOST_OFFSET;
	else
		offset = 0;

	cs4224_lock(slice);

	/* update CTRLA/CTRLB registers */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLA + offset,
		ctrla);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_STX0_TX_OUTPUT_CTRLB + offset,
		ctrlb);

	cs4224_unlock(slice);

	return status;
}

unsigned char cs4224_datarate_less_than_10G(struct cs4224_rules_t *rules)
{
	unsigned int data_rate = cs4224_query_data_rate(rules);

	if (data_rate < 10000000)
		return CS_TRUE;

	return CS_FALSE;
}

int cs4224_update_clkdiv_ctrl_intf(unsigned int slice,
				   enum e_cs4224_cfg_sides_t intf,
					 struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned short reg_data, old_data;
	unsigned short offset = 0;

	status |= cs4224_get_cfg_side(slice, &intf);
	if (status != CS_OK)
		return status;

	if (intf == CS4224_CFG_LINE_SIDE)
		offset = 0x0000;
	else
		offset = CS4224_LINE_TO_HOST_OFFSET;

	cs4224_lock(slice);

	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKDIV_CTRL + offset,
		&reg_data);
	old_data = reg_data;

	if (rules->clkdiv.enable) {
		if (rules->clkdiv.rdiv != CS4224_RULE_DISABLED) {
			reg_data &= ~0x000f;
			reg_data |= rules->clkdiv.rdiv;
		}
		if (rules->clkdiv.ddiv != CS4224_RULE_DISABLED) {
			reg_data &= ~0x0070;
			reg_data |= (rules->clkdiv.ddiv << 4);
		}
		if (rules->clkdiv.fastdiv != CS4224_RULE_DISABLED) {
			reg_data &= ~0xf000;
			reg_data |= (rules->clkdiv.fastdiv << 12);
		}
	} else
		/* Only time clkdiv rule is not enabled is when using the
		 * fracdiv without any integer divide (so ~10G rates, etc).
		 * In that case reset it to defaults.
		 * LINE dft == HOST dft
		 */
		reg_data = CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKDIV_CTRL_dft;

	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKDIV_CTRL + offset,
			reg_data);

	cs4224_unlock(slice);

	return status;
}

int cs4224_update_clkdiv_ctrl(unsigned int slice,
			      struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) || cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_update_clkdiv_ctrl_intf(
			slice, CS4224_CFG_LINE_SIDE, rules);
	}
	if (cs4224_is_hw_duplex(slice) || !cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_update_clkdiv_ctrl_intf(
			slice, CS4224_CFG_HOST_SIDE, rules);
	}

	return status;
}

int cs4224_fracdiv_cdr_init(unsigned int slice,
			    enum e_cs4224_datapath_dir_t dir,
				  unsigned short divisor,
				  unsigned int numerator)
{
	int status = CS_OK;
	unsigned short reg_data, old_data;

	unsigned short frac0_reset_addr;
	unsigned short clkdiv_ctrl_addr;
	unsigned short frac0_width_addr;
	unsigned short frac0_intdiv_addr;
	unsigned short frac0_numerator0_addr;
	unsigned short frac0_numerator1_addr;
	unsigned short frac0_stage1preload0_addr;
	unsigned short frac0_stage1preload1_addr;
	unsigned short frac0_dither_addr;
	unsigned short frac0_stage_en_addr;
	unsigned short frac0_power_down_addr;
	unsigned short mseq_options_addr;
	unsigned short mseq_options;
	enum e_cs4224_mseq_id mseq_id;

	/* sides and datapath_dir are the same enum */
	status |= cs4224_get_cfg_side(slice, &dir);
	if (status != CS_OK)
		return status;

	if (dir == CS4224_LINE_RX_TO_HOST_TX_DIR) {
		frac0_reset_addr = CS4224_PP_LINE_SDS_COMMON_FRAC0_RESET;
		clkdiv_ctrl_addr =
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKDIV_CTRL;
		frac0_width_addr = CS4224_PP_LINE_SDS_COMMON_FRAC0_WIDTH;
		frac0_intdiv_addr = CS4224_PP_LINE_SDS_COMMON_FRAC0_INTDIV;
		frac0_numerator0_addr =
			CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_NUMERATOR0;
		frac0_numerator1_addr =
			CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_NUMERATOR1;
		frac0_stage1preload0_addr =
			CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_STAGE1PRELOAD0;
		frac0_stage1preload1_addr =
			CS4224_PP_LINE_SDS_COMMON_RDIVFRAC0_STAGE1PRELOAD1;
		frac0_dither_addr = CS4224_PP_LINE_SDS_COMMON_FRAC0_DITHER;
		frac0_stage_en_addr = CS4224_PP_LINE_SDS_COMMON_FRAC0_STAGE_EN;
		frac0_power_down_addr =
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		mseq_options_addr = CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS_SHADOW;
		mseq_id = CS4224_DPLX_LINE_MSEQ;
	} else {
		frac0_reset_addr = CS4224_PP_HOST_SDS_COMMON_FRAC0_RESET;
		clkdiv_ctrl_addr =
			CS4224_PP_HOST_SDS_COMMON_SRX0_RX_CLKDIV_CTRL;
		frac0_width_addr = CS4224_PP_HOST_SDS_COMMON_FRAC0_WIDTH;
		frac0_intdiv_addr = CS4224_PP_HOST_SDS_COMMON_FRAC0_INTDIV;
		frac0_numerator0_addr =
			CS4224_PP_HOST_SDS_COMMON_RDIVFRAC0_NUMERATOR0;
		frac0_numerator1_addr =
			CS4224_PP_HOST_SDS_COMMON_RDIVFRAC0_NUMERATOR1;
		frac0_stage1preload0_addr =
			CS4224_PP_HOST_SDS_COMMON_RDIVFRAC0_STAGE1PRELOAD0;
		frac0_stage1preload1_addr =
			CS4224_PP_HOST_SDS_COMMON_RDIVFRAC0_STAGE1PRELOAD1;
		frac0_dither_addr = CS4224_PP_HOST_SDS_COMMON_FRAC0_DITHER;
		frac0_stage_en_addr = CS4224_PP_HOST_SDS_COMMON_FRAC0_STAGE_EN;
		frac0_power_down_addr =
			CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB;
		mseq_options_addr = CS4224_PP_HOST_SDS_DSP_MSEQ_OPTIONS_SHADOW;
		mseq_id = CS4224_DPLX_HOST_MSEQ;
	}

	cs4224_lock(slice);

	/* find out if microsequencer is running */
	status |= cs4224_reg_get_channel(slice, mseq_options_addr,
						&mseq_options);

	/* if microsequencer is running */
	if (0 == (0x0008 & mseq_options))
		/* stall the microsequencer */
		status |= cs4224_mseq_stall(slice, mseq_id, CS_TRUE);

	status |= cs4224_reg_get_channel(slice, frac0_power_down_addr,
						&reg_data);
	old_data = reg_data;
	reg_data &= ~0x0010; /* power-up FracN for configuration */
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(slice, frac0_power_down_addr,
							reg_data);

	status |= cs4224_reg_get_channel(slice, frac0_reset_addr, &reg_data);
	old_data = reg_data;
	reg_data &= ~0x0001; /* pull out of reset CDR fracn */
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(slice, frac0_reset_addr,
							reg_data);

	status |= cs4224_reg_get_channel(slice, clkdiv_ctrl_addr, &reg_data);
	old_data = reg_data;
	reg_data &= ~0x000f;
	reg_data |= 0x9; /* Set the RDIV_SEL field to Fractional-N */
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(slice, clkdiv_ctrl_addr,
							reg_data);

	status |= cs4224_reg_get_channel(slice, frac0_width_addr, &reg_data);
	old_data = reg_data;
	reg_data &= ~0x0003; /* Setup to use a 24 bit accumulator */
	reg_data |= CS4224_FRACDIV_ACCUM_WIDTH_24BIT;
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(slice, frac0_width_addr,
							reg_data);

	status |= cs4224_reg_get_channel(slice, frac0_intdiv_addr, &reg_data);
	old_data = reg_data;
	reg_data &= ~0x00ff;
	reg_data |= (divisor & 0xff); /* Configure the denominator value */
	if (old_data != reg_data)
		status |= cs4224_reg_set_channel(slice, frac0_intdiv_addr,
							reg_data);

	/* Configure the numerator value */
	status |= cs4224_reg_set_channel(slice, frac0_numerator0_addr,
						numerator & 0xFFFF);
	status |= cs4224_reg_set_channel(slice, frac0_numerator1_addr,
						(numerator >> 16) & 0xFF);

	/* Configure the stage 1 preload value */
	status |= cs4224_reg_set_channel(slice, frac0_stage1preload0_addr,
						0x5DC6);
	status |= cs4224_reg_set_channel(slice, frac0_stage1preload1_addr,
						0x0034);

	status |= cs4224_reg_set_channel(slice, frac0_dither_addr, 0x0000);

	/* Enable stage1 only */
	status |= cs4224_reg_set_channel(slice, frac0_stage_en_addr, 0x0408);

	status |= cs4224_reg_get_channel(slice, frac0_reset_addr, &reg_data);
	reg_data |= 0x0001; /* reset CDR fracn */
	status |= cs4224_reg_set_channel(slice, frac0_reset_addr, reg_data);
	reg_data &= ~0x0001; /* pull out of reset CDR fracn */
	status |= cs4224_reg_set_channel(slice, frac0_reset_addr, reg_data);

	/* if microsequencer was running */
	if (0 == (0x0008 & mseq_options)) {
		/* un-stall the microsequencer */
		status |= cs4224_mseq_stall(slice, mseq_id, CS_FALSE);
	}

	cs4224_unlock(slice);

	return status;
}

int cs4224_update_cdr_fracn(unsigned int slice, struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	if (rules->fracdiv.enable) {
		/* enable the fractional divider */

		if (cs4224_is_hw_duplex(slice) ||
		    cs4224_line_rx_to_host_tx_dir(slice)) {
			status |= cs4224_fracdiv_cdr_init(
				slice, CS4224_LINE_RX_TO_HOST_TX_DIR,
				rules->fracdiv.divisor,
				rules->fracdiv.numerator);
		}

		if (cs4224_is_hw_duplex(slice) ||
		    !cs4224_line_rx_to_host_tx_dir(slice)) {
			status |= cs4224_fracdiv_cdr_init(
				slice, CS4224_HOST_RX_TO_LINE_TX_DIR,
				rules->fracdiv.divisor,
				rules->fracdiv.numerator);
		}
	}
	/* clkdiv rdiv will control the use of the fracn, so we never have to
	 * reset the fracn settings as long as the clkdiv is re-programmed
	 */

	return status;
}

int cs4224_restore_powered_down_regs(unsigned int slice)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_STX0_MISC,
			CS4224_PP_HOST_SDS_COMMON_STX0_MISC_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1,
			CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB,
			CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG,
			CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB_dft);
	}
	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_STX0_MISC,
			CS4224_PP_LINE_SDS_COMMON_STX0_MISC_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG1,
			CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG1_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DFE_CONFIG,
			CS4224_PP_HOST_SDS_COMMON_SRX0_DFE_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_LSB,
			CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_LSB_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_MSB,
			CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_MSB_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_RX_CONFIG,
			CS4224_PP_HOST_SDS_COMMON_SRX0_RX_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_VCO_CONFIG,
			CS4224_PP_HOST_SDS_COMMON_SRX0_VCO_CONFIG_dft);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB_dft);
	}
	return status;
}

int cs4224_slice_power_down(unsigned int slice)
{
	int status = CS_OK;

	if ((cs4224_is_hw_simplex(slice) &&
	     cs4224_line_rx_to_host_tx_dir(slice)) ||
	     (cs4224_is_hw_duplex(slice))) {
		/* stall the microsequencer */
		status |= cs4224_mseq_stall(
				slice, CS4224_DPLX_LINE_MSEQ, CS_TRUE);

		/* STX_PD=1, STX_PD_MUX=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_STX0_MISC, 0x8033);
		/* SRX_AGC_ENB_LIMAMP=0 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1,
			0x0004);
		/* SRX_DFE_BYPASS_EN=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG,
			0x0441);
		/* SRX_DAC_ENB<19:0>=0xfffff */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB,
			0xffff);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB,
			0x000f);
		/* SRX_AMP_PD=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG,
			0x0044);
		/* SRX_VCO_EN=0, SRX_VCO_PDMUX_EN=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG,
			0x0705);
		/* SRX_CLKDIV_PD=1, SRX_PD_DEMUX=7 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			0x03ff);
	}
	if ((cs4224_is_hw_simplex(slice) &&
	     !cs4224_line_rx_to_host_tx_dir(slice)) ||
	     (cs4224_is_hw_duplex(slice))) {
		/* stall the microsequencer */
		status |= cs4224_mseq_stall(
				slice, CS4224_DPLX_HOST_MSEQ, CS_TRUE);

		/* STX_PD=1, STX_PD_MUX=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_STX0_MISC, 0x8033);
		/* SRX_AGC_ENB_LIMAMP=0 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_AGC_CONFIG1,
			0x0004);
		/* SRX_DFE_BYPASS_EN=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DFE_CONFIG,
			0x0441);
		/* SRX_DAC_ENB<19:0>=0xfffff */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_LSB,
			0xffff);
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_DAC_ENB_MSB,
			0x000f);
		/* SRX_AMP_PD=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_RX_CONFIG,
			0x0044);
		/* SRX_VCO_EN=0, SRX_VCO_PDMUX_EN=1 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_SRX0_VCO_CONFIG,
			0x0705);
		/* SRX_CLKDIV_PD=1, SRX_PD_DEMUX=7 */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			0x03ff);
	}

	return status;
}

int cs4224_apply_workarounds_power_down(unsigned int die)
{
	int status = CS_OK;
	unsigned short data;

	cs4224_lock(die);

	/* get scratch 6 value from die to determine if we have already applied
	 * workarounds
	 */
	status |= cs4224_reg_get(die, CS4224_GLOBAL_SCRATCH6, &data);
	if (CS_IF_SET(data, CS_BIT0))
		/* workarounds already applied, bail out. */
		return CS_OK;

	/* make sure the mseqs are stalled, can't use broadcast otherwise! */
	status |= cs4224_reg_get(
		die, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS_SHADOW, &data);
	if (CS_IF_CLR(data, CS_BIT3)) {
		CS_TRACE(("ERROR: MSEQs on die 0x%x are not stalled.\n"
			  "This method is being used incorrectly.\n", die));
		return CS_ERROR;
	}

	/* Apply workarounds to all port-pairs on the die, Bugzilla #37471 */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x80);
	{
		/* power-down all slices as per Bug #38832 */
		/* STX_PD=1, STX_PD_MUX=1 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_STX0_MISC, 0x8033);
		/* SRX_AGC_ENB_LIMAMP=0 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_AGC_CONFIG1,
			0x0004);
		/* SRX_DFE_BYPASS_EN=1 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_DFE_CONFIG, 0x0441);
		/* SRX_DAC_ENB<19:0>=0xfffff */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB,
			0xffff);
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB,
			0x000f);
		/* SRX_AMP_PD=1 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG, 0x0044);
		/* SRX_VCO_EN=0, SRX_VCO_PDMUX_EN=1 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG, 0x0705);
		/* SRX_CLKDIV_PD=1, SRX_PD_DEMUX=7 */
		status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
			0x03ff);
	}
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x0);

	/* Mark all of the workarounds as complete */
	status |= cs4224_reg_get(die, CS4224_GLOBAL_SCRATCH6, &data);
	status |= cs4224_reg_set(die, CS4224_GLOBAL_SCRATCH6,
					CS_SET(data, CS_BIT0));

	cs4224_unlock(die);

	return status;
}

int cs4224_slice_soft_reset_intf(unsigned int slice,
				 enum e_cs4224_cfg_sides_t intf)
{
	int status = CS_OK;
	int tmp_status = CS_OK;
	unsigned short data = 0;
	unsigned short reg_offset = 0;
	enum e_cs4224_mseq_id mseq_id = CS4224_DPLX_LINE_MSEQ;
	enum e_cs4224_mseq_id other_mseq_id = CS4224_DPLX_HOST_MSEQ;
	unsigned int mate_slice = 0xFF;
	unsigned char stalled = CS_FALSE;
	unsigned char power_savings = CS_FALSE;
	unsigned short other_power_down = 0x0;

	if (intf == CS4224_CFG_LINE_SIDE) {
		reg_offset = 0;
		mseq_id = CS4224_DPLX_LINE_MSEQ;
		other_mseq_id = CS4224_DPLX_HOST_MSEQ;
	} else if (intf == CS4224_CFG_HOST_SIDE) {
		reg_offset = CS4224_LINE_TO_HOST_OFFSET;
		mseq_id = CS4224_DPLX_HOST_MSEQ;
		other_mseq_id = CS4224_DPLX_LINE_MSEQ;
	} else {
		CS_TRACE(("ERROR: intf must be line/host, not splx. intf=%d\n",
			  intf));
		return CS_ERROR;
	}

	/* make sure mseq is stalled on this interface, if not, then stall and
	 * don't unstall after the reset
	 */
	status |= cs4224_query_mseq_is_stalled(slice, mseq_id, &stalled);
	if (!stalled)
		status |= cs4224_mseq_stall(slice, mseq_id, CS_TRUE);

	/* we need to control the power_down on both line and host interfaces.
	 * In simplex, the other interface's mseq may be running, so turn off
	 * power savings so we can mess around with power_down without affecting
	 * traffic
	 */
	tmp_status |= cs4224_simplex_mate_slice(slice, &mate_slice);
	if ((mate_slice & 0xFF) == 0xFF) {
		/* doesn't have a mate slice, which means the other slice is not
		 * configured nor running
		 */
		power_savings = CS_FALSE;
	} else if (tmp_status != CS_OK) {
		/* some other kind of error, not good */
		CS_TRACE(("ERROR getting mate slice\n"));
		status |= tmp_status;
	} else {
		/* we do have a mate slice, or on duplex it's the same slice,
		 * either way check power savings
		 */
		status |= cs4224_query_mseq_power_savings(mate_slice,
							  other_mseq_id,
							  &power_savings);
		if (power_savings)
			status |= cs4224_mseq_enable_power_savings(
				mate_slice, other_mseq_id, CS_FALSE);
	}

	if (status != CS_OK)
		return status;

	cs4224_lock(slice);

	/* clock and power up everything */
	if (intf == CS4224_CFG_LINE_SIDE) {
		/* only touch Rx stuff */
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, &data);
		data |= 0x00ff;
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, data);

		/* host-side reg doesn't have Tx settings */
	} else {
		/* only touch Rx stuff */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_HOSTMISC_CLKEN, 0x0041);

		/* only touch Tx stuff on the other side */
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, &data);
		data |= 0xff00;
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, data);
	}

	/* save the other-side's power down so we can restore it later */
	status |= cs4224_reg_get_channel(
		slice, CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB - reg_offset,
		&other_power_down);

	/* stagger power up */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
		0x001F);
	/* stagger power up only if we won't kill anything, this works because
	 * if it's already on (cleared to 0) then this won't turn it off trying
	 * to write 0x1F into the reg
	 * NOTE: There is an issue (maybe?) in the PRBS checker that when you
	 * clear bit6 you will get a momenary loss of PRBS sync this does not
	 * effect traffic through the device, only the PRBS checker
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB - reg_offset,
		(other_power_down & 0x001F));
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
		0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB - reg_offset,
		0x0000);

	/* reset most of the Rx settings */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_LINEMISC_SOFT_RESET + reg_offset, 0x0031);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_LINEMISC_MPIF_RESET_DOTREG + reg_offset,
		0x0003);

	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_LINEMISC_SOFT_RESET + reg_offset, 0x0000);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_LINEMISC_MPIF_RESET_DOTREG + reg_offset,
		0x0000);

	/* some stuff isn't hit by those resets, so reset those manually */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXLOCKD0_CONTROL + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_RXLOCKD0_CONTROL_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RX0_Config + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_RX0_Config_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_RXELST0_Control + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_RXELST0_Control_dft);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKOUT_CTRL + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CLKOUT_CTRL_dft);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG_EYEMON + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG_EYEMON_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_MONCTRL + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_MONCTRL_dft);

	/* RX_CONFIG on the RX side has a Tx side Tx bit for some reason.
	 * Yes this register is correct, so the LINE_SDRX0_RX_CONFIG
	 * will contain the analog host Tx polarity inversion bit
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CONFIG_dft);

	/* Reset some Tx registers that weren't hit by the Rx soft reset
	 * These are mostly just registers not touched by
	 * cs4224_slice_enter_operational_state
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_COMMON_TX0_Config - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_TX0_Config_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_COMMON_TXELST0_Control - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_TXELST0_Control_dft);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_CONFIG - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_CONFIG_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_TUNE - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVER_TUNE_dft);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVERCML_TUNE - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_STX0_DRIVERCML_TUNE_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_HOST_SDS_COMMON_STX0_MISC - reg_offset,
		CS4224_PP_HOST_SDS_COMMON_STX0_MISC_dft);

	/* Reset some stuff that's normally reset in
	 * cs4224_restore_powered_down_regs
	 */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_LSB_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_DAC_ENB_MSB_dft);
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG + reg_offset,
		CS4224_PP_LINE_SDS_COMMON_SRX0_VCO_CONFIG_dft);

	/* Unfortunately the soft reset wipes out the per-port workarounds on
	 * this slice, so re-apply them
	 * NOTE: Sync this block of writes with those in the
	 * cs4224_apply_workarounds method
	 */

	/* Adjust the LC-VCO bias current because the defaults are wrong.
	 * See bugzilla #36619,39270 for details
	 */
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_VCOBIAS + reg_offset,
		   0x0907);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL00 + reg_offset,
		   0x3222);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL01 + reg_offset,
		   0x4444);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL02 + reg_offset,
		   0x5555);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL03 + reg_offset,
		   0x6666);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL10 + reg_offset,
		   0x7654);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL11 + reg_offset,
		   0x8877);
	status |= cs4224_reg_set_channel(
		   slice,
		   CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL12 + reg_offset,
		   0xa999);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL13 + reg_offset,
		0xccba);
	status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_PHSEL + reg_offset,
			0x001e);

	/* clear bit 13 (bug 37471) */
	data = CS_CLR(CS4224_PP_LINE_SDS_COMMON_SRX0_RX_SPARE_dft, CS_BIT13);
	status |= cs4224_reg_set_channel(
		slice,
		CS4224_PP_LINE_SDS_COMMON_SRX0_RX_SPARE + reg_offset,
		data);

	/* clear common_tx_sr reset */
	status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_SOFT_RESET + reg_offset,
			0x0000);

	/* force driver charge pump high to eliminate cross-talk */
	status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA + reg_offset,
			0x80dd);

	/* DATAPATH_CTRL is different between line and host, but the
	 * host bits that are different are reserved so it doesn't
	 * matter if we try to write them
	 */
	data = CS_SET(CS4224_PP_LINE_LINEMISC_DATAPATH_CTRL_dft, CS_BIT1);
	status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_LINE_LINEMISC_DATAPATH_CTRL + reg_offset,
			data);

	/* Reset power-down and clken before proceeding */
	if (intf == CS4224_CFG_LINE_SIDE) {
		/* reset Rx stuff */
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, &data);
		data &= 0xff00;
		data |= 0x0060;
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, data);
		/* no Tx stuff in host reg */
	} else {
		/* reset Rx stuff */
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_HOSTMISC_CLKEN, 0x0000);

		/* reset Tx stuff on other side */
		status |= cs4224_reg_get_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, &data);
		data &= 0x00ff;
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_LINEMISC_CLKEN, data);
	}

	/* set power down to default on this interface */
	status |= cs4224_reg_set_channel(
		slice, CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB + reg_offset,
		CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB_dft);

	if (!power_savings) {
		/* restore power down on the other interface */
		status |= cs4224_reg_set_channel(
			slice,
			CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB - reg_offset,
			other_power_down);
	}

	cs4224_unlock(slice);

	/* re-enable power savings on the other interface if needed */
	if (power_savings) {
		status |= cs4224_mseq_enable_power_savings(mate_slice,
								other_mseq_id,
								CS_TRUE);
	}

	return status;
}

int cs4224_slice_soft_reset(unsigned int slice)
{
	int status = CS_OK;

	if (cs4224_is_hw_duplex(slice) || cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_slice_soft_reset_intf(slice,
							CS4224_CFG_LINE_SIDE);
	}
	if (cs4224_is_hw_duplex(slice) || !cs4224_line_rx_to_host_tx_dir(
		slice)) {
		status |= cs4224_slice_soft_reset_intf(slice,
							CS4224_CFG_HOST_SIDE);
	}

	return status;
}

int cs4224_apply_workarounds(unsigned int die)
{
	int status = CS_OK;
	unsigned short data;

	cs4224_lock(die);

	/* scratch6 value determines if we have already applied workarounds */
	status |= cs4224_reg_get(die, CS4224_GLOBAL_SCRATCH6, &data);
	if (CS_IF_SET(data, CS_BIT0)) {
		/* workarounds already applied, bail out. */
		return CS_OK;
	}

	/* make sure the mseqs are stalled, can't use broadcast otherwise! */
	status |= cs4224_reg_get(
		die, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS_SHADOW, &data);
	if (CS_IF_CLR(data, CS_BIT3)) {
		CS_TRACE(("ERROR: MSEQs on die 0x%x are not stalled.\n"
			  "This method is being used incorrectly.\n", die));
		return CS_ERROR;
	}

	/* At this point, the device is essentially right out of reset */

	/* Change the temperature and voltage monitors to poll */
	status |= cs4224_enable_monitor_sense_points(die, CS_FALSE);

	/* Increase MDIO timeout as per Bugzilla 35875 */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_MDIO_CONFIG,
				 CS4224_GLOBAL_MDIO_CONFIG_pref);

	/* load the 0.9v monitor calibration constant */
	status |= cs4224_reg_get(die, CS4224_EFUSE_PDF_MON_GAIN_DATA, &data);
	status |= cs4224_reg_set(die, CS4224_MONITOR_CAL_OVERRIDE, data);

	/* load the 1.8v monitor calibration constant */
	status |= cs4224_reg_get(die, CS4224_EFUSE_PDF_MON_LUT15, &data);
	status |= cs4224_reg_set(die, CS4224_MONITOR_LUT_VALUE0, data);

	/* make use of the temperature and 0.9v voltage calibration constants */
	status |= cs4224_reg_set(die, CS4224_MONITOR_CAL_CONST_OVERRIDE_ENA,
					0x0004);
	status |= cs4224_reg_set(die, CS4224_MONITOR_CAL_CONST_APPLY, 0x0005);

	/* make use of the 1.8v voltage calibration constant */
	status |= cs4224_reg_set(die, CS4224_MONITOR_LUT_RANGE0, 0xffff);
	status |= cs4224_reg_set(die, CS4224_MONITOR_LUT_SELECT, 0x0000);
	status |= cs4224_reg_set(die, CS4224_MONITOR_LUT_LOCAL_SELECT, 0x0040);
	status |= cs4224_reg_set(die, CS4224_MONITOR_LUT_APPLY, 0x0040);

	/* Apply workarounds to all port-pairs (PP) on the die */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x80);

	/* Adjust the LC-VCO bias current because the defaults are wrong.
	 * See bugzilla #36619,39270 for details
	 * NOTE: Sync these values with those in the cs4224_slice_soft_reset
	 * method
	 */
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_VCOBIAS,
			0x0907);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL00,
			0x3222);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL01,
			0x4444);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL02,
			0x5555);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL03,
			0x6666);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL10,
			0x7654);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL11,
			0x8877);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL12,
			0xa999);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_RXVCO0_LC_BIASVAL13,
			0xccba);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_DSP_MSEQ_CAL_RX_PHSEL, 0x001e);

	/* clear bit 13 (bug 37471) */
	data = CS_CLR(CS4224_PP_LINE_SDS_COMMON_SRX0_RX_SPARE_dft, CS_BIT13);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_SPARE, data);

	/* clear common_tx_sr reset */
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_SOFT_RESET, 0x0);

	/* force driver charge pump high to eliminate cross-talk */
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA, 0x80dd);

	/* DATAPATH_CTRL is different between line and host, but the
	 * host bits that are different are reserved so it doesn't
	 * matter if we try to write them
	 */
	data = CS_SET(CS4224_PP_LINE_LINEMISC_DATAPATH_CTRL_dft,
		      CS_BIT1);
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_LINEMISC_DATAPATH_CTRL, data);

	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x0);

	cs4224_unlock(die);

	return status;
}

int cs4224_apply_post_ucode_dwld_workarounds(unsigned int die)
{
	int status = CS_OK;
	unsigned short data;

	cs4224_lock(die);

	/* make sure the mseqs are stalled, can't use broadcast otherwise! */
	status |= cs4224_reg_get(
		die, CS4224_PP_LINE_SDS_DSP_MSEQ_OPTIONS_SHADOW, &data);
	if (CS_IF_CLR(data, CS_BIT3)) {
		CS_TRACE(("ERROR: MSEQs on die 0x%x are not stalled.\n"
			  "This method is being used incorrectly.\n", die));
		return CS_ERROR;
	}

	/* Apply workarounds to all port-pairs on the die, Bugzilla #37471 */
	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x80);

	/* force driver charge pump high (to rail) to elimitnate cross-talk */
	status |= cs4224_reg_set(
			die, CS4224_PP_LINE_SDS_COMMON_SRX0_RX_CPA, 0x80dd);

	status |= cs4224_reg_set(die, CS4224_GLOBAL_BROADCAST, 0x0);

	cs4224_unlock(die);

	return status;
}

int cs4224_dump_debug_info(unsigned int slice, struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	const char *app = cs4224_translate_app_mode(rules->application);
	const char *edc_mode;
	const char *edc_mode2;

	char data_rate_str[15] = "";

	if (rules->application == CS4224_TARGET_APPLICATION_KRAN)
		CS_SNPRINTF((data_rate_str,
			     ARRAY_SIZE(data_rate_str),
			     "0x%04x",
			     rules->kran.data_rates));
	else
		CS_SNPRINTF((data_rate_str,
			     ARRAY_SIZE(data_rate_str),
			     "%uMbps",
			     (cs4224_query_data_rate(rules) / 1000)));

	if (cs4224_is_hw_simplex(slice)) {
		edc_mode =
			cs4224_translate_edc_mode(rules->rx_if.splx_edc_mode);
		edc_mode2 = edc_mode;
	} else {
		edc_mode = cs4224_translate_edc_mode(
				rules->rx_if.dplx_line_edc_mode);
		edc_mode2 = cs4224_translate_edc_mode(
				rules->rx_if.dplx_host_edc_mode);
	}

	switch (hw_id) {
	case CS4224_HW_CS4223: /*  4 port duplex   */
		CS_PRINTF(("\nK2 configured for Duplex CS4223,Max ports = %d\n",
			   cs4224_max_num_ports));
		CS_PRINTF(("slice = %x, die = %x, PP offset = 0x%05x\n",
			   slice & 0xff,
			   cs4224_get_die_from_slice(slice),
			   cs4224_adj_pp(slice, 0)));
		CS_PRINTF(("application = %s, data rate = %s, ",
			   app, data_rate_str));
		CS_PRINTF(("line edc_mode = %s, host edc_mode = %s\n",
			   edc_mode, edc_mode2));

		break;

	case CS4224_HW_CS4224: /* 16 port simplex  */
	case CS4224_HW_CS4343: /*  8 port duplex   */
	case CS4224_HW_CS4221: /* 10 port simplex  */
	case CS4224_HW_CS4227: /*  2 port duplex   */
	case CS4224_HW_CS4210: /* 16 port simplex  */
	case CS4224_HW_CS4341: /*  8 port duplex   */
		status = CS_ERROR;
		printf("SKU not yet supported !\n");
		break;
	default:
		CS_PRINTF(("\nERROR: K2 Efuse SKU not configured\n\n"));
		status = CS_ERROR;

		break;
	}

	return status;
}

unsigned char cs4224_is_eeprom_finished(unsigned int die)
{
	unsigned short efuse_general_status;
	unsigned short global_pin_status;
	unsigned short eeprom_loader_status;

	cs4224_reg_get(die, CS4224_EFUSE_GENERAL_STATUS, &efuse_general_status);
	if (efuse_general_status != 0x0001) {
		/* EFUSE has not finished loading */
		return CS_FALSE;
	}

	cs4224_reg_get(die, CS4224_GLOBAL_PIN_STATUS, &global_pin_status);
	/* 0xf00d means the eeprom loader is still running
	 * 0xffff means the refclk isn't stable
	 */
	if (global_pin_status == 0xf00d ||
	    global_pin_status == 0xffff ||
	    global_pin_status == 0xbada) {
		return CS_FALSE;
	} else if ((global_pin_status & 0x4000) == 0)
		/* EEPROM not present */
		return CS_TRUE;

	cs4224_reg_get(die, CS4224_EEPROM_LOADER_STATUS, &eeprom_loader_status);
	if (eeprom_loader_status == 0xbada)
		return CS_FALSE;

	if ((eeprom_loader_status & 0x0007) != 0)
		/* EEPROM load abort, load failed, or load done */
		return CS_TRUE;

	return CS_FALSE;
}

int cs4224_wait_for_eeprom_finished(
	unsigned int die, unsigned int max_iterations,
	unsigned int ms_delay_between_iterations)
{
	int status = CS_OK;
	unsigned char finished = CS_FALSE;
	unsigned int attempt = 0;

	do {
		finished = cs4224_is_eeprom_finished(die);

		if (!finished) {
			if (ms_delay_between_iterations == 0)
				CS_MDELAY(1);
			else
				CS_MDELAY(ms_delay_between_iterations);
		}
	} while ((attempt++ < max_iterations) && (finished == CS_FALSE));

	if (finished == CS_TRUE) {
		status = CS_OK;
	} else {
		CS_TRACE(("ERROR: EEPROM not loading, die=%x\n", die));
		status = CS_ERROR;
	}

	return status;
}

int cs4224_update_scratch_regs(unsigned int slice)
{
	int status = CS_OK;
	unsigned short reg_addr = 0x0;
	unsigned short inc = 0x0;
	unsigned short counter = 0x0;
	unsigned char local_slice = slice & 0xff;

	/* which scratch you use is based off of the PP,
	 * ex PP2 is GLOBAL_SCRATCH2
	 */
	reg_addr = CS4224_GLOBAL_SCRATCH0 + (cs4224_adj_pp(slice, 0x0) >> 12);

	/* increment is based off of the die, ex slice 2 is die 1 which is the
	 * upper bits of GLOBAL_SCRATCH2, so inc will be 0x0100.
	 * See global_scratch_allocation.txt for more info
	 */
	inc = (cs4224_get_die_from_slice(local_slice) == 1) ? 0x0100 : 0x0001;

	cs4224_lock(slice);

	status |= cs4224_reg_get_channel(slice, reg_addr, &counter);
	counter += inc;
	status |= cs4224_reg_set_channel(slice, reg_addr, counter);

	cs4224_unlock(slice);

	return status;
}

int cs4224_check_rules(unsigned int slice, struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned char bad_state = CS_FALSE;
	unsigned short valid_edc_modes = 0;

	switch (rules->application) {
	case CS4224_TARGET_APPLICATION_OC12:
	case CS4224_TARGET_APPLICATION_1G:
	case CS4224_TARGET_APPLICATION_1G_FC:
	case CS4224_TARGET_APPLICATION_2G_FC:
	case CS4224_TARGET_APPLICATION_4G_FC:
	case CS4224_TARGET_APPLICATION_5G:
	case CS4224_TARGET_APPLICATION_7p5G:
	case CS4224_TARGET_APPLICATION_8G_FC:
	case CS4224_TARGET_APPLICATION_16G_FC:
	case CS4224_TARGET_APPLICATION_15G:
	case CS4224_TARGET_APPLICATION_10G_KR:
	case CS4224_TARGET_APPLICATION_40G_KR:
	case CS4224_TARGET_APPLICATION_KRAN:
	case CS4224_TARGET_APPLICATION_FCAN:
	case CS4224_TARGET_APPLICATION_8G:
	case CS4224_TARGET_APPLICATION_8p5G:
	case CS4224_TARGET_APPLICATION_10G_FC:
		printf("Target application mode not yet supported!\n");
		status = CS_ERROR;
		break;
	case CS4224_TARGET_APPLICATION_10G: {
		valid_edc_modes =
			CS_HSIO_EDC_MODE_SR |
			CS_HSIO_EDC_MODE_10G_BP |
			CS_HSIO_EDC_MODE_CX1 |
			CS_HSIO_EDC_MODE_SMLRM |
			CS_HSIO_EDC_MODE_DWDM |
			CS_HSIO_EDC_MODE_ZR;

		if (cs4224_is_hw_simplex(slice))
			bad_state =
				!(rules->rx_if.splx_edc_mode & valid_edc_modes);
		else
			bad_state =
				!((rules->rx_if.dplx_host_edc_mode &
				   valid_edc_modes) &&
				   (rules->rx_if.dplx_line_edc_mode &
				    valid_edc_modes));

		if (bad_state) {
			CS_TRACE(("ERROR: target app only supports EDC modes"));
			CS_TRACE(("SR, CX1, 10G_BP, DWDM, and ZR\n"));
			status |= CS_ERROR;
		}
		break;
	}
	}

	/* validate simplex/duplex rules */
	if (cs4224_is_hw_simplex(slice)) {
		/* These traceloss settings are for the driver only */
		bad_state =
			rules->rx_if.splx_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_CR4 ||
			rules->rx_if.splx_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_27dB ||
			rules->rx_if.splx_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_15dB ||
			rules->rx_if.splx_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_LAB;
		if (bad_state) {
			CS_TRACE(("ERROR: Traceloss settings of 15/27dB or "));
			CS_TRACE(("CR4/LAB are for the tx_if.*_driver."));
			CS_TRACE(("traceloss setting only.\n"));
			status |= CS_ERROR;
		}

		/* FEC available for simplex parts only */
		bad_state = rules->enable_fec;
		if (bad_state) {
			CS_TRACE(("ERROR: FEC not supported on simplex\n"));
			status |= CS_ERROR;
		}
	} else {
		/* These traceloss settings are for the driver only */
		bad_state =
			rules->rx_if.dplx_host_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_15dB ||
			rules->rx_if.dplx_host_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_27dB ||
			rules->rx_if.dplx_host_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_CR4 ||
			rules->rx_if.dplx_host_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_LAB ||
			rules->rx_if.dplx_line_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_15dB ||
			rules->rx_if.dplx_line_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_27dB ||
			rules->rx_if.dplx_line_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_CR4 ||
			rules->rx_if.dplx_line_eq.traceloss ==
				CS_HSIO_TRACE_LOSS_LAB;
		if (bad_state) {
			CS_TRACE(("ERROR: Traceloss settings of 15/27dB or "));
			CS_TRACE(("CR4/LAB are for the tx_if.*_driver."));
			CS_TRACE(("traceloss setting only.\n"));
			status |= CS_ERROR;
		}
	}

	/* validate the refclk is correct */
	if ((rules->ref_clk_rate > 180000) || (rules->ref_clk_rate < 50000)) {
		/* ref clk can only be 50-180MHz */
		CS_TRACE(("ERROR: ref_clk_rate is the internal reference "));
		CS_TRACE(("clock rate, which can only be between 50MHz and "));
		CS_TRACE(("180MHz\nIf you are using the REFDIV2_ENB pin then"));
		CS_TRACE((" div your refclk by 2 for the ref_clk_rate val\n"));
		status |= CS_ERROR;
	}

	if (status != CS_OK)
		CS_TRACE(("ERROR: bad rules config on slice %x\n", slice));

	return status;
}

int cs4224_manage_ucode_download(unsigned int slice,
				 struct cs4224_rules_t *rules)
{
	int status = CS_OK;

	unsigned short timestamp[2] = { 0xffff, 0xffff };
	unsigned short yyyy, mmdd, hhmm;
	unsigned int die, other_die;
	unsigned char die_broadcast = CS_FALSE;
	unsigned short version_kr, version_fc;
	unsigned short *ucode_image = NULL;
	unsigned int size_image = 0;
	unsigned char stalled = CS_FALSE;
	enum e_cs4224_ucode_image required_image = CS4224_UCODE_IMAGE_ANY;
	enum e_cs4224_ucode_image loaded_image = CS4224_UCODE_IMAGE_NONE;

	/* die being programmed */
	die = cs4224_get_die_from_slice(slice);
	/* the other die */
	other_die = die ^ 0x1;

	cs4224_lock(slice);

	/* Check to see if the microcode has already been programmed */
	status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_TIMESTAMP0,
					&timestamp[0]);
	status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_VERSION_KR,
					&version_kr);
	status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_VERSION_FC,
					&version_fc);

	if (cs4224_max_num_dies == 2) {
		status |= cs4224_reg_get(other_die,
					 CS4224_GLOBAL_UCODE_TIMESTAMP0,
					 &timestamp[1]);
	}
	cs4224_unlock(slice);

	/* decide if we should broadcast ucode to both dies */
	if (rules->enable_die_broadcast &&
	    (cs4224_max_num_dies == 2) &&
	    (timestamp[0] == 0) &&
	    (timestamp[1] == 0))
		die_broadcast = CS_TRUE;

	/* apply die workarounds, if necessary */
	status |= cs4224_apply_workarounds(die);
	if (die_broadcast)
		status |= cs4224_apply_workarounds(other_die);

	if (status != CS_OK)
		return status;

	/* determine what our microcode requirements are */
	if (rules->application == CS4224_TARGET_APPLICATION_KRAN)
		required_image = CS4224_UCODE_IMAGE_KR;
	else if (rules->application == CS4224_TARGET_APPLICATION_FCAN)
		required_image = CS4224_UCODE_IMAGE_FC;

	/* determine what is currently loaded */
	if (version_kr != 0)
		loaded_image = CS4224_UCODE_IMAGE_KR;
	else if (version_fc != 0)
		loaded_image = CS4224_UCODE_IMAGE_FC;

	/* if the currently loaded ucode is not what we need */
	if ((loaded_image & required_image) == 0) {
		if (required_image != CS4224_UCODE_IMAGE_FC) {
			ucode_image = cs4224_kr_ucode_image;
			size_image = sizeof(cs4224_kr_ucode_image)
				/ sizeof(unsigned short);
		}

		if (die_broadcast) {
			CS_PRINTF(("Broadcasting image\n"));
			/* broadcast to both dies */
			status |= cs4224_ucode_data_prgm_image_broadcast(
				slice, ucode_image, size_image);
		} else { /* unicast */
			CS_PRINTF(("Unicasting image\n"));
			/* only broadcast to single die */
			status |= cs4224_ucode_data_prgm_image(slice,
								ucode_image,
								size_image);
		}

		/* check that the ucode programmed correctly */
		if (status != CS_OK) {
			CS_TRACE(("ERROR downloading microcode, exiting\n"));
			return status;
		}

		/* check if post-setup in the ucode image unstalled the mseqs
		 * (it shouldn't). Note you can check any mseq on this die,
		 * since they should all be configured equally
		 */
		status |= cs4224_query_mseq_is_stalled(slice,
							CS4224_DPLX_LINE_MSEQ,
							&stalled);
		if (!stalled) {
			/* Iterate over all the slices on this die */
			unsigned int first_slice = 0;
			unsigned int last_slice = 0;
			unsigned int mid_slice =
					CS4224_MAX_NUM_SLICES(slice) / 2;
			unsigned int upper_bits = slice & 0xffffff00;
			unsigned char i = 0;
			unsigned int slice_i = 0;

			if (die_broadcast || CS4224_MAX_NUM_DIES(slice) == 1) {
				/* reset all the slices */
				first_slice = 0;
				last_slice = CS4224_MAX_NUM_SLICES(slice) - 1;
			} else {
				/* only reset mseqs on this die */
				if (slice < mid_slice) {
					first_slice = 0;
					last_slice = mid_slice - 1;
				} else {
					first_slice = mid_slice;
					last_slice = CS4224_MAX_NUM_SLICES(
						slice)
							- 1;
				}
			}

			for (i = first_slice; i <= last_slice; i++) {
				slice_i = upper_bits | i;

				/* stall the microsequencer(s) again */
				if (cs4224_is_hw_simplex(slice_i)) {
					status |= cs4224_mseq_stall(
						slice_i, CS4224_SPLX_MSEQ,
						CS_TRUE);
				} else {
					status |= cs4224_mseq_stall(
						slice_i, CS4224_DPLX_LINE_MSEQ,
						CS_TRUE);
					status |= cs4224_mseq_stall(
						slice_i, CS4224_DPLX_HOST_MSEQ,
						CS_TRUE);
				}
			}
		} /* stall mseq block */

		/* re-apply some die workarounds that got removed in the ucode
		 * download
		 */
		status |= cs4224_apply_post_ucode_dwld_workarounds(die);
		if (die_broadcast)
			status |= cs4224_apply_post_ucode_dwld_workarounds(
				other_die);
	} /* pgm ucode block */

	if (rules->show_debug_info) {
		/* Get the timestamp of the loaded microcode */
		status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_TIMESTAMP1,
						&yyyy);
		status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_TIMESTAMP0,
						&mmdd);
		status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_TIMESTAMP2,
						&hhmm);
		status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_VERSION_KR,
						&version_kr);
		status |= cs4224_reg_get(die, CS4224_GLOBAL_UCODE_VERSION_FC,
						&version_fc);
		/* Note that for now both KR and FC versions are set to 1 */
		if ((version_kr != 0) && (version_fc != 0)) {
			CS_TRACE(("ERROR: uCode version is both KR + FC, KR("));
			CS_TRACE(("v.%04x),FC(v.%04x) built %04x/%04x/%04x\n",
				  version_kr, version_fc, yyyy, mmdd, hhmm));
			return CS_ERROR;
		}
		if (version_kr != 0) {
			CS_PRINTF(("uCode on die %x: KR(v.%04x), built on",
				   die, version_kr));
			CS_PRINTF((" (yyyy/mmdd/hhmm) =  %04x/%04x/%04x\n",
				   yyyy, mmdd, hhmm));
		}
		if (version_fc != 0) {
			CS_PRINTF(("uCode on die %x: FC(v.%04x), built on",
				   die, version_fc));
			CS_PRINTF((" (yyyy/mmdd/hhmm) =  %04x/%04x/%04x\n",
				   yyyy, mmdd, hhmm));
		}
		if (cs4224_max_num_dies == 2 && die_broadcast) {
			/* Get the timestamp of the other loaded microcode */
			status |= cs4224_reg_get(
				other_die, CS4224_GLOBAL_UCODE_TIMESTAMP1,
				&yyyy);
			status |= cs4224_reg_get(
				other_die, CS4224_GLOBAL_UCODE_TIMESTAMP0,
				&mmdd);
			status |= cs4224_reg_get(
				other_die, CS4224_GLOBAL_UCODE_TIMESTAMP2,
				&hhmm);
			status |= cs4224_reg_get(
				other_die, CS4224_GLOBAL_UCODE_VERSION_KR,
				&version_kr);
			status |= cs4224_reg_get(
				other_die, CS4224_GLOBAL_UCODE_VERSION_FC,
				&version_fc);
			if (version_kr != 0) {
				CS_PRINTF(("uCode on die %x: KR(v.%04x) built",
					   die, version_kr));
				CS_PRINTF((" (yyyy/mmdd/hhmm)=%04x/%04x/%04x\n",
					   yyyy, mmdd, hhmm));
			}
			if (version_fc != 0) {
				CS_PRINTF(("uCode on die %x: FC(v.%04x) built",
					   die, version_fc));
				CS_PRINTF((" (yyyy/mmdd/hhmm)=%04x/%04x/%04x\n",
					   yyyy, mmdd, hhmm));
			}
		}
	}

	/* power down all slices for bug #38832 */
	status |= cs4224_apply_workarounds_power_down(die);
	if (die_broadcast)
		status |= cs4224_apply_workarounds_power_down(other_die);

	return status;
}

int cs4224_slice_enter_operational_state(unsigned int slice,
					 struct cs4224_rules_t *rules)
{
	int status = CS_OK;
	unsigned int die = cs4224_get_die_from_slice(slice);
	unsigned short data = 0;

	cs4224_sku_hw_id = CS4224_HW_UNDEF;

	/* Before we do anything, do some sanity checking of the rules */
	status |= cs4224_check_rules(slice, rules);
	if (status != CS_OK)
		return status;

	/* Before configuring the device, the die MUST have been reset and the
	 * EEPROM must already be loaded (if an EEPROM is available). Otherwise
	 * this method will exit immediately because it's unsafe to configure
	 * the device in this state.
	 */
	status |= cs4224_reg_get(die, CS4224_GLOBAL_SCRATCH6, &data);
	if (CS_IF_CLR(data, CS_BIT1)) {
		CS_TRACE(("ERROR: cs4224_hard_reset hasn't been run on die\n"));
		return CS_ERROR;
	}
	if (!cs4224_is_eeprom_finished(die)) {
		CS_TRACE(("ERROR: EEPROM is still loading; aborting.\n"));
		return CS_ERROR;
	}

	/* Program the API version number */
	status |= cs4224_init_api_version(slice);

	/* Increment the slice configure count */
	status |= cs4224_update_scratch_regs(slice);

	/* Reset any static state stored by the slice */
	cs4224_reset_static_state_for_slice(slice);

	/* stall the microsequencer(s), always */
	if (cs4224_is_hw_simplex(slice)) {
		status |= cs4224_mseq_stall(slice, CS4224_SPLX_MSEQ, CS_TRUE);
	} else {
		status |= cs4224_mseq_stall(
					slice, CS4224_DPLX_LINE_MSEQ, CS_TRUE);
		status |= cs4224_mseq_stall(
					slice, CS4224_DPLX_HOST_MSEQ, CS_TRUE);
	}

	/* Re-assert the squelch after stalling the microsequencer if it is not
	 * already squelched
	 */
	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_HOST_SDS_COMMON_STX0_SQUELCH, 0x0001);
	}
	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_set_channel(
			slice, CS4224_PP_LINE_SDS_COMMON_STX0_SQUELCH, 0x0001);
	}

	if ((cs4224_is_hw_simplex(slice) &&
	     (rules->rx_if.splx_edc_mode == CS_HSIO_EDC_MODE_DISABLED)) ||
	    (cs4224_is_hw_duplex(slice) &&
	     (rules->rx_if.dplx_line_edc_mode == CS_HSIO_EDC_MODE_DISABLED)) ||
	    (cs4224_is_hw_duplex(slice) &&
	     (rules->rx_if.dplx_host_edc_mode == CS_HSIO_EDC_MODE_DISABLED))) {
		/* When using an eeprom image the mseq can set the squelch bit
		 * before we are able to stall it. This will force the bit high
		 * if we're in disabled mode and the mseq won't be unstalled.
		 */
		rules->unsquelch_driver = CS_TRUE;
	}

	/* dump out debug info */
	if (rules->show_debug_info == CS_TRUE)
		status |= cs4224_dump_debug_info(slice, rules);

	/* Apply post-reset configuration and program the microcode */
	status |= cs4224_manage_ucode_download(slice, rules);
	if (status != CS_OK) {
		/* ucode download has a lot of potential to go wrong, exit */
		CS_TRACE(("ERROR: Ucode download on slice %x failed, exiting\n",
			  slice));
		return status;
	}

	/* Apply a soft reset to put the slice in a known state.
	 * This only really helps dynamic reconfig.
	 */
	if (rules->application != CS4224_TARGET_APPLICATION_FCAN)
		status |= cs4224_slice_soft_reset(slice);

	status |= cs4224_config_target_application(slice, rules);

	status |= cs4224_update_clkdiv_ctrl(slice, rules);

	status |= cs4224_init_driver_trace_loss(slice, rules);

	status |= cs4224_init_edc_mode_set(slice, rules);

	status |= cs4224_update_cdr_fracn(slice, rules);

	if (cs4224_is_hw_simplex(slice))
		status |= cs4224_mseq_enable_power_savings(
			slice, CS4224_SPLX_MSEQ, rules->enable_power_savings);
	else {
		if (rules->application != CS4224_TARGET_APPLICATION_FCAN) {
			/* the FC-AN ucode configures the line side DFE mode,
			 * do not muck around with power savings here...
			 */
			status |= cs4224_mseq_enable_power_savings(
				slice, CS4224_DPLX_LINE_MSEQ,
				rules->enable_power_savings);
		}
		status |= cs4224_mseq_enable_power_savings(
			slice, CS4224_DPLX_HOST_MSEQ,
			rules->enable_power_savings);
	}

	status |= cs4224_init_alt_coarse_tuning(slice, rules);
	status |= cs4224_disable_tx_driver_if_req(slice, rules);
	status |= cs4224_init_ac_decoupling_caps(slice, rules);
	status |= cs4224_init_mseq_dyn_reconfig(slice, rules);

	cs4224_lock(slice);

	/* coarse tune VCOs because ref clock is not at 106.25Mhz */

	/* Enable then trigger VCO coarse tuning */
	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_get_channel(slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				&data);
		status |= cs4224_reg_set_channel(slice,
				CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				0x01e7);
		CS_MDELAY(1);

		/* init_vco has built-in delays after coarse tuning */
		status |= cs4224_init_vco(slice,
				CS4224_PP_LINE_SDS_COMMON_RXVCO0_CONTROL);

		if ((data & 0x1c0) != 0x1c0) {
			/* demux enabled, needs staggered power up */
			status |= cs4224_reg_set_channel(slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				    (data | 0x1f));
		}
		status |= cs4224_reg_set_channel(slice,
				    CS4224_PP_LINE_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				    data);
	}

	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_reg_get_channel(slice,
				CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				&data);
		status |= cs4224_reg_set_channel(slice,
				CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				0x01e7);
		CS_MDELAY(1);

		/* init_vco has built-in delays after coarse tuning */
		status |= cs4224_init_vco(slice,
				CS4224_PP_HOST_SDS_COMMON_RXVCO0_CONTROL);

		if ((data & 0x1c0) != 0x1c0) {
			/* demux enabled, needs staggered power up */
			status |= cs4224_reg_set_channel(slice,
				CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				(data | 0x1f));
		}

		status |= cs4224_reg_set_channel(slice,
				CS4224_PP_HOST_SDS_DSP_MSEQ_POWER_DOWN_LSB,
				data);
	}

	if (cs4224_is_hw_duplex(slice) ||
	    cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_mseq_squelch_ctrl(
			slice, CS4224_LINE_RX_TO_HOST_TX_DIR,
			rules->tx_auto_squelch);
	}
	if (cs4224_is_hw_duplex(slice) ||
	    !cs4224_line_rx_to_host_tx_dir(slice)) {
		status |= cs4224_mseq_squelch_ctrl(
			slice, CS4224_HOST_RX_TO_LINE_TX_DIR,
			rules->tx_auto_squelch);
	}

	/* Unsquelching before the microcode is unstalled to prevent ordering
	 * issues. See bug #41747 for details.
	 */
	if (rules->unsquelch_driver) {
		if (cs4224_is_hw_duplex(slice) ||
		    cs4224_line_rx_to_host_tx_dir(slice)) {
			/* remove any pending squelch requests */
			status |= cs4224_clear_mailbox(slice,
							CS4224_DPLX_LINE_MSEQ);
			status |= cs4224_reg_set_channel(
				slice, CS4224_PP_HOST_SDS_COMMON_STX0_SQUELCH,
				0x0000);
		}
		if (cs4224_is_hw_duplex(slice) ||
		    !cs4224_line_rx_to_host_tx_dir(slice)) {
			/* remove any pending squelch requests */
			status |= cs4224_clear_mailbox(slice,
							CS4224_DPLX_HOST_MSEQ);
			status |= cs4224_reg_set_channel(
				slice, CS4224_PP_LINE_SDS_COMMON_STX0_SQUELCH,
				0x0000);
		}
	}

	/* do not start microsequencer if protocol running on the line side */
	if ((rules->application != CS4224_TARGET_APPLICATION_10G_KR) &&
	    (rules->application != CS4224_TARGET_APPLICATION_40G_KR) &&
	    (rules->application != CS4224_TARGET_APPLICATION_KRAN) &&
	    (rules->application != CS4224_TARGET_APPLICATION_FCAN)) {
		/* un-stall the microsequencer(s), if required */
		if (cs4224_is_hw_simplex(slice)) {
			if (rules->rx_if.splx_edc_mode !=
			    CS_HSIO_EDC_MODE_DISABLED) {
				status |= cs4224_mseq_stall(
					slice, CS4224_SPLX_MSEQ, CS_FALSE);
			}
		} else { /* duplex */
			if (rules->rx_if.dplx_line_edc_mode !=
			    CS_HSIO_EDC_MODE_DISABLED)
				status |= cs4224_mseq_stall(
					slice, CS4224_DPLX_LINE_MSEQ, CS_FALSE);
			if (rules->rx_if.dplx_host_edc_mode !=
			    CS_HSIO_EDC_MODE_DISABLED)
				status |= cs4224_mseq_stall(
					slice, CS4224_DPLX_HOST_MSEQ, CS_FALSE);
		}
	}

	cs4224_unlock(slice);

	return status;
}

unsigned int cs4224_get_die_from_slice(unsigned int slice)
{
	unsigned int die = 0;
	enum e_cs4224_hardware_id hw_id = cs4224_hw_id(slice);

	/* Save the upper bits to pass thru to the die parameter */
	unsigned int upper_bits = slice & 0xffffff00;

	switch (hw_id) {
	case CS4224_HW_CS4223: /*  4 port duplex   */
	case CS4224_HW_CS4227: /*  2 port duplex   */
		die = 0;
		break;

	case CS4224_HW_CS4343: /*  8 port duplex   */
	case CS4224_HW_CS4341:
		/* Only the lower 8 bits belong to the API. They represent
		 * the duplex channel (0-7) of the device.
		 */
		if ((slice & 0xFF) > 3)
			die = 0;
		else
			die = 1;
		break;

	case CS4224_HW_CS4224: /* 16 port simplex  */
	case CS4224_HW_CS4210:
		/* Only the lower 8 bits belong to the API. They represent
		 * the simplex channel (0-15) of the device.
		 */
		if ((slice & 0xFF) > 7)
			die = 0;
		else
			die = 1;
		break;

	case CS4224_HW_CS4221: /* 10 port simplex */
		/* Only the lower 8 bits belong to the API. They represent
		 * the duplex channel (0-9) of the device.
		 */
		if ((slice & 0xFF) > 4)
			die = 0;
		else
			die = 1;
		break;

	default:
		CS_TRACE(("ERROR: HW ID not valid, hw_id = %x\n", hw_id));
		die = 0;
		break;
	}

	return die | upper_bits;
}

int cs4224_reg_get_channel(unsigned int slice, unsigned int addr,
			   unsigned short *data)
{
	unsigned int die = 0;
	int status = CS_OK;

	/* If the address has not already been adjusted
	 * then do the port pair adjustment here
	 */
	if ((addr >= 0x1000) && (addr < 0x5000)) {
		/* Mask out the upper bits if they have already been adjusted */
		addr = (addr | 0x3000) & 0x1fff;

		/* guard against reading from bad registers */
		if (!cs4224_diags_register_can_read((unsigned short)addr)) {
			*data = 0xbeef;
			return CS_OK;
		}

		/* Now adjust them to access the proper port pair */
		addr = cs4224_adj_pp(slice, addr);

		/* CS_PRINTF(("after slice = %d, addr = %x\n", slice, addr));*/
	}
	/* Handle the MSEQ_PS space */
	else if ((addr >= 0x300) && (addr < 0x500)) {
		/* Mask out any upper bits if they have already been adjusted */
		addr = (addr | 0x300) & 0x3ff;
		addr = cs4224_adj_mseq(slice, addr);
	}

	/* get the die associated with this slice */
	die = cs4224_get_die_from_slice(slice);

	/* Set the die but make sure to pass the upper 24 bits of the
	 * slice parameter through to the lower level API. These are
	 * used by the customer to address multiple devices
	 */
	status |= cs4224_reg_get(die | (slice >> 8 << 8), addr, data);

	return status;
}

unsigned short cs4224_reg_read(unsigned int slice, unsigned int addr)
{
	unsigned short data;
	int status = CS_OK;

	status |= cs4224_reg_get_channel(slice, addr, &data);
	if (status != CS_OK) {
		CS_TRACE(
			("ERROR: reg get failed; status %x, returning 0xDEAD\n",
			 status));
		data = 0xDEAD;
	}

	return data;
}

int cs4224_reg_set_channel(unsigned int slice, unsigned int addr,
			   unsigned short data)
{
	unsigned int die = 0;

	/* If the PP address has not already been adjusted
	 * then do the port pair adjustment here
	 */
	if ((addr >= 0x1000) && (addr < 0x5000)) {
		/* Mask out the upper bits if they have already been adjusted */
		addr = (addr | 0x3000) & 0x1fff;

		/* Now adjust them to access the proper port pair */
		addr = cs4224_adj_pp(slice, addr);
	}
	/* Handle the MSEQ_PS space */
	else if ((addr >= 0x300) && (addr < 0x500)) {
		/* Mask out any upper bits if they have already been adjusted */
		addr = (addr | 0x300) & 0x3ff;
		addr = cs4224_adj_mseq(slice, addr);
	}

	/* get the die associated with this slice */
	die = cs4224_get_die_from_slice(slice);

	/* Set the die but make sure to pass the upper 24 bits of the
	 * slice parameter through to the lower level API. These are
	 * used by the customer to address multiple devices
	 */
	return cs4224_reg_set(die | (slice >> 8 << 8), addr, data);
}

int cs4224_reg_set_chk_channel(unsigned int slice, unsigned int addr,
			       unsigned short data)
{
	int status = CS_OK;
	unsigned short get_data = 0;

	status |= cs4224_reg_set_channel(slice, addr, data);
	status |= cs4224_reg_get_channel(slice, addr, &get_data);
	if (get_data != data) {
		CS_TRACE(
			("ERROR: Data was written to register\n"));
		CS_PRINTF(
			("reg addr: %4x data written: %4x data read: %4x\n",
			 addr, data, get_data));
		status |= CS_ERROR;
	}
	return status;
}

const char *cs4224_translate_edc_mode(enum e_cs4224_edc_mode edc_mode)
{
	switch (edc_mode) {
		case CS_HSIO_EDC_MODE_DISABLED: {
			return "CS_HSIO_EDC_MODE_DISABLED";
		}
		case CS_HSIO_EDC_MODE_CX1: {
			return "CS_HSIO_EDC_MODE_CX1";
		}
		case CS_HSIO_EDC_MODE_SR: {
			return "CS_HSIO_EDC_MODE_SR";
		}
		case CS_HSIO_EDC_MODE_ZR: {
			return "CS_HSIO_EDC_MODE_ZR";
		}
		case CS_HSIO_EDC_MODE_DWDM: {
			return "CS_HSIO_EDC_MODE_DWDM";
		}
		case CS_HSIO_EDC_MODE_10G_BP: {
			return "CS_HSIO_EDC_MODE_10G_BP";
		}
		case CS_HSIO_EDC_MODE_15G_BP: {
			return "CS_HSIO_EDC_MODE_15G_BP";
		}
		case CS_HSIO_EDC_MODE_15G_BP_27dB: {
			return "CS_HSIO_EDC_MODE_15G_BP_27dB";
		}
		case CS_HSIO_EDC_MODE_5G_BP: {
			return "CS_HSIO_EDC_MODE_5G_BP";
		}
		case CS_HSIO_EDC_MODE_7p5G_BP: {
			return "CS_HSIO_EDC_MODE_7p5G_BP";
		}
		case CS_HSIO_EDC_MODE_8p5G_BP: {
			return "CS_HSIO_EDC_MODE_8p5G_BP";
		}
		case CS_HSIO_EDC_MODE_FCAN: {
			return "CS_HSIO_EDC_MODE_FCAN";
		}
		case CS_HSIO_EDC_MODE_SMLRM: {
			return "CS_HSIO_EDC_MODE_SMLRM";
		}
	}

	return "UNKNOWN EDC MODE";
}

const char *cs4224_translate_app_mode(enum e_cs4224_target_application app_mode)
{
	switch (app_mode) {
	case CS4224_TARGET_APPLICATION_1G: {
		return "CS4224_TARGET_APPLICATION_1G";
	}
	case CS4224_TARGET_APPLICATION_5G: {
		return "CS4224_TARGET_APPLICATION_5G";
	}
	case CS4224_TARGET_APPLICATION_7p5G: {
		return "CS4224_TARGET_APPLICATION_7p5G";
	}
	case CS4224_TARGET_APPLICATION_8G: {
		return "CS4224_TARGET_APPLICATION_8G";
	}
	case CS4224_TARGET_APPLICATION_8p5G: {
		return "CS4224_TARGET_APPLICATION_8p5G";
	}
	case CS4224_TARGET_APPLICATION_10G: {
		return "CS4224_TARGET_APPLICATION_10G";
	}
	case CS4224_TARGET_APPLICATION_15G: {
		return "CS4224_TARGET_APPLICATION_15G";
	}
	case CS4224_TARGET_APPLICATION_10G_KR: {
		return "CS4224_TARGET_APPLICATION_10G_KR";
	}
	case CS4224_TARGET_APPLICATION_40G_KR: {
		return "CS4224_TARGET_APPLICATION_40G_KR";
	}
	case CS4224_TARGET_APPLICATION_KRAN: {
		return "CS4224_TARGET_APPLICATION_KRAN";
	}
	case CS4224_TARGET_APPLICATION_FCAN: {
		return "CS4224_TARGET_APPLICATION_FCAN";
	}
	case CS4224_TARGET_APPLICATION_16G_FC: {
		return "CS4224_TARGET_APPLICATION_16G_FC";
	}
	case CS4224_TARGET_APPLICATION_10G_FC: {
		return "CS4224_TARGET_APPLICATION_10G_FC";
	}
	case CS4224_TARGET_APPLICATION_8G_FC: {
		return "CS4224_TARGET_APPLICATION_8G_FC";
	}
	case CS4224_TARGET_APPLICATION_4G_FC: {
		return "CS4224_TARGET_APPLICATION_4G_FC";
	}
	case CS4224_TARGET_APPLICATION_2G_FC: {
		return "CS4224_TARGET_APPLICATION_2G_FC";
	}
	case CS4224_TARGET_APPLICATION_1G_FC: {
		return "CS4224_TARGET_APPLICATION_1G_FC";
	}
	case CS4224_TARGET_APPLICATION_OC12: {
		return "CS4224_TARGET_APPLICATION_OC12";
	}
	}

	return "UNKNOWN APP MODE";
}

const char *cs4224_switch_translate_state(enum e_cs4224_switch_action_t state)
{
	switch (state) {
	case CS4224_SWITCH_DISABLE:
		return "CS4224_SWITCH_DISABLE";
	case CS4224_SWITCH_DUPLEX_BROADCAST_1_to_1:
		return "CS4224_SWITCH_DUPLEX_BROADCAST_1_to_1";
	case CS4224_SWITCH_DUPLEX_BROADCAST_0_to_0:
		return "CS4224_SWITCH_DUPLEX_BROADCAST_0_to_0";
	case CS4224_SWITCH_DUPLEX_BROADCAST_0_to_1:
		return "CS4224_SWITCH_DUPLEX_BROADCAST_0_to_1";
	case CS4224_SWITCH_DUPLEX_BROADCAST_1_to_0:
		return "CS4224_SWITCH_DUPLEX_BROADCAST_1_to_0";
	case CS4224_SWITCH_DUPLEX_SWITCH_2x2:
		return "CS4224_SWITCH_DUPLEX_SWITCH_2x2";
	case CS4224_SWITCH_SIMPLEX_SWITCH_2x2:
		return "CS4224_SWITCH_SIMPLEX_SWITCH_2x2";
	case CS4224_SWITCH_SIMPLEX_BROADCAST:
		return "CS4224_SWITCH_SIMPLEX_BROADCAST";

	default:
		break;
	}

	return "???";
}

const char *cs4224_translate_cfg_side(enum e_cs4224_cfg_sides_t side)
{
	switch (side) {
	case CS4224_CFG_HOST_SIDE:
		return "host";
	case CS4224_CFG_LINE_SIDE:
		return "line";
	case CS4224_CFG_SIMPLEX:
		return "splx";
	default:
		break;
	}
	return "????";
}

#if !defined(CS_SKIP_UCODE_DOWNLOAD)

int cs4224_ucode_data_prgm_image(unsigned int slice, unsigned short microcode[],
				 unsigned int len_buffer)
{
	unsigned int i;
	int status = CS_OK;
	unsigned short data0, data1;
	unsigned int die;

	/* get the die associated to this slice */
	die = cs4224_get_die_from_slice(slice);

	/* toggle the checksum calculator, note that the file data will
	 * assert the checksum bit to start calculating the checksum
	 */
	cs4224_lock(slice);
	cs4224_reg_set(die, CS4224_GLOBAL_DWNLD_CHECKSUM_CTRL, 0x01);
	cs4224_reg_set(die, CS4224_GLOBAL_DWNLD_CHECKSUM_CTRL, 0x00);
	cs4224_unlock(slice);

	for (i = 0; i < len_buffer; i += 2) {
		unsigned short addr = microcode[i];
		unsigned short data = microcode[i + 1];
		unsigned short access_status = 0;
		short max_iterations = 100;

		/* Program data here */
		cs4224_lock(slice);

		status |= cs4224_reg_set(die, addr, data);

		/* If this is a write to the EEPROM_MICRO_ACCESS_CONTROL
		 * then wait for the previous access to finish.
		 */
		if (addr == CS4224_EEPROM_MICRO_ACCESS_CONTROL) {
			/* Wait for EEPROM_MICRO_ACCESS_STATUS.MicroAccessDone
			 * bit to be set.
			 */
			while (CS_IF_CLR(access_status, CS_BIT0) &&
			       (max_iterations > 0)) {
				cs4224_reg_get(
					die, CS4224_EEPROM_MICRO_ACCESS_STATUS,
					&access_status);
				max_iterations -= 1;
			}

			if (max_iterations <= 0) {
				CS_TRACE(("ERROR: Timed out on die %x\n", die));
				return CS_ERROR;
			}
		}

		cs4224_unlock(slice);
	}

	status |= cs4224_reg_get(die, CS4224_GLOBAL_DWNLD_CHECKSUM_STATUS,
					&data0);
	if (data0 == 0x1) {
		status |= cs4224_reg_get(die, CS4224_GLOBAL_DWNLD_CHECKSUM_HW,
						&data0);
		status |= cs4224_reg_get(die, CS4224_GLOBAL_DWNLD_CHECKSUM_SW,
						&data1);
		CS_TRACE(
			("ERROR: Comms issue! checksum: img=0x%04x,hw=0x%04x\n",
			 data1, data0));
		status |= CS_ERROR;
	}

	return status;
}

unsigned int cs4224_calc_crc16(unsigned int input_data, unsigned int crc_init)
{
	unsigned int i, j, crc;
	unsigned int tmp, msb, data = 0;

	crc = crc_init & 0x0000ffff;

	for (i = 0; i < 4; i++) {
		if (i == 0)
			data = (input_data & 0xff000000) >> 24;
		if (i == 1)
			data = (input_data & 0x00ff0000) >> 16;
		if (i == 2)
			data = (input_data & 0x0000ff00) >> 8;
		if (i == 3)
			data = (input_data & 0xff0000ff) >> 0;

		tmp = 0x000080;

		for (j = 0; j < 8; j++) {
			msb = crc & 0x8000;
			crc <<= 1;
			crc &= 0x0000ffff;
			if (data & tmp)
				msb ^= 0x00008000;
			if (msb)
				crc ^= 0x1021;
			tmp >>= 1;
		}
	}
	return crc;
}

enum e_cs4224_ucode_crc16_t {
	CS4224_UCODE_PRE_CRC16_CALC = 1, /* Pre CRC-16 calculation state */
	CS4224_UCODE_ACTIVE_CRC16_CALC = 2, /* CRC-16 calculation state */
	CS4224_UCODE_POST_CRC16_CALC = 3, /* Post CRC-16 calculation state */
};

int cs4224_ucode_data_prgm_image_broadcast(unsigned int slice,
					   unsigned short microcode[],
						 unsigned int len_buffer)
{
	unsigned int i;
	int status = CS_OK;
	int tmp_status = CS_OK;
	unsigned short checksum, data;
	unsigned int crc_sw = 0xffff;
	unsigned int crc_hw = 0xffff;
	unsigned int input_data_hw, input_data_sw;
	enum e_cs4224_ucode_crc16_t crc_sm = CS4224_UCODE_PRE_CRC16_CALC;

	/* upper bits of slice used to pick chip being used */
	unsigned int upper_bits = slice & 0xffffff00;

	if (!microcode || len_buffer == 0)
		return CS_ERROR;

	/* Ensure the MDIO timeout is increased on both dies, Bugzilla 35875 */
	status |= cs4224_reg_set(0 | upper_bits, CS4224_GLOBAL_MDIO_CONFIG,
					CS4224_GLOBAL_MDIO_CONFIG_pref);
	status |= cs4224_reg_set(1 | upper_bits, CS4224_GLOBAL_MDIO_CONFIG,
					CS4224_GLOBAL_MDIO_CONFIG_pref);

	/* config to broadcast to both dies */
	data = 0x0100;

	/* force die 1 to listen to die 0's address */
	/* ignore the status returned by the reg set since an I2C transaction
	 * will fail to get the checksum
	 */
	cs4224_reg_set(1 | upper_bits, CS4224_GLOBAL_BROADCAST, data);

	if (status != CS_OK)
		return status;

	for (i = 0; i < len_buffer; i += 2) {
		unsigned short addr = microcode[i];
		unsigned short data = microcode[i + 1];

		if (addr == CS4224_EEPROM_MICRO_ACCESS_CONTROL) {
			CS_TRACE(("ERROR: Cannot program the EEPROM bcast!\n"));

			/* disable broadcasting to both dies */
			cs4224_reg_set(0 | upper_bits, CS4224_GLOBAL_BROADCAST,
				       0x0000);

			return CS_ERROR;
		}

		if (crc_sm == CS4224_UCODE_ACTIVE_CRC16_CALC) {
			/* calculate our own crc that will mirror the one in
			 * the image, without the below BROADCAST trap
			 */
			input_data_sw = (addr << 16) + data;
			crc_sw = cs4224_calc_crc16(input_data_sw, crc_sw);
		}

		if (addr == CS4224_GLOBAL_BROADCAST) {
			/* intercept to ensure the broadcast bit remains set */
			data |= 0x0100;
		}

		if (crc_sm == CS4224_UCODE_ACTIVE_CRC16_CALC) {
			/* calculate our own crc that that will mirror the
			 * HW calculated one,
			 * with the above BROADCAST trap
			 */
			input_data_hw = (addr << 16) + data;
			crc_hw = cs4224_calc_crc16(input_data_hw, crc_hw);
		}

		cs4224_lock(slice);

		/* Program hw here */
		tmp_status = cs4224_reg_set(0 | upper_bits, addr, data);

		cs4224_unlock(slice);

		if (addr != CS4224_GLOBAL_BROADCAST) {
			/* ignore the status of writes to BROADCAST */
			status |= tmp_status;
		}

		if (status != CS_OK) {
			/* some kind of legitimate write issue */
			CS_TRACE(
				("ERROR: cs4224_reg_set(0x%x, 0x%x, 0x%x)\n",
				 upper_bits,
				 addr,
				 data));

			/* disable broadcasting to both dies */
			cs4224_reg_set(0 | upper_bits, CS4224_GLOBAL_BROADCAST,
				       0x0000);

			return CS_ERROR;
		}

		/* update our checksum calculator state machine */
		if (addr == CS4224_GLOBAL_DWNLD_CHECKSUM_CTRL) {
			if (data == 0x0001) {
				/* HW CRC-16 calculator enabled */
				crc_sm = CS4224_UCODE_ACTIVE_CRC16_CALC;
			} else {
				/* HW CRC-16 calculator disabled */
				crc_sm = CS4224_UCODE_POST_CRC16_CALC;
			}
		}
	}

	/* disable broadcasting to both dies */
	status |= cs4224_reg_set(0 | upper_bits, CS4224_GLOBAL_BROADCAST,
					0x0000);

	/* get the image checksum */
	status |= cs4224_reg_get(0 | upper_bits,
					CS4224_GLOBAL_DWNLD_CHECKSUM_SW,
					&checksum);

	/* compare our mirror checksum with the image checksum, this
	 * checks that OUR program memory or wherever we got the image from
	 * isn't corrupted
	 */
	if (crc_sw != checksum) {
		CS_TRACE(("ERROR: (image corrupt?),calc=0x%04x, image=0x%04x\n",
			  crc_sw,
			  checksum));
		status |= CS_ERROR;
	}

	/* write our calculated hardware checksum to CHECKSUM_SW reg so the
	 * CHECKSUM_STATUS clears
	 */
	status |= cs4224_reg_set(0 | upper_bits,
				 CS4224_GLOBAL_DWNLD_CHECKSUM_SW,
				 crc_hw);
	status |= cs4224_reg_set(1 | upper_bits,
				 CS4224_GLOBAL_DWNLD_CHECKSUM_SW,
				 crc_hw);

	/* check that the chip hw checksum and our calculated
	 * crc_hw checksum match
	 */
	status |= cs4224_reg_get(0 | upper_bits,
				 CS4224_GLOBAL_DWNLD_CHECKSUM_STATUS,
				 &data);
	if (data == 0x1) {
		status |= cs4224_reg_get(0 | upper_bits,
					 CS4224_GLOBAL_DWNLD_CHECKSUM_HW,
					 &checksum);
		CS_TRACE(("ERROR: Comms issue checksum calc=0x%04x,hw=0x%04x\n",
			  crc_hw,
			  checksum));
		status |= CS_ERROR;
	}

	status |= cs4224_reg_get(1 | upper_bits,
				 CS4224_GLOBAL_DWNLD_CHECKSUM_STATUS,
				 &data);
	if (data == 0x1) {
		status |= cs4224_reg_get(1 | upper_bits,
					 CS4224_GLOBAL_DWNLD_CHECKSUM_HW,
					 &checksum);
		CS_TRACE(("ERROR: checksum match, calc=0x%04x, hw=0x%04x\n",
			  crc_hw,
			  checksum));
		status |= CS_ERROR;
	}

	if (status != CS_OK) {
		/* Clear the timestamps */
		status |= cs4224_reg_set(0 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP0,
					 0x0);
		status |= cs4224_reg_set(0 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP1,
					 0x0);
		status |= cs4224_reg_set(0 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP2,
					 0x0);
		status |= cs4224_reg_set(0 | upper_bits,
					 CS4224_GLOBAL_UCODE_VERSION_KR,
					 0x0);
		status |= cs4224_reg_set(0 | upper_bits,
					 CS4224_GLOBAL_UCODE_VERSION_FC,
					 0x0);

		status |= cs4224_reg_set(1 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP0,
					 0x0);
		status |= cs4224_reg_set(1 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP1,
					 0x0);
		status |= cs4224_reg_set(1 | upper_bits,
					 CS4224_GLOBAL_UCODE_TIMESTAMP2,
					 0x0);
		status |= cs4224_reg_set(1 | upper_bits,
					 CS4224_GLOBAL_UCODE_VERSION_KR,
					 0x0);
		status |= cs4224_reg_set(1 | upper_bits,
					 CS4224_GLOBAL_UCODE_VERSION_FC,
					 0x0);
	}

	return status;
}

#endif /* !CS_SKIP_UCODE_DOWNLOAD */
