// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright 2017-2018 NXP
 */
#include <common.h>
#include <cpu_func.h>
#include <malloc.h>
#include <errno.h>
#include <netdev.h>
#include <fsl_ifc.h>
#include <fdt_support.h>
#include <linux/libfdt.h>
#include <env.h>
#include <i2c.h>
#include "pca953x.h"

#include <asm/io.h>
#include <asm/gpio.h>
#include <asm/arch/clock.h>
#include <asm/arch/sci/sci.h>
#include <asm/arch/imx8-pins.h>
#include <asm/arch/snvs_security_sc.h>
#include <dm.h>
#include <usb.h>
#include <asm/arch/iomux.h>
#include <asm/arch/sys_proto.h>
#include <power-domain.h>
#include <asm/arch/lpcg.h>
#include <bootm.h>
#include <linux/delay.h>

DECLARE_GLOBAL_DATA_PTR;

#define ENET_INPUT_PAD_CTRL	((SC_PAD_CONFIG_OD_IN << PADRING_CONFIG_SHIFT) | (SC_PAD_ISO_OFF << PADRING_LPCONFIG_SHIFT) \
						| (SC_PAD_28FDSOI_DSE_18V_10MA << PADRING_DSE_SHIFT) | (SC_PAD_28FDSOI_PS_PU << PADRING_PULL_SHIFT))

#define ENET_NORMAL_PAD_CTRL	((SC_PAD_CONFIG_NORMAL << PADRING_CONFIG_SHIFT) | (SC_PAD_ISO_OFF << PADRING_LPCONFIG_SHIFT) \
						| (SC_PAD_28FDSOI_DSE_18V_10MA << PADRING_DSE_SHIFT) | (SC_PAD_28FDSOI_PS_PU << PADRING_PULL_SHIFT))

#define GPIO_PAD_CTRL	((SC_PAD_CONFIG_NORMAL << PADRING_CONFIG_SHIFT) | (SC_PAD_ISO_OFF << PADRING_LPCONFIG_SHIFT) \
						| (SC_PAD_28FDSOI_DSE_DV_HIGH << PADRING_DSE_SHIFT) | (SC_PAD_28FDSOI_PS_PU << PADRING_PULL_SHIFT))

#define UART_PAD_CTRL	((SC_PAD_CONFIG_OUT_IN << PADRING_CONFIG_SHIFT) | (SC_PAD_ISO_OFF << PADRING_LPCONFIG_SHIFT) \
						| (SC_PAD_28FDSOI_DSE_DV_HIGH << PADRING_DSE_SHIFT) | (SC_PAD_28FDSOI_PS_PU << PADRING_PULL_SHIFT))

static iomux_cfg_t uart0_pads[] = {
	SC_P_UART0_RX | MUX_PAD_CTRL(UART_PAD_CTRL),
	SC_P_UART0_TX | MUX_PAD_CTRL(UART_PAD_CTRL),
};

static void setup_iomux_uart(void)
{
	imx8_iomux_setup_multiple_pads(uart0_pads, ARRAY_SIZE(uart0_pads));
}

int board_early_init_f(void)
{
	sc_pm_clock_rate_t rate = SC_80MHZ;
	int ret;

	/* Set UART0 clock root to 80 MHz */
	ret = sc_pm_setup_uart(SC_R_UART_0, rate);
	if (ret)
		return ret;

	setup_iomux_uart();

	return 0;
}

#if IS_ENABLED(CONFIG_FEC_MXC)
#include <miiphy.h>

#define MAX7322_I2C_ADDR		0x68
#define MAX7322_I2C_BUS		2 /* I2C2 */

int board_phy_config(struct phy_device *phydev)
{
	if (phydev->addr == 1) {
		/* This is needed to drive the pads to 1.8V instead of 1.5V */
		uint8_t value;
		struct udevice *bus;
		struct udevice *i2c_dev = NULL;
		int ret;

		ret = uclass_get_device_by_seq(UCLASS_I2C, MAX7322_I2C_BUS, &bus);
		if (ret) {
			printf("%s: Can't find bus\n", __func__);
			return -EINVAL;
		}

		ret = dm_i2c_probe(bus, MAX7322_I2C_ADDR, 0, &i2c_dev);
		if (ret) {
			printf("%s: Can't find device id=0x%x\n",
				__func__, MAX7322_I2C_ADDR);
			return -ENODEV;
		}

		i2c_set_chip_offset_len(i2c_dev, 0);

		value = 0x1;

		ret = dm_i2c_write(i2c_dev, 0x0, (const uint8_t *)&value, 1);
		if (ret) {
			printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
			return -EIO;
		}

		mdelay(1);
	}

	if (phydev->drv->config)
		phydev->drv->config(phydev);

	return 0;
}

#endif

#define LVDS_ENABLE IMX_GPIO_NR(1, 6)
#define MIPI_ENABLE IMX_GPIO_NR(1, 7)
#define DEBUG_LED   IMX_GPIO_NR(2, 15)
#define IOEXP_RESET IMX_GPIO_NR(1, 12)


static void board_gpio_init(void)
{
	int ret;
	struct gpio_desc desc;
	struct udevice *dev;

	/* enable i2c port expander assert reset line first */
	/* we can't use dm_gpio_lookup_name for GPIO1_12, because the func will probe the
	 * uclass list until find the device. The expander device is at begin of the list due to
	 * I2c nodes is prior than gpio in the DTS. So if the func goes through the uclass list,
	 * probe to expander will fail, and exit the dm_gpio_lookup_name func. Thus, we always
	 * fail to get the device
	*/
	ret = uclass_get_device_by_seq(UCLASS_GPIO, 1, &dev);
	if (ret) {
		printf("%s failed to find GPIO1 device, ret = %d\n", __func__, ret);
		return;
	}

	desc.dev = dev;
	desc.offset = 12;
	desc.flags = 0;

	ret = dm_gpio_request(&desc, "ioexp_rst");
	if (ret) {
		printf("%s request ioexp_rst failed ret = %d\n", __func__, ret);
		return;
	}

	dm_gpio_set_dir_flags(&desc, GPIOD_IS_OUT | GPIOD_IS_OUT_ACTIVE);


	ret = dm_gpio_lookup_name("GPIO2_15", &desc);
	if (ret) {
		printf("%s lookup GPIO@2_15 failed ret = %d\n", __func__, ret);
		return;
	}

	ret = dm_gpio_request(&desc, "debug_led");
	if (ret) {
		printf("%s request debug_led failed ret = %d\n", __func__, ret);
		return;
	}

	dm_gpio_set_dir_flags(&desc, GPIOD_IS_OUT | GPIOD_IS_OUT_ACTIVE);


	ret = dm_gpio_lookup_name("GPIO1_6", &desc);
	if (ret) {
		printf("%s lookup GPIO@1_6 failed ret = %d\n", __func__, ret);
		return;
	}

	ret = dm_gpio_request(&desc, "lvds_enable");
	if (ret) {
		printf("%s request lvds_enable failed ret = %d\n", __func__, ret);
		return;
	}

	dm_gpio_set_dir_flags(&desc, GPIOD_IS_OUT | GPIOD_IS_OUT_ACTIVE);

	ret = dm_gpio_lookup_name("GPIO1_7", &desc);
	if (ret) {
		printf("%s lookup GPIO@1_7 failed ret = %d\n", __func__, ret);
		return;
	}

	ret = dm_gpio_request(&desc, "mipi_enable");
	if (ret) {
		printf("%s request mipi_enable failed ret = %d\n", __func__, ret);
		return;
	}

	dm_gpio_set_dir_flags(&desc, GPIOD_IS_OUT | GPIOD_IS_OUT_ACTIVE);
}

int checkboard(void)
{
#ifdef CONFIG_TARGET_IMX8QM_LPDDR4_VAL
	puts("Board: iMX8QM LPDDR4 VAL\n");
#else
	puts("Board: iMX8QM DDR4 VAL\n");
#endif

	print_bootinfo();

	return 0;
}

int board_init(void)
{
	board_gpio_init();


#ifdef CONFIG_IMX_SNVS_SEC_SC_AUTO
	{
		int ret = snvs_security_sc_init();

		if (ret)
			return ret;
	}
#endif

	return 0;
}

void board_quiesce_devices(void)
{
	const char *power_on_devices[] = {
		"dma_lpuart0",
	};

	imx8_power_off_pd_devices(power_on_devices, ARRAY_SIZE(power_on_devices));
}

/*
 * Board specific reset that is system reset.
 */
void reset_cpu(void)
{
	/* TODO */
}

#ifdef CONFIG_OF_BOARD_SETUP
int ft_board_setup(void *blob, struct bd_info *bd)
{
	return 0;
}
#endif

int board_late_init(void)
{
	build_info();

#ifdef CONFIG_ENV_VARS_UBOOT_RUNTIME_CONFIG
	env_set("board_name", "VAL");
	env_set("board_rev", "iMX8QM");
#endif

	env_set("sec_boot", "no");
#ifdef CONFIG_AHAB_BOOT
	env_set("sec_boot", "yes");
#endif

#ifdef CONFIG_ENV_IS_IN_MMC
	board_late_mmc_env_init();
#endif

	return 0;
}
