/* SPDX-License-Identifier: GPL-2.0 */
/*
 * NXP XCVR ALSA SoC Digital Audio Interface (DAI) driver
 *
 * Copyright 2019 NXP
 */

#ifndef __FSL_XCVR_H
#define __FSL_XCVR_H

#include <linux/regmap.h>
#include <linux/reset.h>
#include <sound/dmaengine_pcm.h>
#include <sound/pcm_iec958.h>

#define FSL_XCVR_MODE_SPDIF	0
#define FSL_XCVR_MODE_ARC	1
#define FSL_XCVR_MODE_EARC	2

/* XCVR Registers */
#define FSL_XCVR_REG_OFFSET		0x800 /* regs offset */
#define FSL_XCVR_FIFO_SIZE		0x80  /* 128 */
#define FSL_XCVR_FIFO_WMK_RX		(FSL_XCVR_FIFO_SIZE >> 1)   /* 64 */
#define FSL_XCVR_FIFO_WMK_TX		(FSL_XCVR_FIFO_SIZE >> 1)   /* 64 */
#define FSL_XCVR_MAXBURST_RX		(FSL_XCVR_FIFO_WMK_RX >> 2) /* 16 */
#define FSL_XCVR_MAXBURST_TX		(FSL_XCVR_FIFO_WMK_TX >> 2) /* 16 */

#define FSL_XCVR_RX_FIFO_ADDR		0x0C00
#define FSL_XCVR_TX_FIFO_ADDR		0x0E00

#define FSL_XCVR_VERSION		0x00  /* Version */
#define FSL_XCVR_EXT_CTRL		0x10  /* Control */
#define FSL_XCVR_EXT_STATUS		0x20  /* Status */
#define FSL_XCVR_EXT_IER0		0x30  /* Interrupt en 0 */
#define FSL_XCVR_EXT_IER1		0x40  /* Interrupt en 1 */
#define FSL_XCVR_EXT_ISR		0x50  /* Interrupt status */
#define FSL_XCVR_EXT_ISR_SET		0x54  /* Interrupt status */
#define FSL_XCVR_EXT_ISR_CLR		0x58  /* Interrupt status */
#define FSL_XCVR_EXT_ISR_TOG		0x5C  /* Interrupt status */
#define FSL_XCVR_IER			0x70  /* Interrupt en for M0+ */
#define FSL_XCVR_ISR			0x80  /* Interrupt status */
#define FSL_XCVR_ISR_SET		0x84  /* Interrupt status set */
#define FSL_XCVR_ISR_CLR		0x88  /* Interrupt status clear */
#define FSL_XCVR_ISR_TOG		0x8C  /* Interrupt status toggle */
#define FSL_XCVR_PHY_AI_CTRL		0x90
#define FSL_XCVR_PHY_AI_CTRL_SET	0x94
#define FSL_XCVR_PHY_AI_CTRL_CLR	0x98
#define FSL_XCVR_PHY_AI_CTRL_TOG	0x9C
#define FSL_XCVR_PHY_AI_WDATA		0xA0
#define FSL_XCVR_PHY_AI_RDATA		0xA4
#define FSL_XCVR_CLK_CTRL		0xB0
#define FSL_XCVR_RX_DPTH_CTRL		0x180 /* RX datapath ctrl reg */
#define FSL_XCVR_RX_DPTH_CTRL_SET	0x184
#define FSL_XCVR_RX_DPTH_CTRL_CLR	0x188
#define FSL_XCVR_RX_DPTH_CTRL_TOG	0x18c

#define FSL_XCVR_RX_CS_DATA_0		0x190
#define FSL_XCVR_RX_CS_DATA_1		0x194
#define FSL_XCVR_RX_CS_DATA_2		0x198
#define FSL_XCVR_RX_CS_DATA_3		0x19C
#define FSL_XCVR_RX_CS_DATA_4		0x1A0
#define FSL_XCVR_RX_CS_DATA_5		0x1A4

#define FSL_XCVR_RX_DPTH_CNTR_CTRL	0x1C0
#define FSL_XCVR_RX_DPTH_CNTR_CTRL_SET	0x1C4
#define FSL_XCVR_RX_DPTH_CNTR_CTRL_CLR	0x1C8
#define FSL_XCVR_RX_DPTH_CNTR_CTRL_TOG	0x1CC

#define FSL_XCVR_RX_DPTH_TSCR		0x1D0
#define FSL_XCVR_RX_DPTH_BCR		0x1D4
#define FSL_XCVR_RX_DPTH_BCTR		0x1D8
#define FSL_XCVR_RX_DPTH_BCRR		0x1DC

#define FSL_XCVR_TX_DPTH_CTRL		0x220 /* TX datapath ctrl reg */
#define FSL_XCVR_TX_DPTH_CTRL_SET	0x224
#define FSL_XCVR_TX_DPTH_CTRL_CLR	0x228
#define FSL_XCVR_TX_DPTH_CTRL_TOG	0x22C
#define FSL_XCVR_TX_CS_DATA_0		0x230 /* TX channel status bits regs */
#define FSL_XCVR_TX_CS_DATA_1		0x234
#define FSL_XCVR_TX_CS_DATA_2		0x238
#define FSL_XCVR_TX_CS_DATA_3		0x23C
#define FSL_XCVR_TX_CS_DATA_4		0x240
#define FSL_XCVR_TX_CS_DATA_5		0x244

#define FSL_XCVR_TX_DPTH_CNTR_CTRL	0x260
#define FSL_XCVR_TX_DPTH_CNTR_CTRL_SET	0x264
#define FSL_XCVR_TX_DPTH_CNTR_CTRL_CLR	0x268
#define FSL_XCVR_TX_DPTH_CNTR_CTRL_TOG	0x26C

#define FSL_XCVR_TX_DPTH_TSCR		0x270
#define FSL_XCVR_TX_DPTH_BCR		0x274
#define FSL_XCVR_TX_DPTH_BCTR		0x278
#define FSL_XCVR_TX_DPTH_BCRR		0x27C

#define FSL_XCVR_DEBUG_REG_0		0x2E0
#define FSL_XCVR_DEBUG_REG_1		0x2F0

#define FSL_XCVR_MAX_REG		FSL_XCVR_DEBUG_REG_1

#define FSL_XCVR_EXT_CTRL_CORE_RESET	BIT(31)

#define FSL_XCVR_EXT_CTRL_RX_CMDC_RESET	BIT(30)
#define FSL_XCVR_EXT_CTRL_TX_CMDC_RESET	BIT(29)
#define FSL_XCVR_EXT_CTRL_CMDC_RESET(t) (t ? BIT(29) : BIT(30))

#define FSL_XCVR_EXT_CTRL_RX_DPTH_RESET	BIT(28)
#define FSL_XCVR_EXT_CTRL_TX_DPTH_RESET	BIT(27)
#define FSL_XCVR_EXT_CTRL_DPTH_RESET(t) (t ? BIT(27) : BIT(28))

#define FSL_XCVR_EXT_CTRL_TX_RX_MODE	BIT(26)
#define FSL_XCVR_EXT_CTRL_DMA_RD_DIS	BIT(25)
#define FSL_XCVR_EXT_CTRL_DMA_WR_DIS	BIT(24)
#define FSL_XCVR_EXT_CTRL_DMA_DIS(t)	(t ? BIT(24) : BIT(25))
#define FSL_XCVR_EXT_CTRL_SPDIF_MODE	BIT(23)
#define FSL_XCVR_EXT_CTRL_SLEEP_MODE	BIT(21)

#define FSL_XCVR_EXT_CTRL_TX_FWM_SHFT	0
#define FSL_XCVR_EXT_CTRL_TX_FWM_MASK	GENMASK(6, 0)
#define FSL_XCVR_EXT_CTRL_TX_FWM(i)	(((i) << FSL_XCVR_EXT_CTRL_TX_FWM_SHFT) \
					  & FSL_XCVR_EXT_CTRL_TX_FWM_MASK)
#define FSL_XCVR_EXT_CTRL_RX_FWM_SHFT	8
#define FSL_XCVR_EXT_CTRL_RX_FWM_MASK	GENMASK(14, 8)
#define FSL_XCVR_EXT_CTRL_RX_FWM(i)	(((i) << FSL_XCVR_EXT_CTRL_RX_FWM_SHFT) \
					  & FSL_XCVR_EXT_CTRL_RX_FWM_MASK)
#define FSL_XCVR_EXT_CTRL_PAGE_SHFT	16
#define FSL_XCVR_EXT_CTRL_PAGE_MASK	GENMASK(19, 16)
#define FSL_XCVR_EXT_CTRL_PAGE(i)	(((i) << FSL_XCVR_EXT_CTRL_PAGE_SHFT) \
					  & FSL_XCVR_EXT_CTRL_PAGE_MASK)

#define FSL_XCVR_EXT_STUS_NT_FIFO_ENTR	GENMASK(7, 0)
#define FSL_XCVR_EXT_STUS_NR_FIFO_ENTR	GENMASK(15, 8)
#define FSL_XCVR_EXT_STUS_CM0_SLEEPING	BIT(16)
#define FSL_XCVR_EXT_STUS_CM0_DEEP_SLP	BIT(17)
#define FSL_XCVR_EXT_STUS_CM0_SLP_HACK	BIT(18)
#define FSL_XCVR_EXT_STUS_RX_CMDC_RSTO	BIT(23)
#define FSL_XCVR_EXT_STUS_TX_CMDC_RSTO	BIT(24)
#define FSL_XCVR_EXT_STUS_RX_CMDC_COTO	BIT(25)
#define FSL_XCVR_EXT_STUS_TX_CMDC_COTO	BIT(26)
#define FSL_XCVR_EXT_STUS_HB_STATUS	BIT(27)
#define FSL_XCVR_EXT_STUS_NEW_UD4_REC	BIT(28)
#define FSL_XCVR_EXT_STUS_NEW_UD5_REC	BIT(29)
#define FSL_XCVR_EXT_STUS_NEW_UD6_REC	BIT(30)
#define FSL_XCVR_EXT_STUS_HPD_INPUT	BIT(31)

#define FSL_XCVR_IRQ_NEW_CS		BIT(0)
#define FSL_XCVR_IRQ_NEW_UD		BIT(1)
#define FSL_XCVR_IRQ_MUTE		BIT(2)
#define FSL_XCVR_IRQ_CMDC_RESP_TO	BIT(3)
#define FSL_XCVR_IRQ_ECC_ERR		BIT(4)
#define FSL_XCVR_IRQ_PREAMBLE_MISMATCH	BIT(5)
#define FSL_XCVR_IRQ_FIFO_UOFL_ERR	BIT(6)
#define FSL_XCVR_IRQ_HOST_WAKEUP	BIT(7)
#define FSL_XCVR_IRQ_HOST_OHPD		BIT(8)
#define FSL_XCVR_IRQ_DMAC_NO_DATA_REC	BIT(9)
#define FSL_XCVR_IRQ_DMAC_FMT_CHG_DET	BIT(10)
#define FSL_XCVR_IRQ_HB_STATE_CHG	BIT(11)
#define FSL_XCVR_IRQ_CMDC_STATUS_UPD	BIT(12)
#define FSL_XCVR_IRQ_TEMP_UPD		BIT(13)
#define FSL_XCVR_IRQ_DMA_RD_REQ		BIT(14)
#define FSL_XCVR_IRQ_DMA_WR_REQ		BIT(15)
#define FSL_XCVR_IRQ_DMAC_BME_BIT_ERR	BIT(16)
#define FSL_XCVR_IRQ_PREAMBLE_MATCH	BIT(17)
#define FSL_XCVR_IRQ_M_W_PRE_MISMATCH	BIT(18)
#define FSL_XCVR_IRQ_B_PRE_MISMATCH	BIT(19)
#define FSL_XCVR_IRQ_UNEXP_PRE_REC	BIT(20)
#define FSL_XCVR_IRQ_ARC_MODE		BIT(21)
#define FSL_XCVR_IRQ_CH_UD_OFLOW	BIT(22)
#define FSL_XCVR_IRQ_EARC_ALL		(FSL_XCVR_IRQ_NEW_CS | \
					 FSL_XCVR_IRQ_NEW_UD | \
					 FSL_XCVR_IRQ_MUTE | \
					 FSL_XCVR_IRQ_FIFO_UOFL_ERR | \
					 FSL_XCVR_IRQ_HOST_WAKEUP | \
					 FSL_XCVR_IRQ_ARC_MODE)

#define FSL_XCVR_ISR_CMDC_TX_EN		BIT(3)
#define FSL_XCVR_ISR_HPD_TGL		BIT(15)
#define FSL_XCVR_ISR_DMAC_SPARE_INT	BIT(19)
#define FSL_XCVR_ISR_SET_SPDIF_RX_INT	BIT(20)
#define FSL_XCVR_ISR_SET_SPDIF_TX_INT	BIT(21)
#define FSL_XCVR_ISR_SET_SPDIF_MODE(t)	(t ? BIT(21) : BIT(20))
#define FSL_XCVR_ISR_SET_ARC_CM_INT	BIT(22)
#define FSL_XCVR_ISR_SET_ARC_SE_INT	BIT(23)

#define FSL_XCVR_PHY_AI_ADDR_MASK	GENMASK(7, 0)
#define FSL_XCVR_PHY_AI_RESETN		BIT(15)
#define FSL_XCVR_PHY_AI_TOG_PLL		BIT(24)
#define FSL_XCVR_PHY_AI_TOG_DONE_PLL	BIT(25)
#define FSL_XCVR_PHY_AI_TOG_PHY		BIT(26)
#define FSL_XCVR_PHY_AI_TOG_DONE_PHY	BIT(27)
#define FSL_XCVR_PHY_AI_RW_MASK		BIT(31)

#define FSL_XCVR_RX_DPTH_CTRL_PAPB_FIFO_STATUS	BIT(0)
#define FSL_XCVR_RX_DPTH_CTRL_DIS_PRE_ERR_CHK	BIT(1)
#define FSL_XCVR_RX_DPTH_CTRL_DIS_NOD_REC_CHK	BIT(2)
#define FSL_XCVR_RX_DPTH_CTRL_ECC_VUC_BIT_CHK	BIT(3)
#define FSL_XCVR_RX_DPTH_CTRL_EN_CMP_PAR_CALC	BIT(4)
#define FSL_XCVR_RX_DPTH_CTRL_RST_PKT_CNT_FIFO	BIT(5)
#define FSL_XCVR_RX_DPTH_CTRL_STORE_FMT		BIT(6)
#define FSL_XCVR_RX_DPTH_CTRL_EN_PAR_CALC	BIT(7)
#define FSL_XCVR_RX_DPTH_CTRL_UDR		BIT(8)
#define FSL_XCVR_RX_DPTH_CTRL_CSR		BIT(9)
#define FSL_XCVR_RX_DPTH_CTRL_UDA		BIT(10)
#define FSL_XCVR_RX_DPTH_CTRL_CSA		BIT(11)
#define FSL_XCVR_RX_DPTH_CTRL_CLR_RX_FIFO	BIT(12)
#define FSL_XCVR_RX_DPTH_CTRL_DIS_B_PRE_ERR_CHK	BIT(13)
#define FSL_XCVR_RX_DPTH_CTRL_PABS		BIT(19)
#define FSL_XCVR_RX_DPTH_CTRL_DTS_CDS		BIT(20)
#define FSL_XCVR_RX_DPTH_CTRL_BLKC		BIT(21)
#define FSL_XCVR_RX_DPTH_CTRL_MUTE_CTRL		BIT(22)
#define FSL_XCVR_RX_DPTH_CTRL_MUTE_MODE		BIT(23)
#define FSL_XCVR_RX_DPTH_CTRL_FMT_CHG_CTRL	BIT(24)
#define FSL_XCVR_RX_DPTH_CTRL_FMT_CHG_MODE	BIT(25)
#define FSL_XCVR_RX_DPTH_CTRL_LAYB_CTRL		BIT(26)
#define FSL_XCVR_RX_DPTH_CTRL_LAYB_MODE		BIT(27)
#define FSL_XCVR_RX_DPTH_CTRL_PRC		BIT(28)
#define FSL_XCVR_RX_DPTH_CTRL_COMP		BIT(29)
#define FSL_XCVR_RX_DPTH_CTRL_FSM		GENMASK(31, 30)

#define FSL_XCVR_TX_DPTH_CTRL_CS_ACK		BIT(0)
#define FSL_XCVR_TX_DPTH_CTRL_UD_ACK		BIT(1)
#define FSL_XCVR_TX_DPTH_CTRL_CS_MOD		BIT(2)
#define FSL_XCVR_TX_DPTH_CTRL_UD_MOD		BIT(3)
#define FSL_XCVR_TX_DPTH_CTRL_VLD_MOD		BIT(4)
#define FSL_XCVR_TX_DPTH_CTRL_FRM_VLD		BIT(5)
#define FSL_XCVR_TX_DPTH_CTRL_EN_PARITY		BIT(6)
#define FSL_XCVR_TX_DPTH_CTRL_EN_PREAMBLE	BIT(7)
#define FSL_XCVR_TX_DPTH_CTRL_EN_ECC_INTER	BIT(8)
#define FSL_XCVR_TX_DPTH_CTRL_BYPASS_FEM	BIT(10)
#define FSL_XCVR_TX_DPTH_CTRL_FRM_FMT		BIT(11)
#define FSL_XCVR_TX_DPTH_CTRL_STRT_DATA_TX	BIT(14)
#define FSL_XCVR_TX_DPTH_CTRL_ADD_CYC_TX_OE_STR	BIT(15)
#define FSL_XCVR_TX_DPTH_CTRL_ADD_CYC_TX_OE_END	BIT(16)
#define FSL_XCVR_TX_DPTH_CTRL_CLK_RATIO		BIT(29)
#define FSL_XCVR_TX_DPTH_CTRL_TM_NO_PRE_BME	GENMASK(31, 30)

#define FSL_XCVR_PHY_AI_CTRL_AI_RESETN		BIT(15)

#define FSL_XCVR_PLL_CTRL0			0x00
#define FSL_XCVR_PLL_CTRL0_SET			0x04
#define FSL_XCVR_PLL_CTRL0_CLR			0x08
#define FSL_XCVR_PLL_NUM			0x20
#define FSL_XCVR_PLL_DEN			0x30
#define FSL_XCVR_PLL_PDIV			0x40
#define FSL_XCVR_PLL_BANDGAP_SET		0x54
#define FSL_XCVR_PHY_CTRL			0x00
#define FSL_XCVR_PHY_CTRL_SET			0x04
#define FSL_XCVR_PHY_CTRL_CLR			0x08
#define FSL_XCVR_PHY_CTRL2			0x70
#define FSL_XCVR_PHY_CTRL2_SET			0x74
#define FSL_XCVR_PHY_CTRL2_CLR			0x78

#define FSL_XCVR_PLL_BANDGAP_EN_VBG		BIT(0)
#define FSL_XCVR_PLL_CTRL0_HROFF		BIT(13)
#define FSL_XCVR_PLL_CTRL0_PWP			BIT(14)
#define FSL_XCVR_PLL_CTRL0_CM0_EN		BIT(24)
#define FSL_XCVR_PLL_CTRL0_CM1_EN		BIT(25)
#define FSL_XCVR_PLL_CTRL0_CM2_EN		BIT(26)
#define FSL_XCVR_PLL_PDIVx(v, i)		((v & 0x7) << (4 * i))

#define FSL_XCVR_PHY_CTRL_PHY_EN		BIT(0)
#define FSL_XCVR_PHY_CTRL_RX_CM_EN		BIT(1)
#define FSL_XCVR_PHY_CTRL_TSDIFF_OE		BIT(5)
#define FSL_XCVR_PHY_CTRL_SPDIF_EN		BIT(8)
#define FSL_XCVR_PHY_CTRL_ARC_MODE_SE_EN	BIT(9)
#define FSL_XCVR_PHY_CTRL_ARC_MODE_CM_EN	BIT(10)
#define FSL_XCVR_PHY_CTRL_TX_CLK_MASK		GENMASK(26, 25)
#define FSL_XCVR_PHY_CTRL_TX_CLK_HDMI_SS	BIT(25)
#define FSL_XCVR_PHY_CTRL_TX_CLK_AUD_SS		BIT(26)
#define FSL_XCVR_PHY_CTRL2_EARC_TXMS		BIT(14)

#define FSL_XCVR_CS_DATA_0_FS_MASK		GENMASK(31, 24)
#define FSL_XCVR_CS_DATA_0_FS_32000		0x3000000
#define FSL_XCVR_CS_DATA_0_FS_44100		0x0000000
#define FSL_XCVR_CS_DATA_0_FS_48000		0x2000000
#define FSL_XCVR_CS_DATA_0_FS_64000		0xB000000
#define FSL_XCVR_CS_DATA_0_FS_88200		0x8000000
#define FSL_XCVR_CS_DATA_0_FS_96000		0xA000000
#define FSL_XCVR_CS_DATA_0_FS_176400		0xC000000
#define FSL_XCVR_CS_DATA_0_FS_192000		0xE000000

#define FSL_XCVR_CS_DATA_0_CH_MASK		0x3A
#define FSL_XCVR_CS_DATA_0_CH_U2LPCM		0x00
#define FSL_XCVR_CS_DATA_0_CH_UMLPCM		0x20
#define FSL_XCVR_CS_DATA_0_CH_U1BAUD		0x30

#define FSL_XCVR_CS_DATA_1_CH_MASK		0xF000
#define FSL_XCVR_CS_DATA_1_CH_2			0x0000
#define FSL_XCVR_CS_DATA_1_CH_8			0x7000
#define FSL_XCVR_CS_DATA_1_CH_16		0xB000
#define FSL_XCVR_CS_DATA_1_CH_32		0x3000

/* Data memory structures */
#define FSL_XCVR_RX_CS_CTRL_0		0x20 /* First  RX CS control register */
#define FSL_XCVR_RX_CS_CTRL_1		0x24 /* Second RX CS control register */
#define FSL_XCVR_RX_CS_BUFF_0		0x80 /* First  RX CS buffer */
#define FSL_XCVR_RX_CS_BUFF_1		0xA0 /* Second RX CS buffer */
#define FSL_XCVR_CAP_DATA_STR		0x300 /* Capabilities data structure */

#define FSL_XCVR_CAPDS_SIZE	256
#define SPDIF_NUM_RATES 7

struct fsl_xcvr_soc_data {
	const char *fw_name;
	bool spdif_only;
	bool use_edma;
};

struct fsl_xcvr {
	const struct fsl_xcvr_soc_data *soc_data;
	struct platform_device *pdev;
	struct regmap *regmap;
	struct clk *ipg_clk;
	struct clk *pll_ipg_clk;
	struct clk *phy_clk;
	struct clk *spba_clk;
	struct clk *pll8k_clk;
	struct clk *pll11k_clk;
	struct reset_control *reset;
	u8 streams;
	u32 mode;
	u32 arc_mode;
	void __iomem *ram_addr;
	struct snd_dmaengine_dai_dma_data dma_prms_rx;
	struct snd_dmaengine_dai_dma_data dma_prms_tx;
	struct snd_aes_iec958 rx_iec958;
	struct snd_aes_iec958 tx_iec958;
	u8 cap_ds[FSL_XCVR_CAPDS_SIZE];
	struct snd_pcm_hw_constraint_list spdif_constr_rates;
	u32 spdif_constr_rates_list[SPDIF_NUM_RATES];
};

const struct attribute_group *fsl_xcvr_get_attr_grp(void);

#endif /* __FSL_XCVR_H */
