/*
 * Copyright (C) 2015 MediaTek Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#ifndef BUILD_LK
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>

#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/miscdevice.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/kobject.h>
#include <linux/platform_device.h>
#include <linux/atomic.h>

#include <linux/module.h>
#include <linux/backlight.h>
#include <video/Backlight_I2C_map.h>
#include <video/sgm37604a.h>

/* I2C Slave Setting */
#define sgm37604a_SLAVE_ADDR_WRITE	0x36

//static struct i2c_client *new_client;
static unsigned short s_last_backlight_level = 0;
static int sgm37604a_driver_probe(struct i2c_client *client,
	const struct i2c_device_id *id);
static int sgm37604a_driver_remove(struct i2c_client *client);

static const struct i2c_device_id sgm37604a_i2c_id[] = {
	{"sgm37604a", 0},
	{},
};

static const struct of_device_id sgm37604a_match_table[] = {
	{.compatible = "qcom,sgm37604a"},
	{},
};

MODULE_DEVICE_TABLE(of, sgm37604a_match_table);
MODULE_DEVICE_TABLE(i2c, sgm37604a_i2c_id);


static struct i2c_driver sgm37604a_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "sgm37604a",
		.of_match_table = sgm37604a_match_table,
	},
	.probe = sgm37604a_driver_probe,
	.remove = sgm37604a_driver_remove,
	.id_table = sgm37604a_i2c_id,
};

static DEFINE_MUTEX(sgm37604a_i2c_access);

/* I2C Function For Read/Write */
int sgm37604a_read_byte(struct sgm37604a *sgm, unsigned char cmd, unsigned char *returnData)
{
	char cmd_buf[2] = { 0x00, 0x00 };
	char readData = 0;
	int ret = 0;

	mutex_lock(&sgm37604a_i2c_access);

	cmd_buf[0] = cmd;
	ret = i2c_master_send(sgm->client, &cmd_buf[0], 1);
	ret = i2c_master_recv(sgm->client, &cmd_buf[1], 1);
	if (ret < 0) {
		mutex_unlock(&sgm37604a_i2c_access);
		return 0;
	}

	readData = cmd_buf[1];
	*returnData = readData;

	mutex_unlock(&sgm37604a_i2c_access);

	return 1;
}

int sgm37604a_write_byte(struct sgm37604a *sgm, unsigned char cmd, unsigned char writeData)
{
	unsigned char write_data[2] = { 0 };
	int ret = 0;

	mutex_lock(&sgm37604a_i2c_access);
	write_data[0] = cmd;
	write_data[1] = writeData;

	ret = i2c_master_send(sgm->client, write_data, 2);
	if (ret < 0) {
		mutex_unlock(&sgm37604a_i2c_access);
		pr_err("[%s] I2C write fail!!!\n", __func__);
		return 0;
	}

	mutex_unlock(&sgm37604a_i2c_access);

	return 1;
}

static void sgm37604a_parse_dt(struct device *dev, struct sgm37604a *sgm)
{
	struct device_node *np = dev->of_node;
	int ret;

	sgm->en_gpio = of_get_named_gpio(np, "lcm-bl-en-gpio", 0);
	if (gpio_is_valid(sgm->en_gpio)) {
		ret = gpio_request(sgm->en_gpio, "lcm-bl-en-gpio");
		if (ret < 0) {
			pr_err("[%s]failed to request lcm-bl-en-gpio\n", __func__);
		}
		ret = gpio_direction_output(sgm->en_gpio, 1);
		if (ret) {
			pr_err("[%s]unable to set direction for gpio [%d]\n", __func__, sgm->en_gpio);
		}
	}
}

int sgm37604a_backlight_on_enable(struct i2c_client *client)
{
	struct sgm37604a *sgm ;
	int rc;

	pr_info("[%s] enter \n", __func__);
	if(!client) {
		pr_err("[%s]: not found IIC client\n", __func__);
		return 0;
	}
	sgm = i2c_get_clientdata(client);
	if(gpio_is_valid(sgm->en_gpio)) {
		gpio_set_value(sgm->en_gpio, 1);
	}
	rc = sgm37604a_write_byte(sgm,0x11,0x00);
	return rc;
}

int sgm37604a_backlight_off_disable(struct i2c_client *client)
{
	struct sgm37604a *sgm ;
	int rc = 1;

	pr_info("[%s] enter \n", __func__);
	if(!client) {
		pr_err("[%s] : not found IIC client\n", __func__);
		return 0;
	}
	sgm= i2c_get_clientdata(client);
	if(gpio_is_valid(sgm->en_gpio)) {
		gpio_set_value(sgm->en_gpio, 0);
	}
	s_last_backlight_level = 0;
	return rc;
}

void sgm37604a_backlight_on_envent(struct device_node *node)
{
	struct i2c_client *client = of_find_i2c_device_by_node(node);
	pr_info("[%s] enter \n", __func__);
	sgm37604a_backlight_on_enable(client);
}
EXPORT_SYMBOL(sgm37604a_backlight_on_envent);

void sgm37604a_backlight_off_envent(struct device_node *node)
{
	struct i2c_client *client = of_find_i2c_device_by_node(node);
	pr_info("[%s] enter \n", __func__);
	sgm37604a_backlight_off_disable(client);
}
EXPORT_SYMBOL(sgm37604a_backlight_off_envent);

int sgm37604a_backlight_set(struct backlight_device *bl)
{
	struct sgm37604a *sgm = bl_get_data(bl);
	int brightness = (unsigned int)bl->props.brightness;
	int sgm_level = 0;
	if(brightness > 255)
		brightness = 255;
	sgm_level = backlight_i2c_map[brightness];

	pr_info("[%s]hx83102p backlight: level = %d sgm_level = %d\n", __func__,brightness,sgm_level);
	if (brightness == s_last_backlight_level) {
		pr_info("[%s]:same brightness,not setting!brightness = %d,sgm_level = %d\n", __func__, brightness, sgm_level);
		return 0;
	} else {
		sgm37604a_write_byte(sgm, 0x1A,(backlight_i2c_map[brightness] & 0x0F));
		sgm37604a_write_byte(sgm, 0x19,(backlight_i2c_map[brightness] >> 4));
		s_last_backlight_level = brightness;
	}
	return 0;
}


static struct backlight_ops sgm37604a_backlight_ops = {
	.options = BL_CORE_SUSPENDRESUME,
	.update_status = sgm37604a_backlight_set,
};

static int sgm37604a_driver_probe(struct i2c_client *client,
	const struct i2c_device_id *id)
{
	struct sgm37604a *sgm;
	struct backlight_properties props;

	pr_info("[%s]Enter\n", __func__);
	sgm = devm_kzalloc(&client->dev, sizeof(struct sgm37604a), GFP_KERNEL);
	if (!sgm) {
		pr_err("Out of memory\n");
		return -ENOMEM;
	}
	
	memset(&props, 0, sizeof(struct backlight_properties));
	props.type = BACKLIGHT_RAW; // 设置背光类型为原始类型
	props.brightness = 255;
	props.max_brightness = 255;

	sgm37604a_parse_dt(&client->dev, sgm);
	mutex_init(&sgm->i2c_rw_lock);

	sgm->bl= devm_backlight_device_register(&client->dev, "sgm37604a",&client->dev, sgm, &sgm37604a_backlight_ops,&props);
	if (IS_ERR(sgm->bl)) {
		dev_err(&client->dev, "failed to register backlight device\n"); 
		return PTR_ERR(sgm->bl); 
	}
	
	sgm->dev = &client->dev;
	sgm->client = client;
	i2c_set_clientdata(client, sgm);
	mutex_init(&sgm->i2c_rw_lock);
	pr_info("[%s]End\n", __func__);

	return 0;

}

static int sgm37604a_driver_remove(struct i2c_client *client)
{
	pr_info("[%s]Enter\n", __func__);

	//new_client = NULL;
	i2c_unregister_device(client);

	return 0;
}

#define sgm37604a_BUSNUM 5

static int __init sgm37604a_init(void)
{
	pr_info("[%s]Enter\n", __func__);

	if (i2c_add_driver(&sgm37604a_driver) != 0)
		pr_info("[%s] failed to register sgm37604a i2c driver.\n", __func__);
	else
		pr_info("[%s] Success to register sgm37604a i2c driver.\n", __func__);


	return 0;
}

static void __exit sgm37604a_exit(void)
{
	i2c_del_driver(&sgm37604a_driver);
}

module_init(sgm37604a_init);
module_exit(sgm37604a_exit);

MODULE_DESCRIPTION("SGM37604A BACKLIGHT Driver");
MODULE_LICENSE("GPL");

#endif
