#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <video/lcd_bias.h>

static int lcd_bias_write_reg(struct lcd_bais_data *bias, u8 reg, u8 val)
{
	int ret;
	if(!bias || !bias->client) {
		pr_info("%s: IIC is error!", __func__);
		return -EINVAL;
	}

	mutex_lock(&bias->i2c_rw_lock);
	ret = i2c_smbus_write_byte_data(bias->client, reg, val);
	mutex_unlock(&bias->i2c_rw_lock);
	if (ret < 0) {
		pr_err("%s: i2c write fail: can't write 0x%02X to reg 0x%02X: %d\n",
				__func__, val, reg, ret);
		return ret;
	}
	return ret;
}

static int lcd_bias_read_reg(struct lcd_bais_data *bias, int reg)
{
	int value;
	if(!bias || !bias->client) {
		pr_info("%s: IIC is error!", __func__);
		return -EINVAL;
	}

	mutex_lock(&bias->i2c_rw_lock);
	value = i2c_smbus_read_byte_data(bias->client, reg);
	mutex_unlock(&bias->i2c_rw_lock);
	if (value < 0) {
		pr_err("%s: i2c read fail: can't read reg 0x%02X: %d\n",
				__func__, reg, value);
		return value;
	}

	pr_info("%s reg=%02X, value=%02X\n", __func__, reg, value);
	return value;
}

static void lcd_bias_set_vpn(struct lcd_bais_data *bias, unsigned int en, unsigned int select)
{
    int level = 6000;
	pr_debug("%s:enter\n", __func__);
    if (select == sm5109c_bias) {
		pr_info("%s:devices is sm5109c_bias\n", __func__);
		level = (level - 4000) / 100;  //eg.  5.0V= 4.0V + Hex 0x0A (Bin 0 1010) * 100mV
		pr_info("%s:sm5109c_bias  level= %d\n", __func__,level);
		 if (en) {		
            lcd_bias_write_reg(bias, LCD_BIAS_POSCNTL_REG, (u8)level);
            lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENP1);
            mdelay(1);
            lcd_bias_write_reg(bias, LCD_BIAS_NEGCNTL_REG, (u8)level);
            mdelay(5);
            lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENN1);
        } else {
            lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENP0);
            mdelay(5);
            lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENN0);
        }
    } else if (select == ocp2131_bias) {
		pr_info("%s:ocp2131_bias\n", __func__);
		if(en) {
			lcd_bias_write_reg(bias, OCP2131_Positive_Output, OCP2131_Set_voltage);
			lcd_bias_write_reg(bias, OCP2131_Negative_Output, OCP2131_Set_voltage);
			lcd_bias_write_reg(bias, OCP2131_Set_vsp_vsn_enable, OCP2131_Set_value);
                        lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENP1);
                        udelay(5000);
                        lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENN1);
		} else {
			lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENP0);
			udelay(5000);
			lcd_bias_set_state(bias, LCD_BIAS_GPIO_STATE_ENN0);
		}
	}
}

static long lcd_bias_set_state(struct lcd_bais_data *bias, unsigned int s)
{
    int ret = 0;
	BUG_ON(!((unsigned int)(s) < (unsigned int)(LCD_BIAS_GPIO_STATE_MAX)));

    switch (s) {
	case LCD_BIAS_GPIO_STATE_ENP0:	
		gpio_set_value(bias->enp_gpio, 0);
		break;
	case LCD_BIAS_GPIO_STATE_ENP1:
		gpio_set_value(bias->enp_gpio, 1);
		break;
	case LCD_BIAS_GPIO_STATE_ENN0:	
		gpio_set_value(bias->enn_gpio, 0);
		break;
	case LCD_BIAS_GPIO_STATE_ENN1:
		gpio_set_value(bias->enn_gpio, 1);
		break;		
	default:
		return -EINVAL;
	}

    return ret; /* Good! */
}

static void lcd_bias_parse_dt(struct lcd_bais_data *bias)
{
	int ret;

	bias->enp_gpio = of_get_named_gpio(bias->dev->of_node, "qualcomm,enp", 0);
	pr_debug("enp_gpio: %d\n", bias->enp_gpio);

	bias->enn_gpio = of_get_named_gpio(bias->dev->of_node, "qualcomm,enn", 0);
	pr_debug("enn_gpio: %d\n", bias->enn_gpio);

	if (gpio_is_valid(bias->enp_gpio)) {
		pr_debug("enp_gpio is valid \n");
		ret = gpio_request(bias->enp_gpio, "lcm-enp-gpio");
		if (ret < 0) {
			pr_err("failed to request lcm-enp-gpio\n");
		}
		ret = gpio_direction_output(bias->enp_gpio, 1);
		if (ret) {
			pr_err("unable to set direction for gpio [%d]\n", bias->enp_gpio);
		}
	}
	if (gpio_is_valid(bias->enn_gpio)) {
		pr_debug("enn_gpio is valid \n");
		ret = gpio_request( bias->enn_gpio, "lcm-enn-gpio");
		if (ret < 0) {
			pr_err("failed to request lcm-enn-gpio\n");//????
		}
		ret = gpio_direction_output(bias->enn_gpio, 1);
		if (ret) {
			pr_err("unable to set direction for gpio [%d]\n", bias->enp_gpio);
		}
	}
}



static int lcd_bias_probe(struct i2c_client *client,
							const struct i2c_device_id *id)
{
	struct lcd_bais_data *bias;
	unsigned char ocp_chip_id;
	unsigned char ocp2132_id = 0x0D;;
	pr_info("Enter lcd_bias_probe!\n");
	bias = devm_kzalloc(&client->dev, sizeof(struct lcd_bais_data), GFP_KERNEL);
	if (!bias) {
		pr_err("Out of memory\n");
		return -ENOMEM;
	}
	bias->dev = &client->dev;
	bias->client = client;
	i2c_set_clientdata(client, bias);
	lcd_bias_parse_dt(bias);
	mutex_init(&bias->i2c_rw_lock);
	ocp_chip_id = lcd_bias_read_reg(bias, OCP2312_chip_id);
	if (ocp_chip_id == ocp2132_id) {
		bias->select_bias_flag = ocp2131_bias;
		pr_info("%s: is ocp2131 bias,chip_id = %02x\n", __func__, ocp_chip_id);
	} else {
		bias->select_bias_flag = sm5109c_bias;
		pr_info("%s: is sm5109c bias\n", __func__);
	}
	lcd_bias_set_vpn(bias, 1, bias->select_bias_flag);
	return 0;
}

static int lcd_bias_remove(struct i2c_client *client)
{
	struct lcd_bais_data *bias = i2c_get_clientdata(client);
	mutex_destroy(&bias->i2c_rw_lock);
	pr_info("Enter lcd_bias_remove\n");
	return 0;
}

/*****************************************************************************
 * * i2c driver configuration
 * *****************************************************************************/
static const struct i2c_device_id lcd_bias_id[] = {
	{"lcd_bias", 0},
	{},
};
static const struct of_device_id lcd_bias_match_table[] = {
	{.compatible = "qualcomm,i2c_lcd_bias"},
	{},
};

MODULE_DEVICE_TABLE(of, lcd_bias_match_table);
MODULE_DEVICE_TABLE(i2c, lcd_bias_id);
static struct i2c_driver lcd_bias_driver = {
	.driver = {
		.name = "lcd_bias",
		.owner 	= THIS_MODULE,
		.of_match_table = lcd_bias_match_table,
	},
	.id_table = lcd_bias_id,
	.probe = lcd_bias_probe,
	.remove = lcd_bias_remove,
};

static int __init lcd_bias_init(void)
{
	int ret = 0;
	ret = i2c_add_driver(&lcd_bias_driver);
	if (ret != 0) {
		pr_err("lcd bias driver init failed!");
		return ret;
	} else {
		pr_info("%s Complete\n", __func__);
	}
	return ret;
}

static void __exit lcd_bias_exit(void)
{
	i2c_del_driver(&lcd_bias_driver);
}

module_init(lcd_bias_init);
module_exit(lcd_bias_exit);

MODULE_AUTHOR("OCP");
MODULE_DESCRIPTION("OCP2131 Bias IC Driver");
MODULE_LICENSE("GPL v2");
