// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * briana CT7113 SMBus temperature sensor driver
 * Copyright (C) 2021 jaylin
 *
 * Based on:
 * Texas Instruments TMP103 SMBus temperature sensor driver
 *
 * Copyright (C) 2014 Heiko Schocher <hs@denx.de>
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/err.h>
#include <linux/mutex.h>
#include <linux/device.h>
#include <linux/jiffies.h>
#include <linux/regmap.h>
#include <linux/of_device.h>

#define CT7113_TEMP_REG		0x00
#define CT7113_CONF_REG		0x01
#define CT7113_TLOW_REG		0x02
#define CT7113_THIGH_REG	0x03

#define CT7113_CONF_M0		0x01
#define CT7113_CONF_M1		0x02
#define CT7113_CONF_LC		0x04
#define CT7113_CONF_FL		0x08
#define CT7113_CONF_FH		0x10
#define CT7113_CONF_CR0		0x20
#define CT7113_CONF_CR1		0x40
#define CT7113_CONF_ID		0x80
#define CT7113_CONF_SD		(CT7113_CONF_M1)
#define CT7113_CONF_SD_MASK	(CT7113_CONF_M0 | CT7113_CONF_M1)

#define CT7113_CONFIG		(CT7113_CONF_CR1 | CT7113_CONF_M1)
#define CT7113_CONFIG_MASK	(CT7113_CONF_CR0 | CT7113_CONF_CR1 | \
				 CT7113_CONF_M0 | CT7113_CONF_M1)

static inline int CT7113_reg_to_mc(s8 val)
{
	return val * 1000;
}

static inline u8 CT7113_mc_to_reg(int val)
{
	return DIV_ROUND_CLOSEST(val, 1000);
}

static ssize_t CT7113_temp_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct sensor_device_attribute *sda = to_sensor_dev_attr(attr);
	struct regmap *regmap = dev_get_drvdata(dev);
	unsigned int regval;
	int ret;
	int temp_read_retry = 3;

	while(temp_read_retry--) {
		dev_err(dev, "%s-temp_read_retry:%d\n", __func__, temp_read_retry);
		mdelay(20);
		ret = regmap_read(regmap, sda->index, &regval);
		if (ret < 0) {
			dev_err(dev, "regmap_read error! ErrorCode:%d, read again!\n", ret);
			continue;
		}
		if (regval <= 0) {
			dev_err(dev, "read temp val error: %d, read again!\n", regval);
			continue;
		}
		break;
	}

	if (ret < 0) {
		dev_err(dev, "regmap_read error! ErrorCode:%d\n", ret);
		return sprintf(buf, "regmap_read error! ErrorCode:%d\n", ret);
	}

	dev_err(dev, "CT7117_temp_show:%d\n", CT7113_reg_to_mc(regval));

	return sprintf(buf, "temp:%d\n", CT7113_reg_to_mc(regval));
}

static ssize_t CT7113_temp_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	struct sensor_device_attribute *sda = to_sensor_dev_attr(attr);
	struct regmap *regmap = dev_get_drvdata(dev);
	long val;
	int ret;

	if (kstrtol(buf, 10, &val) < 0)
		return -EINVAL;

	val = clamp_val(val, -55000, 127000);
	ret = regmap_write(regmap, sda->index, CT7113_mc_to_reg(val));
	return ret ? ret : count;
}

static SENSOR_DEVICE_ATTR_RO(temp1_input, CT7113_temp, CT7113_TEMP_REG);

static SENSOR_DEVICE_ATTR_RW(temp1_min, CT7113_temp, CT7113_TLOW_REG);

static SENSOR_DEVICE_ATTR_RW(temp1_max, CT7113_temp, CT7113_THIGH_REG);

static struct attribute *CT7113_attrs[] = {
	&sensor_dev_attr_temp1_input.dev_attr.attr,
	&sensor_dev_attr_temp1_min.dev_attr.attr,
	&sensor_dev_attr_temp1_max.dev_attr.attr,
	NULL
};
ATTRIBUTE_GROUPS(CT7113);

static bool CT7113_regmap_is_volatile(struct device *dev, unsigned int reg)
{
	return reg == CT7113_TEMP_REG;
}

static const struct regmap_config CT7113_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = CT7113_THIGH_REG,
	.volatile_reg = CT7113_regmap_is_volatile,
};

static int CT7113_probe(struct i2c_client *client)
{
	struct device_node *np = client->dev.of_node;
	struct device *dev = &client->dev;
	struct device *hwmon_dev;
	const char *name;
	struct regmap *regmap;
	int ret;
	unsigned int regval;
	int temp_read_retry = 3;

	dev_err(&client->dev, "CT7117_probe start\n");

	regmap = devm_regmap_init_i2c(client, &CT7113_regmap_config);
	if (IS_ERR(regmap)) {
		dev_err(dev, "failed to allocate register map\n");
		return PTR_ERR(regmap);
	}
	i2c_set_clientdata(client, regmap);
	name = of_get_property(np, "label", NULL) ? : client->name;
	hwmon_dev = devm_hwmon_device_register_with_groups(dev, name,
						      regmap, CT7113_groups);
	if (IS_ERR(hwmon_dev)) {
		dev_err(&client->dev, "failed to register hwmon\n");
		return PTR_ERR(hwmon_dev);
	}
	dev_err(&client->dev, "CT7117_probe successfully\n");

	ret = regmap_update_bits(regmap, CT7113_CONF_REG,
				  CT7113_CONF_CR0, 0x1);
	if (ret < 0)
		dev_err(dev, "regmap_update_bits error! ErrorCode:%d\n", ret);

	dev_err(dev, "CT7117_temp_show\n");

	while(temp_read_retry--) {
		dev_err(dev, "%s-temp_read_retry:%d\n", __func__, temp_read_retry);
		mdelay(20);
		ret = regmap_read(regmap, CT7113_TEMP_REG, &regval);
		if (ret < 0) {
			dev_err(dev, "regmap_read error! ErrorCode:%d, read again!\n", ret);
			continue;
		}
		if (regval <= 0) {
			dev_err(dev, "read temp val error: %d, read again!\n", regval);
			continue;
		}
		break;
	}

	dev_err(dev, "temp00:%d\n", regval);
	dev_err(dev, "temp:%d\n", CT7113_reg_to_mc(regval));



	return PTR_ERR_OR_ZERO(hwmon_dev);
}

static int __maybe_unused CT7113_suspend(struct device *dev)
{
	struct regmap *regmap = dev_get_drvdata(dev);

	return regmap_update_bits(regmap, CT7113_CONF_REG,
				  CT7113_CONF_SD_MASK, 0);
}

static int __maybe_unused CT7113_resume(struct device *dev)
{
	struct regmap *regmap = dev_get_drvdata(dev);

	return regmap_update_bits(regmap, CT7113_CONF_REG,
				  CT7113_CONF_SD_MASK, CT7113_CONF_SD);
}

static SIMPLE_DEV_PM_OPS(CT7113_dev_pm_ops, CT7113_suspend, CT7113_resume);

static const struct i2c_device_id CT7113_id[] = {
	{ "CT7113", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, CT7113_id);

static const struct of_device_id __maybe_unused CT7113_of_match[] = {
	{ .compatible = "briana,CT7113" },
	{ },
};
MODULE_DEVICE_TABLE(of, CT7113_of_match);

static struct i2c_driver CT7113_driver = {
	.driver = {
		.name	= "CT7113",
		.of_match_table = of_match_ptr(CT7113_of_match),
		.pm	= &CT7113_dev_pm_ops,
	},
	.probe_new	= CT7113_probe,
	.id_table	= CT7113_id,
};

module_i2c_driver(CT7113_driver);

MODULE_AUTHOR("Heiko Schocher <hs@denx.de>");
MODULE_DESCRIPTION("Texas Instruments CT7113 temperature sensor driver");
MODULE_LICENSE("GPL");
