#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/kthread.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>
#include <linux/rfkill.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/suspend.h>
#include <linux/time.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/uaccess.h>

static struct class *pir_class;
static struct cdev *pir_cdevp = NULL;
static int pir_config_register = 0x02A6110;
static int pir_dev_major = 0;
static int pir_dev_minor = 0;
static int pir_wake_up_flag = 0;

bool check_flag = false;
int direct_link, serial_in;
int irq;
struct input_dev *input;
struct timer_list my_timer;

static void mytimerset(void);

void pir_wake_up_function(int wake_up_from_panel);

void pir_wake_up_function(int wake_up_from_panel)
{
    pir_wake_up_flag = wake_up_from_panel;
}
EXPORT_SYMBOL(pir_wake_up_function);

void my_timer_callback(struct timer_list *t)
{
    if (gpio_get_value(direct_link) == 1) {
        if (pir_wake_up_flag) {
            input_event(input,EV_KEY,KEY_POWER,1);
            input_sync(input);
            input_event(input,EV_KEY,KEY_POWER,0);
            input_sync(input);
            pir_wake_up_flag = 0;
        } else {
            input_event(input,EV_KEY,191,1);
            input_sync(input);
            input_event(input,EV_KEY,191,0);
            input_sync(input);
        }
        gpio_direction_output(direct_link,0);
        gpio_direction_input(direct_link);
    }
    mod_timer(&my_timer, jiffies + msecs_to_jiffies(1000));
}

static void mytimerset(void)
{
    timer_setup(&my_timer, my_timer_callback, 0);
    mod_timer(&my_timer, jiffies + msecs_to_jiffies(10000));
}

void write_serial_in(unsigned long regval)
{
    int i;
    unsigned long regmask = 0x1000000;
    unsigned char nextbit;

    for (i = 0; i < 25; i++) {
        nextbit = (regval & regmask) != 0;
        regmask >>= 1;
        gpio_set_value(serial_in,0);
        gpio_set_value(serial_in,1);
        gpio_set_value(serial_in,nextbit);
        udelay(120);
    }

    gpio_set_value(serial_in,0);
    udelay(650);
}

void read_direct_link(void)
{
    int i;
    int direct_link_value;
    int value;
    unsigned int uibitmask;
    unsigned long ulbitmask;
    unsigned long serial_in_status;

    gpio_direction_output(direct_link,1);
    udelay(140);

    uibitmask = 0x4000;
    direct_link_value = 0;
    for (i = 0; i < 15; i++) {
        gpio_direction_output(direct_link,0);
        gpio_direction_output(direct_link,1);
        gpio_direction_input(direct_link);
        udelay(3);
        value = gpio_get_value(direct_link);

        if (value)
            direct_link_value |= uibitmask;

        uibitmask>>=1;
    }

    ulbitmask = 0x1000000;
    serial_in_status = 0;
    for (i = 0; i < 25; i++) {
        gpio_direction_output(direct_link,0);
        gpio_direction_output(direct_link,1);
        gpio_direction_input(direct_link);
        udelay(3);
        value = gpio_get_value(direct_link);

        if(value)
            serial_in_status |= ulbitmask;

        ulbitmask >>= 1;
    }

    printk("[PIR] direct_link=[%04X]", direct_link_value);
    printk("[PIR] serial_in=[%08X]", (unsigned int)serial_in_status);

    if (serial_in_status == pir_config_register) {
        check_flag = true;
    } else {
        check_flag = false;
    }
}

struct file_operations pir_gpio_fops = {
    .owner = THIS_MODULE
};

static const struct of_device_id pir_gpio_dt_ids[] = {
    {.compatible = "pir-sensor", },
    { /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, pir_gpio_dt_ids);

static int pir_probe(struct platform_device *pdev)
{
    dev_t devt;
    int i;
    int error;
    int ret;

    direct_link = of_get_named_gpio(pdev->dev.of_node, "pir_direct_link",0);
    gpio_request(direct_link, "PIR_DIRECT_LINK");
    serial_in = of_get_named_gpio(pdev->dev.of_node, "pir_ser_in",0);
    gpio_request_one(serial_in,GPIOF_DIR_OUT, "PIR_SERIAL_IN");
    gpio_set_value(serial_in, 0);
    gpio_direction_output(direct_link, 0);
    if (direct_link < 0 || serial_in < 0) {
        goto error_ret;
    }

    ret = alloc_chrdev_region(&devt, 0, 1, "pir_node");
    if (ret < 0) {
        goto error_ret;
    }

    pir_cdevp = kmalloc(sizeof(struct cdev), GFP_KERNEL);
    cdev_init(pir_cdevp, &pir_gpio_fops);

    pir_cdevp->owner = THIS_MODULE;
    pir_dev_major = MAJOR(devt);
    pir_dev_minor = MINOR(devt);

    ret = cdev_add(pir_cdevp, MKDEV(pir_dev_major, pir_dev_minor), 1);
    if (ret < 0) {
        goto error_ret;
    }

    //=====INPUT=====
    input = input_allocate_device();
    if (!input) {
        return -ENOMEM;
    }

    platform_set_drvdata(pdev, input);

    input->evbit[0] = BIT(EV_KEY);
    input->name = pdev->name;
    input->phys = "gpio-keys/input0";
    input->dev.parent = &pdev->dev;

    input->id.bustype = BUS_HOST;
    input->id.vendor = 0x0001;
    input->id.product = 0x0001;
    input->id.version = 0x0100;

    input_set_capability(input, EV_KEY, 191);
    input_set_capability(input, EV_KEY, KEY_POWER);

    error = input_register_device(input);
    if (error) {
        printk(KERN_ERR "Unable to register gpio-keys input device\n");
    }

    //=====INPUT=====
    pir_class = class_create(THIS_MODULE, "pir_node");
    if (IS_ERR(pir_class)) {
        printk("[pir_gpio] Fail : create gpio_class\n");
        goto error_cdev;
    } else {
        device_create(pir_class, NULL, MKDEV(pir_dev_major, pir_dev_minor), NULL, "pir_node");
    }
    printk(KERN_INFO "pir_sensor driver success loaded\n");

    for (i = 0; i < 10; i++){
        if (!check_flag) {
            write_serial_in(pir_config_register);
            read_direct_link();
        } else {
            break;
        }
    }

    if (check_flag) {
        printk("[PIR] Setting success!");
        mytimerset();
    } else {
        printk("[PIR] Setting failed!");
    }

    return 0;

error_cdev:
    kfree(pir_cdevp);

error_ret:
    return -1;
}

static int pir_remove(struct platform_device *pdev)
{
    gpio_free(direct_link);
    gpio_free(serial_in);
    kfree(pir_cdevp);

    return 0;
}

static struct platform_driver pir_driver = {
    .driver = {
        .name   = "pir",
        .owner  = THIS_MODULE,
        .of_match_table = pir_gpio_dt_ids,
    },
    .probe  = pir_probe,
    .remove = pir_remove,
};

static int __init pir_gpio_init(void)
{
    return platform_driver_register(&pir_driver);
}

static void __exit pir_gpio_exit(void)
{
    platform_driver_unregister(&pir_driver);
}

late_initcall(pir_gpio_init);
module_exit(pir_gpio_exit);

MODULE_AUTHOR("Wistron 2021/08/18");
MODULE_DESCRIPTION("PIR driver");
MODULE_LICENSE("GPL");
