#include <rtk_switch.h>
#include <rtk_error.h>
#include <l2.h>
#include <eee.h>
#include <vlan.h>
#include <rate.h>
#include <storm.h>
#include <trap.h>
#include <acl.h>
#include <qos.h>
#include <rtl8367c_asicdrv_phy.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/gn_low_level_api.h>
#include <linux/gn.h>
#include <linux/rtl83_switch_config.h>

gn_status_t rtl83_config_init(unsigned char *host_mac)
{
    rtk_vlan_cfg_t vlanCfg;
    rtk_mac_t hostMac;
    rtk_l2_ucastAddr_t l2Addr;
    rtk_uint32 retVal, i, eee_state = -1;
    rtk_uint32 ruleNum;
    rtk_filter_field_t field1;
    rtk_filter_cfg_t cfg;
    rtk_filter_action_t act;
    rtk_dscp_t dscp;
    rtk_uint32 port[2] = {1, 3};

    if (rtk_vlan_init() != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    /* Delete VLAN 1 */
    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    if (rtk_vlan_set(1, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    /* Create VLAN 0 */
    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    RTK_PORTMASK_ALLPORT_SET(vlanCfg.mbr);
    RTK_PORTMASK_ALLPORT_SET(vlanCfg.untag);
    if (rtk_vlan_set(0, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    /* PVID 0 */
    if (rtk_vlan_portPvid_set(EXT_PORT0, 0, 0) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    if (rtk_vlan_portPvid_set(UTP_PORT1, 0, 0) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    if (rtk_vlan_portPvid_set(UTP_PORT3, 0, 0) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    if (rtk_vlan_reservedVidAction_set(RESVID_ACTION_TAG, RESVID_ACTION_UNTAG) != 0)
    {
        ERMSG("rtk_vlan_reservedVidAction_set fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    /* Initial ACL */
    if ((retVal = rtk_filter_igrAcl_init()) != RT_ERR_OK)
    {
        ERMSG("rtk_filter_igrAcl_init fail");
        return retVal;
    }

    /* CDP ACL */
    /* CDP = 01:00:0C:CC:CC:CC */
    /* Reset Config & Action */
    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-00-0C-CC-CC-CC */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0x0C;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0xCC;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0xCC;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0xCC;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT1 & EXT_PORT0 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, EXT_PORT0);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to EXT_PORT0 + UTP_PORT1 + UTP_PORT3 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT3);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 0*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(0, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-00-0C-CC-CC-CC */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0x0C;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0xCC;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0xCC;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0xCC;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT3 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT3);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to UTP_PORT1 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 1*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(1, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    /* LLDP ACL */
    /* LLDP = 01:80:C2:00:00:0E */
    /* Reset Config & Action */
    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-80-C2-00-00-0E */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x80;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xC2;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0x0E;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT1 & EXT_PORT0 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, EXT_PORT0);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to EXT_PORT0 + UTP_PORT1 + UTP_PORT3 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT3);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 2*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(2, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-80-C2-00-00-0E */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x80;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xC2;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0x0E;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT3 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT3);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to UTP_PORT1 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 3*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(3, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    /* 802.1X ACL*/
    /* 802.1X = 01:80:C2:00:00:03 */
    /* Reset Config & Action */
    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-80-C2-00-00-03 */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x80;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xC2;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0x03;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT1 & EXT_PORT0 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, EXT_PORT0);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to EXT_PORT0 + UTP_PORT1 + UTP_PORT3 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT3);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 4*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(4, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-80-C2-00-00-03 */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0x01;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0x80;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xC2;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0x00;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0x03;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT3 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT3);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to UTP_PORT1 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);

    /* Set Action to assign VLAN 0 */
    act.actEnable[FILTER_ENACT_CVLAN_INGRESS] = TRUE;
    act.filterCvlanVid = 0;

    /*Add ACL configuration to ID 5*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(5, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    /* Write host MAC 00-01-02-03-04-05 */
    memset(&hostMac, 0x00, sizeof(rtk_mac_t));
    memset(&l2Addr, 0x00, sizeof(rtk_l2_ucastAddr_t));
    memcpy(hostMac.octet, host_mac, sizeof(hostMac.octet));
    GNMSG("rtl8363nbvb-hostMac: %pM", hostMac.octet);
    l2Addr.port = EXT_PORT0;
    l2Addr.is_static = 1;
    if (rtk_l2_addr_add(&hostMac, &l2Addr) != 0)
    {
        ERMSG("rtk_l2_addr_add fail");
        return (gn_status_t) RT_ERR_FAILED;
    }

    /* QoS init */
    if ((retVal = rtk_qos_init(8)) != RT_ERR_OK)
        return retVal;

    /* Configure DSCP to priority as following */
    /* User can change this based on their need */
    /* DSCP 0-7   => Priority 0 */
    /* DSCP 8-15  => Priority 1 */
    /* DSCP 16-23 => Priority 2 */
    /* DSCP 24-31 => Priority 3 */
    /* DSCP 32-39 => Priority 4 */
    /* DSCP 40-47 => Priority 5 */
    /* DSCP 48-55 => Priority 6 */
    /* DSCP 56-63 => Priority 7 */
    for (dscp = 0; dscp <= 63; dscp++)
    {
        if ((retVal = rtk_qos_dscpPriRemap_set(dscp, (dscp / 8))) != RT_ERR_OK)
            return retVal;
    }
    GNMSG("RTL8363NB-VB QoS priority enabled");

    for (i = 0; i < 2; i++) {
	 if (rtk_eee_portEnable_set(port[i], DISABLED)) {
	     ERMSG("!!! port[%d]: rtk_eee_portEnable_set() fail", port[i]);
	     return -1;
	 }
    }
    mdelay(20);
    for (i = 0; i < 2; i++) {
	 eee_state = -1;
	 if (!rtk_eee_portEnable_get(port[i], &eee_state)) {
	     GNMSG("rtk_eee_portEnable_get(): port[%d] state:[%d]", port[i], eee_state);
	 } else {
	     ERMSG("!!! port[%d]: rtk_eee_portEnable_get() fail", port[i]);
	     return -1;
	 }
    }

    GNMSG("RTL8363NB-VB configuration initial");

    return 0;
}

gn_status_t NetEthMiiReadReg(gn_port_t port, rtk_port_phy_reg_t reg, rtk_port_phy_data_t *pData)
{
	unsigned int rtk_port;
	rtk_api_ret_t ret;

       /* Port */
       switch (port)
       {
	      case LAN_PORT:
			rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             default:
			return GN_FAILURE;
			break;
        }
	ret = rtk_port_phyReg_get(rtk_port, reg, pData);
	return GN_SUCCESS;
}
EXPORT_SYMBOL(NetEthMiiReadReg);

gn_status_t NetEthMiiWriteReg(gn_port_t port, rtk_port_phy_reg_t reg, rtk_port_phy_data_t regData)
{
	unsigned int rtk_port;

       /* Port */
       switch (port)
       {
	      case LAN_PORT:
		rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             default:
			return GN_FAILURE;
			break;
        }

	if (rtk_port_phyReg_set(rtk_port, reg, regData) != 0){
		printk("rtk_port_phyReg_get fail");
		return GN_FAILURE;
	}

	return GN_SUCCESS;
}
EXPORT_SYMBOL(NetEthMiiWriteReg);

gn_status_t NetEswEthProtectSet(gn_port_t port, gn_enable_t enable)
{
    rtk_port_t rtk_port;

    /* Port */
    switch (port)
    {
        case LAN_PORT:
            rtk_port = UTP_PORT1;
            break;
        case PC_PORT:
            rtk_port = UTP_PORT3;
            break;
        case CPU_PORT:
            rtk_port = EXT_PORT0;
            break;
        default:
            return GN_FAILURE;
            break;
    }

    if (enable == GN_ENABLE)
    {
        if (rtk_stp_mstpState_set(0, rtk_port, STP_STATE_DISABLED) != 0)
        {
            printk("rtk_stp_mstpState_set fail");
            return GN_FAILURE;
        }
    }
    else
    {
        if (rtk_stp_mstpState_set(0, rtk_port, STP_STATE_FORWARDING) != 0)
        {
            ERMSG("rtk_stp_mstpState_set fail");
            return GN_FAILURE;
        }
    }

    return GN_SUCCESS;
}

gn_status_t NetEswEthProtectGet(gn_port_t port, rtk_int8 *enable)
{
    rtk_port_t rtk_port;
    rtk_stp_state_t state;

    /* Port */
    switch (port)
    {
        case LAN_PORT:
            rtk_port = UTP_PORT1;
            break;
        case PC_PORT:
            rtk_port = UTP_PORT3;
            break;
        case CPU_PORT:
            rtk_port = EXT_PORT0;
            break;
        default:
            return GN_FAILURE;
            break;
    }

    if (rtk_stp_mstpState_get(0, rtk_port, &state) != 0)
    {
        ERMSG("rtk_stp_mstpState_get fail");
        return GN_FAILURE;
    }

    if (state == STP_STATE_DISABLED)
        *enable = GN_ENABLE;
    else
        *enable = GN_DISABLE;

    return GN_SUCCESS;
}

gn_status_t NetEswEthProtectReset(void)
{
    /* Disables protect mode on LAN and IMP(CPU) port */
    if (rtk_stp_mstpState_set(0, UTP_PORT1, STP_STATE_FORWARDING) != 0)
    {
        ERMSG("rtk_stp_mstpState_set fail");
        return GN_FAILURE;
    }

    if (rtk_stp_mstpState_set(0, EXT_PORT0, STP_STATE_FORWARDING) != 0)
    {
        ERMSG("rtk_stp_mstpState_set fail");
        return GN_FAILURE;
    }

    return GN_SUCCESS;
}

gn_status_t NetEswEthProtectStatus(rtk_int8 *status)
{
	//rtk_port_t rtk_port;
	rtk_stp_state_t lanState, cpuState;
#if 0
    /* Port */
    switch (port)
    {
        case LAN_PORT:
            rtk_port = UTP_PORT1;
            break;
        case CPU_PORT:
            rtk_port = EXT_PORT0;
            break;
        default:
            return GN_FAILURE;
            break;
    }
#endif
    if (rtk_stp_mstpState_get(0, UTP_PORT1, &lanState) != 0)
    {
        ERMSG("rtk_stp_mstpState_get fail");
        return GN_FAILURE;
    }

    if (rtk_stp_mstpState_get(0, EXT_PORT0, &cpuState) != 0)
    {
        ERMSG("rtk_stp_mstpState_get fail");
        return GN_FAILURE;
    }

    if (((lanState == STP_STATE_DISABLED) ||(cpuState == STP_STATE_DISABLED)) ||
		((lanState == STP_STATE_DISABLED) && (cpuState == STP_STATE_DISABLED)))
        *status = 1;
    else
        *status = 0;

    return GN_SUCCESS;
}

gn_status_t EthFilterBsp(gn_port_t port, rtk_uint32 rate)
{
    rtk_port_t rtk_port;
    rtk_meter_id_t meterIdx;

    switch (port)
    {
        case LAN_PORT:
            rtk_port = UTP_PORT1;
            meterIdx = 8;
            break;
        case PC_PORT:
            rtk_port = UTP_PORT3;
            meterIdx = 9;
            break;
        case CPU_PORT:
            rtk_port = EXT_PORT0;
            meterIdx = 10;
            break;
        default:
            return GN_FAILURE;
            break;
    }

    if (rtk_rate_shareMeter_set(meterIdx, METER_TYPE_KBPS, rate, ENABLED) != 0)
    {
        ERMSG("rtk_rate_shareMeter_set fail");
        return GN_FAILURE;
    }

    if (rtk_rate_stormControlMeterIdx_set(rtk_port, STORM_GROUP_BROADCAST, meterIdx) != 0)
    {
        ERMSG("rtk_rate_stormControlMeterIdx_set fail");
        return GN_FAILURE;
    }

    if (rtk_rate_stormControlPortEnable_set(rtk_port, STORM_GROUP_BROADCAST, ENABLED) != 0)
    {
        ERMSG("rtk_rate_stormControlPortEnable_set fail");
        return GN_FAILURE;
    }

    return GN_SUCCESS;
}

gn_status_t EthFilterAllPacket(rtk_uint32 rate)
{
    rtk_qid_t qid;

	if (rate == GN_DISABLE) {
		if (rtk_rate_egrQueueBwCtrlEnable_set(EXT_PORT0, 0xFF, DISABLED) != 0)
		{
			ERMSG("rtk_rate_egrQueueBwCtrlEnable_set fail");
		return GN_FAILURE;
		}
		return GN_SUCCESS;
	}

	/* Treat rate as packet/second */
	if (rtk_rate_shareMeter_set(16, METER_TYPE_PPS, rate, ENABLED) != 0)
	{
	ERMSG("rtk_rate_shareMeter_set fail");
		return GN_FAILURE;
	}

	for (qid = 0; qid <= 7; qid++)
	{
	if (rtk_rate_egrQueueBwCtrlRate_set(EXT_PORT0, qid, 16) != 0)
		{
			ERMSG("rtk_rate_egrQueueBwCtrlRate_set fail");
			return GN_FAILURE;
		}
	}

	if (rtk_rate_egrQueueBwCtrlEnable_set(EXT_PORT0, 0xFF, ENABLED) != 0)
	{
		ERMSG("rtk_rate_egrQueueBwCtrlEnable_set fail");
	return GN_FAILURE;
	}

	return GN_SUCCESS;
}

gn_status_t EthAddMcastMac(char action, unsigned char* macAddr)
{
	rtk_uint32                  retVal;
	rtk_l2_mcastAddr_t          l2McastAddr;

	memset(&l2McastAddr, 0x00, sizeof(rtk_l2_mcastAddr_t));
	l2McastAddr.mac.octet[0] = *macAddr;
	l2McastAddr.mac.octet[1] = *(macAddr+1);
	l2McastAddr.mac.octet[2] = *(macAddr+2);
	l2McastAddr.mac.octet[3] = *(macAddr+3);
	l2McastAddr.mac.octet[4] = *(macAddr+4);
	l2McastAddr.mac.octet[5] = *(macAddr+5);

	if (action == 0) {
	        if (rtk_l2_mcastAddr_add(&l2McastAddr) != 0)
	        {
	            ERMSG("rtk_l2_mcastAddr_add fail");
	            return GN_FAILURE;
	        }
	} else {
		RTK_PORTMASK_PORT_SET(l2McastAddr.portmask, UTP_PORT1);
		RTK_PORTMASK_PORT_SET(l2McastAddr.portmask, UTP_PORT3);
		RTK_PORTMASK_PORT_SET(l2McastAddr.portmask, EXT_PORT0);
		if((retVal = rtk_l2_mcastAddr_add(&l2McastAddr)) != 0)
			return retVal;
	}

	return GN_SUCCESS;
}

gn_status_t Tag8021QGlobalEnable(rtk_int32 vid)
{
    rtk_vlan_cfg_t  vlanCfg;
    rtk_vlan_t      pvid;
    rtk_pri_t       priority;
    rtk_uint32 retVal;
    rtk_uint32 ruleNum;
    rtk_filter_field_t field1;
    rtk_filter_cfg_t cfg;
    rtk_filter_action_t act;

    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    RTK_PORTMASK_PORT_SET(vlanCfg.mbr, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(vlanCfg.mbr, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(vlanCfg.untag, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(vlanCfg.untag, UTP_PORT3);
    if (rtk_vlan_set(vid, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return GN_FAILURE;
    }

    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    RTK_PORTMASK_PORT_SET(vlanCfg.mbr, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(vlanCfg.mbr, UTP_PORT3);
    RTK_PORTMASK_PORT_SET(vlanCfg.untag, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(vlanCfg.untag, UTP_PORT1);
    RTK_PORTMASK_PORT_SET(vlanCfg.untag, UTP_PORT3);
    if (rtk_vlan_set(0, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return GN_FAILURE;
    }

    if (rtk_vlan_portPvid_get(EXT_PORT0, &pvid, &priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_get fail");
        return GN_FAILURE;
    }
    pvid = vid;
    if (rtk_vlan_portPvid_set(EXT_PORT0, pvid, priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return GN_FAILURE;
    }
    /* filter untag packet at LAN ingress + redirect to PC + CPU (ID:6) */
    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = 01-00-0C-CC-CC-CC */
    field1.fieldType = FILTER_FIELD_DMAC;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT1 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT1);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to EXT_PORT0 + UTP_PORT3 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT3);

    /* Care only untag packet */
    cfg.careTag.tagType[CARE_TAG_CTAG].value = 0x0;
    cfg.careTag.tagType[CARE_TAG_CTAG].mask = 0x1;

    /*Add ACL configuration to ID 6*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(6, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;


    /* filter BC untag packet at PC ingress + redirect to LAN + CPU (ID:7) */
     memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = FF-FF-FF-FF-FF-FF */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* UTP_PORT3 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, UTP_PORT3);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to EXT_PORT0 + UTP_PORT1 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, EXT_PORT0);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);

    /* Care only untag packet */
    cfg.careTag.tagType[CARE_TAG_CTAG].value = 0x0;
    cfg.careTag.tagType[CARE_TAG_CTAG].mask = 0x1;

    /*Add ACL configuration to ID 7*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(7, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    /* filter BC untag packet at CPU ingress + redirecr to LAN + PC (ID:8) */
    memset(&cfg, 0x00, sizeof(rtk_filter_cfg_t));
    memset(&act, 0x00, sizeof(rtk_filter_action_t));
    memset(&field1, 0x00, sizeof(rtk_filter_field_t));

    /*Add DMAC = FF-FF-FF-FF-FF-FF */
    field1.fieldType = FILTER_FIELD_DMAC;
    field1.filter_pattern_union.dmac.value.octet[0]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[1]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[2]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[3]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[4]   = 0xFF;
    field1.filter_pattern_union.dmac.value.octet[5]   = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[0]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[1]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[2]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[3]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[4]    = 0xFF;
    field1.filter_pattern_union.dmac.mask.octet[5]    = 0xFF;
    if ((retVal = rtk_filter_igrAcl_field_add(&cfg, &field1)) != RT_ERR_OK)
        return retVal;

    /* EXT_PORT0 to active ports*/
    RTK_PORTMASK_PORT_SET(cfg.activeport.value, EXT_PORT0);
    RTK_PORTMASK_ALLPORT_SET(cfg.activeport.mask);

    /* Set Action to redirect to UTP_PORT3 + UTP_PORT1 */
    act.actEnable[FILTER_ENACT_REDIRECT] = TRUE;
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT3);
    RTK_PORTMASK_PORT_SET(act.filterPortmask, UTP_PORT1);

    /* Care only untag packet */
    cfg.careTag.tagType[CARE_TAG_CTAG].value = 0x0;
    cfg.careTag.tagType[CARE_TAG_CTAG].mask = 0x1;

    /*Add ACL configuration to ID 8*/
     if ((retVal = rtk_filter_igrAcl_cfg_add(8, &cfg, &act, &ruleNum)) != RT_ERR_OK)
        return retVal;

    return GN_SUCCESS;
}

gn_status_t Tag8021QGlobalDisable(rtk_int32 vid)
{
    rtk_vlan_cfg_t  vlanCfg;
    rtk_vlan_t      pvid;
    rtk_pri_t       priority;
    rtk_uint32 		retVal;

    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    if (rtk_vlan_set(vid, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return GN_FAILURE;
    }

    memset(&vlanCfg, 0x00, sizeof(rtk_vlan_cfg_t));
    RTK_PORTMASK_ALLPORT_SET(vlanCfg.mbr);
    RTK_PORTMASK_ALLPORT_SET(vlanCfg.untag);
    if (rtk_vlan_set(0, &vlanCfg) != 0)
    {
        ERMSG("rtk_vlan_set fail");
        return GN_FAILURE;
    }

    if (rtk_vlan_portPvid_get(EXT_PORT0, &pvid, &priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_get fail");
        return GN_FAILURE;
    }
    pvid = 0;
    if (rtk_vlan_portPvid_set(EXT_PORT0, pvid, priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return GN_FAILURE;
    }

    if ((retVal = rtk_filter_igrAcl_cfg_del(6)) != RT_ERR_OK)
    {
        ERMSG("rtk_filter_igrAcl_cfg_del fail");
        return retVal;
    }

    if ((retVal = rtk_filter_igrAcl_cfg_del(7)) != RT_ERR_OK)
    {
        ERMSG("rtk_filter_igrAcl_cfg_del fail");
        return retVal;
    }

    if ((retVal = rtk_filter_igrAcl_cfg_del(8)) != RT_ERR_OK)
    {
        ERMSG("rtk_filter_igrAcl_cfg_del fail");
        return retVal;
    }
    return GN_SUCCESS;
}

gn_status_t Tag8021QGlobalUPSet (rtk_int8 qos)
{
    rtk_vlan_t      pvid;
    rtk_pri_t       priority;

    if (rtk_vlan_portPvid_get(EXT_PORT0, &pvid, &priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_get fail");
        return GN_FAILURE;
    }
    priority = qos;
    if (rtk_vlan_portPvid_set(EXT_PORT0, pvid, priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return GN_FAILURE;
    }

    if (rtk_vlan_portPvid_get(UTP_PORT3, &pvid, &priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_get fail");
        return GN_FAILURE;
    }
    priority = qos;
    if (rtk_vlan_portPvid_set(UTP_PORT3, pvid, priority) != 0)
    {
        ERMSG("rtk_vlan_portPvid_set fail");
        return GN_FAILURE;
    }

    return GN_SUCCESS;
}

gn_status_t GetEthMib(gn_port_t port, rtk_stat_port_cntr_t *outData)
{
	 unsigned int	retVal, rtk_port;

       switch (port)
       {
	      case LAN_PORT:
		rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             case CPU_PORT:
			rtk_port = EXT_PORT0;
			break;
             default:
			return GN_FAILURE;
			break;
        }

	if ((retVal = rtk_stat_port_getAll(rtk_port, outData)) != 0)
	return retVal;

	return GN_SUCCESS;
}
EXPORT_SYMBOL(GetEthMib);

gn_status_t ClearEthMib(gn_port_t port)
{
	unsigned int retVal, rtk_port;

       switch (port)
       {
	      case LAN_PORT:
		rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             case CPU_PORT:
			rtk_port = EXT_PORT0;
			break;
             default:
			return GN_FAILURE;
			break;
        }

        if((retVal = rtk_stat_port_reset(rtk_port)) != 0)
		return GN_FAILURE;

    return GN_SUCCESS;
}
EXPORT_SYMBOL(ClearEthMib);

gn_status_t EthSetPortSpeed ( gn_port_t port, gn_port_mode_t mode )
{
	unsigned int retVal, rtk_port;
	rtk_port_phy_ability_t ability;

       switch (port)
       {
	      case LAN_PORT:
		rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             default:
			return GN_FAILURE;
			break;
        }

            memset(&ability, 0x00, sizeof(rtk_port_phy_ability_t));

            ability.FC = 1;
            ability.AsyFC = 1;

            switch(mode)
            {
                case PORT_MODE_AUTO:
                    ability.Full_10     = 1;
                    ability.Half_10     = 1;
                    ability.Full_100    = 1;
                    ability.Half_100    = 1;
                    ability.Full_1000   = 1;
                    ability.AutoNegotiation = 1;
                    break;
                case PORT_MODE_10_HALF:
                    ability.Half_10     = 1;
                    break;
                case PORT_MODE_10_FULL:
                    ability.Full_10     = 1;
                    break;
                case PORT_MODE_100_HALF:
                    ability.Half_100    = 1;
                    break;
                case PORT_MODE_100_FULL:
                    ability.Full_100    = 1;
                    break;
                case PORT_MODE_1000_FULL:
                    ability.Full_1000   = 1;
                    ability.AutoNegotiation = 1;
                    break;
                default:
                    return GN_FAILURE;
                    break;
            }

            if(ability.AutoNegotiation)
            {
                if((retVal = rtk_port_phyAutoNegoAbility_set(rtk_port, &ability)) != RT_ERR_OK)
                    return GN_FAILURE;
            }
            else
            {
                if((retVal = rtk_port_phyForceModeAbility_set(rtk_port, &ability)) != RT_ERR_OK)
                    return GN_FAILURE;
            }

	 return GN_SUCCESS;
}
EXPORT_SYMBOL(EthSetPortSpeed);

gn_status_t EthQueryLink ( gn_ethlink_t *linkStatus )
{
	unsigned int retVal, rtk_port;
	rtk_port_linkStatus_t link;
	rtk_data_t speed;
	rtk_data_t duplex;

       switch (linkStatus->port)
       {
	      case LAN_PORT:
		rtk_port = UTP_PORT1;
			break;
             case PC_PORT:
			rtk_port = UTP_PORT3;
			break;
             default:
			return GN_FAILURE;
			break;
        }

            if((retVal = rtk_port_phyStatus_get(rtk_port, &link, &speed, &duplex)) != RT_ERR_OK)
                return GN_FAILURE;

            linkStatus->status = (gn_link_status_t) link;

            switch(speed)
            {
                case PORT_SPEED_10M:
                    linkStatus->speed = LINK_10M;
                    break;

                case PORT_SPEED_100M:
                    linkStatus->speed = LINK_100M;
                    break;

                case PORT_SPEED_1000M:
                    linkStatus->speed = LINK_1000M;
                    break;

                default:
                    return GN_FAILURE;
                    break;
            }

            switch(duplex)
            {
                case PORT_HALF_DUPLEX:
                    linkStatus->duplex = LINK_HALF;
                    break;
                case PORT_FULL_DUPLEX:
                    linkStatus->duplex = LINK_FULL;
                    break;
                default:
                    return GN_FAILURE;
                    break;
            }
	//WDBG("port:[%d] link:[%d] speed:[%d] duplex:[%d]",
	//				linkStatus->port,  linkStatus->status , linkStatus->speed, linkStatus->duplex);

	return GN_SUCCESS;
}
EXPORT_SYMBOL(EthQueryLink);

gn_status_t EthGetL2AddrMac(rtk_uint32 *pAddress, rtk_l2_ucastAddr_t *pL2_data)
{
	return rtk_l2_addr_next_get(READMETHOD_NEXT_L2UC, 0, pAddress, pL2_data);
}

gn_status_t EthSTPblockAllPort(gn_enable_t enable)
{
    if (enable == GN_ENABLE)
    {
        if (rtk_trap_rmaAction_set(TRAP_BRG_GROUP, RMA_ACTION_DROP) != 0)
        {
            ERMSG("rtk_trap_rmaAction_set fail");
            return GN_FAILURE;
        }
    }
    else
    {
        if (rtk_trap_rmaAction_set(TRAP_BRG_GROUP, RMA_ACTION_FORWARD) != 0)
        {
            ERMSG("rtk_trap_rmaAction_set fail");
            return GN_FAILURE;
        }
    }

    return GN_SUCCESS;
}

gn_status_t EthGetMcastMacEntry(rtk_uint32 *pAddr_id, rtk_l2_mcastAddr_t *pMcastAddr)
{
       return rtk_l2_mcastAddr_next_get(pAddr_id, pMcastAddr);
 }
