#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/leds.h>
#include <linux/regmap.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/timer.h>
#include <linux/delay.h>

#define REG_ID  0x0
#define REG_MONITOR 0x1
#define REG_CONTROL 0x2
#define REG_IRED0   0x3
#define REG_IGRN0   0x4
#define REG_IBLU0   0x5
#define REG_IRED1   0x6
#define REG_IGRN1   0x7
#define REG_IBLU1   0x8
#define REG_ISELA12     0x9
#define REG_ISELA34     0xa
#define REG_ISELB12     0xb
#define REG_ISELB34     0xc
#define REG_ISELC12     0xd
#define REG_ISELC34     0xe

#define ENABLE_NORMAL_MODE 0x80
#define ENABLE_NIGHT_MODE  0x40

#define EN_LED_MASK     0x08
#define SET_LED_COLOR_MASK  0x07

#define HIGH_BYTE_SHIFT         (4)
#define LOW_BYTE_SHIFT          (0)


/**
 * led max current is 24mA
 *
 *  0x00    0uA
 *  0x01  125uA
 *  ...
 *  0x28    5mA
 *  ...
 *  0xc0   24mA
 *  ...
 *  0xff   24mA
 */
#define LED_RED0_BASIC_CURRENT_VALUE 0x00
#define LED_GREEN0_BASIC_CURRENT_VALUE 0x00
#define LED_BLUE0_BASIC_CURRENT_VALUE 0x00
#define LED_RED1_BASIC_CURRENT_VALUE 0x80
#define LED_GREEN1_BASIC_CURRENT_VALUE 0x80
#define LED_BLUE1_BASIC_CURRENT_VALUE 0xC0

/* operation symbol */
enum ktd20xx_mode {
    LR_LED = 0,
};

enum led_color {
    BLACK = 0x0,  /*LED_off*/
    BLUE,
    GREEN,
    CYAN,
    RED,
    MAGENTA,
    YELLOW,
    WHITE
};

enum ktd20xx_state {
    KTD20XX_OFF  = 0x0,
    KTD20XX_ON   = 0x1,
    KTD20XX_BLINK = 0x2,
    KTD20XX_PULSATING_MODE1 = 0x3,
    KTD20XX_PULSATING_MODE2 = 0x4
};

struct ktd20xx {
    struct i2c_client *client;
    struct regmap *regmap;
    struct work_struct work;
    struct mutex lock;

    struct led_classdev cdev_lr;
    struct led_classdev cdev_icon;

    unsigned char lr_brightness;
    unsigned char icon_brightness;

    unsigned char current_lr_brightness;
    unsigned char current_icon_brightness;

    int set_mode;
};

static void ktd20xx_turn_on_channels(struct ktd20xx *data)
{
    int ret;

    ret = regmap_write(data->regmap, REG_ISELA12, 0x77);
    ret = regmap_write(data->regmap, REG_ISELA34, 0x77);
    ret = regmap_write(data->regmap, REG_ISELB12, 0x77);
	ret = regmap_write(data->regmap, REG_ISELB34, 0x77);
    ret = regmap_write(data->regmap, REG_IRED0, LED_RED0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IGRN0, LED_GREEN0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IBLU0, LED_BLUE0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IRED1, LED_RED1_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IGRN1, LED_GREEN1_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IBLU1, LED_BLUE1_BASIC_CURRENT_VALUE); 

    ret = regmap_write(data->regmap, REG_CONTROL, 0x82);
}

static int ktd20xx_init(struct ktd20xx *data)
{
    struct device *dev = &data->client->dev;
    int ret;

    dev_info(dev, "[KTD20xx]%s", __FUNCTION__);
    ret = regmap_write(data->regmap, REG_CONTROL, 0x82);
    if (ret < 0) {
        dev_err(dev, "i2c write to 0x%X failed, ret:%d\n", 0x2, ret);
        goto out;
    }
    ret = regmap_write(data->regmap, REG_IRED0, LED_RED0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IGRN0, LED_GREEN0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IBLU0, LED_BLUE0_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IRED1, LED_RED1_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IGRN1, LED_GREEN1_BASIC_CURRENT_VALUE);
    ret = regmap_write(data->regmap, REG_IBLU1, LED_BLUE1_BASIC_CURRENT_VALUE); 
    ktd20xx_turn_on_channels(data);
    return 0;
out:
    return ret;
}

static void ktd20xx_brightness(struct ktd20xx *data, int brightness)
{
    int ret;

    if(data->set_mode==LR_LED) {
        if(brightness==0) { 
            //A1
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    EN_LED_MASK<<HIGH_BYTE_SHIFT,0x00);
            //A2
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    EN_LED_MASK<<LOW_BYTE_SHIFT,0x00);
            //A3
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    EN_LED_MASK<<HIGH_BYTE_SHIFT,0x00);
            //A4
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    EN_LED_MASK<<LOW_BYTE_SHIFT,0x00);
            //B1
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    EN_LED_MASK<<HIGH_BYTE_SHIFT,0x00);
            //B2
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    EN_LED_MASK<<LOW_BYTE_SHIFT,0x00);
            //B3
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    EN_LED_MASK<<HIGH_BYTE_SHIFT,0x00);
            //B4
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    EN_LED_MASK<<LOW_BYTE_SHIFT,0x00);
        } else if(brightness<7){
            //A1
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x08|brightness)<<HIGH_BYTE_SHIFT);
            //A2
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x08|brightness)<<LOW_BYTE_SHIFT);
            //A3
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x08|brightness)<<HIGH_BYTE_SHIFT);
            //A4
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x08|brightness)<<LOW_BYTE_SHIFT);
            //B1
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x08|brightness)<<HIGH_BYTE_SHIFT);
            //B2
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x08|brightness)<<LOW_BYTE_SHIFT);
            //B3
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x08|brightness)<<HIGH_BYTE_SHIFT);
            //B4
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x08|brightness)<<LOW_BYTE_SHIFT);
            
        } else {
            //A1
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x0f)<<HIGH_BYTE_SHIFT);
            //A2
            ret = regmap_update_bits(data->regmap, REG_ISELA12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x0f)<<LOW_BYTE_SHIFT);
            //A3
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x0f)<<HIGH_BYTE_SHIFT);
            //A4
            ret = regmap_update_bits(data->regmap, REG_ISELA34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x0f)<<LOW_BYTE_SHIFT);
            //B1
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x0f)<<HIGH_BYTE_SHIFT);
            //B2
            ret = regmap_update_bits(data->regmap, REG_ISELB12,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x0f)<<LOW_BYTE_SHIFT);
            //B3
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<HIGH_BYTE_SHIFT,
                    (0x0f)<<HIGH_BYTE_SHIFT);
            //B4
            ret = regmap_update_bits(data->regmap, REG_ISELB34,
                    (EN_LED_MASK|SET_LED_COLOR_MASK)<<LOW_BYTE_SHIFT,
                    (0x0f)<<LOW_BYTE_SHIFT);
        }
    }
}

static void ktd20xx_led_work(struct work_struct *work)
{
    struct ktd20xx *data = container_of(work, struct ktd20xx, work);
  
    mutex_lock(&data->lock);

    if(data->set_mode==LR_LED){
        if(data->lr_brightness != data->current_lr_brightness){
            ktd20xx_brightness(data,data->lr_brightness);
        }
        data->current_lr_brightness = data->lr_brightness;
    }

    
    mutex_unlock(&data->lock);
}

static void ktd20xx_lr_brightness_set(struct led_classdev *cdev, enum led_brightness brightness)
{
    
    struct ktd20xx *data = container_of(cdev, struct ktd20xx, cdev_lr);
    
    data->lr_brightness = brightness;
    data->set_mode=LR_LED;
    schedule_work(&data->work);
}



static const struct regmap_config ktd20xx_regmap_config = {
    .reg_bits = 8,
    .val_bits = 8,
    .max_register = 0xFF,   
};

static int ktd20xx_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    struct ktd20xx *data;
    struct device *dev = &client->dev;
    int ret;

    dev_info(dev, "[KTD20xx]%s start\n", __FUNCTION__);

    if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
        dev_err(dev, "i2c functionality check fail.\n");
        return -ENODEV;
    }

    data = devm_kzalloc(dev, sizeof(struct ktd20xx), GFP_KERNEL);
    if (!data)
        return -ENOMEM;

    data->regmap = devm_regmap_init_i2c(client, &ktd20xx_regmap_config);
    if (IS_ERR(data->regmap)) {
        ret = PTR_ERR(data->regmap);
        dev_err(dev, "regmap allocation failed:%d\n", ret);
        return ret;
    }

    data->client = client;
    mutex_init(&data->lock);
    INIT_WORK(&data->work, ktd20xx_led_work);
    i2c_set_clientdata(client, data);

    data->cdev_lr.name = "lr_led";
    data->cdev_lr.brightness = LED_FULL;
    data->cdev_lr.max_brightness = LED_FULL;
    data->cdev_lr.brightness_set = ktd20xx_lr_brightness_set;
	data->cdev_lr.default_trigger = "none";

    ret = led_classdev_register(dev, &data->cdev_lr);
    if (ret < 0) {
        dev_err(dev, "failed to register ktd20xx(power led), ret:%d\n", ret);
        return ret;
    }
    
    mutex_lock(&data->lock);
    ret = ktd20xx_init(data);
    mutex_unlock(&data->lock);
    
    if(ret < 0) {
        dev_err(dev, "[KTD20xx] ktd20xx init fail.\n");
    }   

    data->current_lr_brightness = 0x0;
    data->current_icon_brightness = 0x0;

    dev_info(dev, "[KTD20xx]%s exit\n", __FUNCTION__);

    return 0;
}


static int ktd20xx_remove(struct i2c_client *client)
{
    struct ktd20xx *data = i2c_get_clientdata(client);

    led_classdev_unregister(&data->cdev_lr);
    led_classdev_unregister(&data->cdev_icon);

    regmap_write(data->regmap, REG_CONTROL, 0);
    
    return 0;
}

static const struct i2c_device_id ktd20xx_id[] = {
    { "ktd2058", 0},
    { "ktd2059", 1},
    { "ktd2060", 2},
    { "ktd2061", 3},
    { "ktd2064", 4},
    {}
};
MODULE_DEVICE_TABLE(i2c, ktd20xx_id);

static const struct of_device_id of_ktd20xx_match[] = {
    { .compatible = "kinetic,ktd2058", },
    { .compatible = "kinetic,ktd2059", },
    { .compatible = "kinetic,ktd2060", },
    { .compatible = "kinetic,ktd2061", },
    { .compatible = "kinetic,ktd2064", },
    {},
};    

static struct i2c_driver ktd20xx_i2c_driver = {
        .driver = {
            .name = "ktd20xx",
            .pm = NULL,
            .of_match_table = of_match_ptr(of_ktd20xx_match),
        },
        .probe = ktd20xx_probe,
        .remove = ktd20xx_remove,
        .id_table = ktd20xx_id,
};

module_i2c_driver(ktd20xx_i2c_driver);

MODULE_DESCRIPTION("LED driver for KTD2058/59/60/61/64");
MODULE_AUTHOR("Wistron");
MODULE_LICENSE("GPL v2");
