// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * leds-aw20036.c   aw20036 led module
 *
 * Copyright (c) 2018 Shanghai Awinic Technology Co., Ltd. All Rights Reserved
 *
 *  Author: awinic
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/firmware.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/debugfs.h>
#include <linux/miscdevice.h>
#include <linux/uaccess.h>
#include <linux/leds.h>
#include <dt-bindings/leds/leds-aw20036.h>
#include <dt-bindings/leds/leds-aw20036-reg.h>
/******************************************************
 *
 * Marco
 *
 ******************************************************/
#define AW20036_I2C_NAME "aw20036_led"

#define AW20036_DRIVER_VERSION "v1.5.0"

#define AW_I2C_RETRIES 2
#define AW_I2C_RETRY_DELAY 1
#define AW_READ_CHIPID_RETRIES 2
#define AW_READ_CHIPID_RETRY_DELAY 1

struct pinctrl *pinctrl_aw20036;
struct pinctrl_state *aw20036_pins_cfg, *aw20036_rst_output0, *aw20036_rst_output1;
unsigned int aw20036debounce;

const struct of_device_id aw20036_of_match[] = {
	{.compatible = "awinic,aw20036_led",},
	{},
};

/******************************************************
 *
 * aw20036 led parameter
 *
 ******************************************************/
#define AW20036_CFG_NAME_MAX        64
#if defined(AW20036_BIN_CONFIG)
static char aw20036_cfg_name[][AW20036_CFG_NAME_MAX] = {
	{"aw20036_led_all_on.bin"},
	{"aw20036_led_red_on.bin"},
	{"aw20036_led_green_on.bin"},
	{"aw20036_led_blue_on.bin"},
	{"aw20036_led_breath_forever.bin"},
	{"aw20036_cfg_led_off.bin"},
};
#elif defined(AW20036_ARRAY_CONFIG)
/*mapped from jabra led manager component*/
/*  off      blue      green     aqua       red     purple    yellow     white */
/*0x00000, 0x0000FF, 0x00FF00, 0x00FFFF, 0xFF0000, 0x800080, 0xFFFF00, 0xFFFFFF*/
struct aw20036_cfg aw20036_cfg_array[] = {
	{aw20036_cfg_led_off, sizeof(aw20036_cfg_led_off)}
	,
	{aw20036_led_blue_on, sizeof(aw20036_led_blue_on)}
	,
	{aw20036_led_green_on, sizeof(aw20036_led_green_on)}
	,
	{aw20036_led_aqua_on, sizeof(aw20036_led_aqua_on)}
	,
	{aw20036_led_red_on, sizeof(aw20036_led_red_on)}
	,
	{aw20036_led_purple_on, sizeof(aw20036_led_purple_on)}
	,
	{aw20036_led_yellow_on, sizeof(aw20036_led_yellow_on)}
	,
	{aw20036_led_all_on, sizeof(aw20036_led_all_on)} /*white*/
	,
	{aw20036_led_breath_forever, sizeof(aw20036_led_breath_forever)}
};
#else
	/*Nothing */
#endif

#define AW20036_IMAX_NAME_MAX       32
static char aw20036_imax_name[][AW20036_IMAX_NAME_MAX] = {
	{"AW20036_IMAX_10mA"},
	{"AW20036_IMAX_20mA"},
	{"AW20036_IMAX_30mA"},
	{"AW20036_IMAX_40mA"},
	{"AW20036_IMAX_60mA"},
	{"AW20036_IMAX_80mA"},
	{"AW20036_IMAX_120mA"},
	{"AW20036_IMAX_160mA"},
	{"AW20036_IMAX_3P3mA"},
	{"AW20036_IMAX_6P7mA"},
	{"AW20036_IMAX_10P0mA"},
	{"AW20036_IMAX_13P3mA"},
	{"AW20036_IMAX_20P0mA"},
	{"AW20036_IMAX_26P7mA"},
	{"AW20036_IMAX_40P0mA"},
	{"AW20036_IMAX_53P3mA"},
};

static int aw20036_led_init(struct aw20036 *aw20036);

/******************************************************
 *
 * aw20036 i2c write/read
 *
 ******************************************************/
static int
aw20036_i2c_write(struct aw20036 *aw20036, unsigned char reg_addr, unsigned char reg_data)
{
	int ret = -1;
	unsigned char cnt = 0;

	while (cnt < AW_I2C_RETRIES) {
		ret = i2c_smbus_write_byte_data(aw20036->i2c, reg_addr, reg_data);
		if (ret < 0)
			pr_err("%s: i2c_write cnt=%d error=%d\n", __func__, cnt, ret);
		else
			break;

		cnt++;
		usleep_range(1000, 2000);
	}

	return ret;
}

static int
aw20036_i2c_read(struct aw20036 *aw20036, unsigned char reg_addr, unsigned char *reg_data)
{
	int ret = -1;
	unsigned char cnt = 0;

	while (cnt < AW_I2C_RETRIES) {
		ret = i2c_smbus_read_byte_data(aw20036->i2c, reg_addr);
		if (ret < 0) {
			pr_err("%s: i2c_read cnt=%d error=%d\n", __func__, cnt, ret);
		} else {
			*reg_data = ret;
			break;
		}
		cnt++;
		usleep_range(1000, 2000);
	}

	return ret;
}

static int aw20036_i2c_write_bits(struct aw20036 *aw20036,
				  unsigned char reg_addr, unsigned int mask,
				  unsigned char reg_data)
{
	unsigned char reg_val;

	aw20036_i2c_read(aw20036, reg_addr, &reg_val);
	reg_val &= mask;
	reg_val |= (reg_data & (~mask));
	aw20036_i2c_write(aw20036, reg_addr, reg_val);

	return 0;
}

/*****************************************************
 *
 * aw20036 led cfg
 *
 *****************************************************/
static int aw20036_reg_page_cfg(struct aw20036 *aw20036, unsigned char page)
{
	aw20036_i2c_write(aw20036, REG_PAGE, page);
	return 0;
}

static int aw20036_imax_cfg(struct aw20036 *aw20036, unsigned char imax)
{
	if (imax > 0xF)
		imax = 0xF;
	aw20036_reg_page_cfg(aw20036, AW20036_REG_PAGE0);
	aw20036_i2c_write_bits(aw20036, REG_GCCR, BIT_IMAX_MASK, imax << 4);

	return 0;
}

/*static int aw20036_dbgdim_cfg(struct aw20036 *aw20036, unsigned int data)
{
	int i;

	aw20036_i2c_write(aw20036, 0xF0, 0xC4);
	for (i = 0; i < AW20036_REG_NUM_PAG4; i = i + 2) {
		aw20036->rgbcolor = data;
		aw20036_i2c_write(aw20036, i, aw20036->rgbcolor);
	}
	return 0;
}

static int aw20036_dbgfdad_cfg(struct aw20036 *aw20036, unsigned int data)
{
	int i;

	aw20036_i2c_write(aw20036, 0xF0, 0xC4);
	for (i = 1; i < AW20036_REG_NUM_PAG4; i = i + 2) {
		aw20036->rgbcolor = data;
		aw20036_i2c_write(aw20036, i, aw20036->rgbcolor);
	}

	return 0;
}*/

static void aw20036_set_brightness(struct led_classdev *cdev, enum led_brightness brightness)
{
	struct aw20036 *aw20036 = container_of(cdev, struct aw20036, cdev);

	aw20036->cdev.brightness = brightness;

	schedule_work(&aw20036->brightness_work);
}

/*****************************************************
 *
 * firmware/cfg update
 *
 *****************************************************/
#if defined(AW20036_ARRAY_CONFIG)
static void
aw20036_update_cfg_array(struct aw20036 *aw20036, unsigned char *p_cfg_data, unsigned int cfg_size)
{
	unsigned int i = 0;
	unsigned char page = 0;

	for (i = 0; i < cfg_size; i += 2) {
		aw20036_i2c_write(aw20036, p_cfg_data[i], p_cfg_data[i + 1]);
		if (p_cfg_data[i] == 0xf0)
			page = p_cfg_data[i + 1];
		if ((page == AW20036_REG_PAGE0)
			&& (p_cfg_data[i] == REG_SWRST)
			&& (p_cfg_data[i + 1] == 0x01)) {
			usleep_range(2000, 2500);
		}
	}
}

static int aw20036_cfg_update_array(struct aw20036 *aw20036)
{
	aw20036_update_cfg_array(aw20036,
				 (aw20036_cfg_array[aw20036->effect].p),
				 aw20036_cfg_array[aw20036->effect].count);
	return 0;
}
#endif
#if defined(AW20036_BIN_CONFIG)
static void aw20036_cfg_loaded(const struct firmware *cont, void *context)
{
	struct aw20036 *aw20036 = context;
	int i = 0;
	unsigned char page = 0;
	unsigned char reg_addr = 0;
	unsigned char reg_val = 0;

	if (!cont) {
		pr_info("%s: failed to read %s\n", __func__, aw20036_cfg_name[aw20036->effect]);
		release_firmware(cont);
		return;
	}
	mutex_lock(&aw20036->cfg_lock);
	pr_info("%s: loaded %s - size: %zu\n", __func__,
		aw20036_cfg_name[aw20036->effect], cont ? cont->size : 0);

	for (i = 0; i < cont->size; i += 2) {
		if (*(cont->data + i) == 0xf0)
			page = *(cont->data + i + 1);
		aw20036_i2c_write(aw20036, *(cont->data + i), *(cont->data + i + 1));
		pr_debug("%s: addr:0x%02x, data:0x%02x\n", __func__,
			 *(cont->data + i), *(cont->data + i + 1));

		if (page == AW20036_REG_PAGE0) {
			reg_addr = *(cont->data + i);
			reg_val = *(cont->data + i + 1);
			/* gcr chip enable delay */
			if ((reg_addr == REG_SWRST) && (reg_val == 0x01))
				usleep_range(2000, 2500);
		}
	}

	release_firmware(cont);
	mutex_unlock(&aw20036->cfg_lock);
	pr_info("%s: cfg update complete\n", __func__);
}

static int aw20036_cfg_update(struct aw20036 *aw20036)
{
	int ret;

	ret = 0;

	if (aw20036->effect < (sizeof(aw20036_cfg_name) / AW20036_CFG_NAME_MAX)) {
		pr_info("%s: cfg name=%s\n", __func__, aw20036_cfg_name[aw20036->effect]);
	} else {
		pr_err("%s: effect 0x%02x over s value\n", __func__, aw20036->effect);
		return (-1);
	}

	return request_firmware_nowait(THIS_MODULE, FW_ACTION_HOTPLUG,
					aw20036_cfg_name[aw20036->effect],
					aw20036->dev, GFP_KERNEL, aw20036,
					aw20036_cfg_loaded);
}
#endif

static void aw20036_brightness_work(struct work_struct *work)
{
	struct aw20036 *aw20036 = container_of(work, struct aw20036, brightness_work);

	if (aw20036->cdev.brightness < 0
	|| aw20036->cdev.brightness >= (sizeof(aw20036_cfg_array) / sizeof(struct aw20036_cfg)))
	{
		pr_info("%s: invalid index\n", __func__);
		return;
	}

	aw20036->effect = aw20036->cdev.brightness;
	aw20036_cfg_update_array(aw20036);
}

static int aw20036_hw_reset(struct aw20036 *aw20036)
{
	if (aw20036 && gpio_is_valid(aw20036->reset_gpio)) {
		gpio_set_value_cansleep(aw20036->reset_gpio, 0);
		usleep_range(1000, 2000);
		gpio_set_value_cansleep(aw20036->reset_gpio, 1);
		usleep_range(2000, 2500);
	} else {
		dev_err(aw20036->dev, "%s:  failed\n", __func__);
	}

	return 0;
}

static int aw20036_hw_off(struct aw20036 *aw20036)
{
	if (aw20036 && gpio_is_valid(aw20036->reset_gpio)) {
		gpio_set_value_cansleep(aw20036->reset_gpio, 0);
		usleep_range(1000, 2000);
	} else {
		dev_err(aw20036->dev, "%s:  failed\n", __func__);
	}

	return 0;
}

/******************************************************
 *
 * irq
 *
 ******************************************************/
static irqreturn_t aw20036_irq(int irq, void *data)
{
	struct aw20036 *aw20036 = data;
	unsigned char reg_val;

	aw20036_i2c_read(aw20036, REG_ISRFLT, &reg_val);
	pr_info("%s: reg INTST=0x%x\n", __func__, reg_val);

	return IRQ_HANDLED;
}

/*****************************************************
 *
 * device tree
 *
 *****************************************************/
static int aw20036_parse_dt(struct device *dev, struct aw20036 *aw20036, struct device_node *np)
{
	aw20036->reset_gpio = of_get_named_gpio(np, "reset-gpio", 0);
	if (aw20036->reset_gpio < 0)
		dev_err(dev, "%s: no reset gpio provided, will not HW reset device\n", __func__);
	else
		dev_info(dev, "%s: reset gpio provided ok\n", __func__);

	aw20036->irq_gpio = of_get_named_gpio(np, "irq-gpio", 0);
	if (aw20036->irq_gpio < 0)
		dev_err(dev, "%s: no irq gpio provided, will not suppport intterupt\n", __func__);
	else
		dev_info(dev, "%s: irq gpio provided ok\n", __func__);

	return 0;
}

/*****************************************************
 *
 * check chip id
 *
 *****************************************************/
static int aw20036_read_chipid(struct aw20036 *aw20036)
{
	int ret = -1;
	unsigned char cnt = 0;
	unsigned char reg_val = 0;

	aw20036_reg_page_cfg(aw20036, AW20036_REG_PAGE0);

	while (cnt++ < AW_READ_CHIPID_RETRIES) {
		ret = aw20036_i2c_read(aw20036, REG_CHIPID, &reg_val);
		if (reg_val == AW20036_CHIPID) {
			pr_info("This Chip is  AW20036    REG_ID: 0x%x\n", reg_val);
			return 0;
		} else if (ret < 0) {
			dev_err(aw20036->dev, "%s: failed to AW20036_REG_ID: %d\n", __func__, ret);
		} else {
			pr_info("This Chip    read register   REG_ID: 0x%x\n", reg_val);
		}
		usleep_range(1000, 2000);
	}
	return -EINVAL;
}

/******************************************************
 *
 * sys group attribute: reg
 *
 ******************************************************/
static ssize_t reg_store(struct device *dev,
				 struct device_attribute *attr, const char *buf,
				 size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	unsigned int databuf[2] = { 0, 0 };

	if (sscanf(buf, "%x %x", &databuf[0], &databuf[1]) == 2)
		aw20036_i2c_write(aw20036, databuf[0], databuf[1]);
	return count;
}

static ssize_t reg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);
	ssize_t len = 0;
	unsigned int i = 0;
	unsigned char reg_val = 0;
	unsigned char reg_page = 0;

	aw20036_reg_page_cfg(aw20036, AW20036_REG_PAGE0);
	for (i = 0; i < AW20036_REG_MAX; i++) {
		if (!reg_page) {
			if (!(aw20036_reg_access[i] & REG_RD_ACCESS))
				continue;
		}
		aw20036_i2c_read(aw20036, i, &reg_val);
		len += snprintf(buf + len, PAGE_SIZE - len, "reg:0x%02x=0x%02x\n", i, reg_val);
	}
	return len;
}

static ssize_t
hwen_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);
	int ret = -1;
	unsigned int databuf[1] = { 0 };

	ret = kstrtou32(buf, 0, &databuf[0]);
	if (ret < 0) {
		dev_err(aw20036->dev, "%s: input data invalid!", __func__);
		return ret;
	}

	if (databuf[0] == 1) {
		aw20036_hw_reset(aw20036);
	} else {
		aw20036_hw_off(aw20036);
		aw20036_led_init(aw20036);
	}

	return count;
}

static ssize_t
hwen_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);
	ssize_t len = 0;

	len +=
	snprintf(buf + len, PAGE_SIZE - len, "hwen=%d\n", gpio_get_value(aw20036->reset_gpio));

	return len;
}

static ssize_t
effect_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	ssize_t len = 0;
	unsigned int i;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);
#if defined(AW20036_BIN_CONFIG)
	for (i = 0; i < sizeof(aw20036_cfg_name) / AW20036_CFG_NAME_MAX; i++) {
		len +=
		snprintf(buf + len, PAGE_SIZE - len, "cfg[%x] = %s\n", i, aw20036_cfg_name[i]);
	}
	len +=
	snprintf(buf + len, PAGE_SIZE - len, "current cfg = %s\n", aw20036_cfg_name[aw20036->effect]);
#elif defined(AW20036_ARRAY_CONFIG)
	for (i = 0; i < sizeof(aw20036_cfg_array) / sizeof(struct aw20036_cfg);
		 i++) {
		len +=
		snprintf(buf + len, PAGE_SIZE - len, "cfg[%x] = %ps\n", i, aw20036_cfg_array[i].p);
	}
	len +=
		snprintf(buf + len, PAGE_SIZE - len, "current cfg = %ps\n",
			 aw20036_cfg_array[aw20036->effect].p);
#else
	/*Nothing */
#endif
	return len;
}

static ssize_t
effect_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t len)
{
	unsigned int databuf[1] = { 0 };
	int ret = -1;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	ret = kstrtou32(buf, 0, &databuf[0]);
	if (ret < 0) {
		dev_err(aw20036->dev, "%s: input data invalid!", __func__);
		return ret;
	}

	aw20036->effect = databuf[0];
#if defined(AW20036_BIN_CONFIG)
	aw20036_cfg_update(aw20036);
#elif defined(AW20036_ARRAY_CONFIG)
	aw20036_cfg_update_array(aw20036);
#else
	/*Nothing */
#endif

	return len;
}

static ssize_t
imax_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t len)
{
	unsigned int databuf[1] = { 0 };
	int ret = -1;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	ret = kstrtou32(buf, 0, &databuf[0]);
	if (ret < 0) {
		dev_err(aw20036->dev, "%s: input data invalid!", __func__);
		return ret;
	}

	aw20036->imax = databuf[0];

	if (aw20036->imax > 0xF)
		aw20036->imax = 0xF;

	aw20036_imax_cfg(aw20036, aw20036->imax);

	return len;
}

static ssize_t
imax_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	ssize_t len = 0;
	unsigned int i;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	for (i = 0; i < sizeof(aw20036_imax_name) / AW20036_IMAX_NAME_MAX; i++) {
		len +=
		snprintf(buf + len, PAGE_SIZE - len, "imax[%x] = %s\n", i, aw20036_imax_name[i]);
	}
	len +=
		snprintf(buf + len, PAGE_SIZE - len,
			 "current id = 0x%02x, imax = %s\n", aw20036->imax,
			 aw20036_imax_name[aw20036->imax]);

	return len;
}

static ssize_t
rgbcolor_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t len)
{
	unsigned int databuf[2] = { 0, 0 };
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	if (sscanf(buf, "%x %x", &databuf[0], &databuf[1]) == 2) {
		aw20036_i2c_write(aw20036, 0xF0, 0xC0);
		aw20036_i2c_write(aw20036, 0x01, 0x00);
		aw20036_i2c_write(aw20036, 0xF0, 0xC1);

		aw20036->rgbcolor = (databuf[1] & 0x00ff0000) >> 16;
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, databuf[0] * 3, aw20036->rgbcolor);

		aw20036->rgbcolor = (databuf[1] & 0x0000ff00) >> 8;
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, databuf[0] * 3 + 1, aw20036->rgbcolor);

		aw20036->rgbcolor = (databuf[1] & 0x000000ff);
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, databuf[0] * 3 + 2, aw20036->rgbcolor);
	}
	return len;
}

static ssize_t
rgbbrightness_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t len)
{
	unsigned int databuf[2] = { 0, 0 };
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	if (sscanf(buf, "%x %x", &databuf[0], &databuf[1]) == 2) {
		aw20036_i2c_write(aw20036, 0xF0, 0xC0);
		aw20036_i2c_write(aw20036, 0x01, 0x00);
		aw20036_i2c_write(aw20036, 0xF0, 0xC2);
		aw20036->rgbbrightness = (databuf[1] & 0x00ff0000) >> 16;
		aw20036_i2c_write(aw20036, databuf[0] * 3, aw20036->rgbbrightness);

		aw20036->rgbbrightness = (databuf[1] & 0x0000ff00) >> 8;
		aw20036_i2c_write(aw20036, databuf[0] * 3 + 1, aw20036->rgbbrightness);

		aw20036->rgbbrightness = (databuf[1] & 0x000000ff);
		aw20036_i2c_write(aw20036, databuf[0] * 3 + 2, aw20036->rgbbrightness);
	}
	return len;
}

static ssize_t
allrgbcolor_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t len)
{
	unsigned int databuf[1] = { 0 };
	unsigned int i;
	int ret = -1;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	ret = kstrtou32(buf, 0, &databuf[0]);
	if (ret < 0) {
		dev_err(aw20036->dev, "%s: input data invalid!", __func__);
		return ret;
	}

	aw20036->rgbcolor = databuf[0];
	aw20036_i2c_write(aw20036, 0xF0, 0xC0);
	aw20036_i2c_write(aw20036, 0x01, 0x00);
	/*Set pag 1 DIM0-DIM35 */
	aw20036_i2c_write(aw20036, 0xF0, 0xC1);
	for (i = 0; i < AW20036_REG_NUM_PAG1; i += 3) {
		aw20036->rgbcolor = (databuf[0] & 0x00ff0000) >> 16;
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, i, aw20036->rgbcolor);

		aw20036->rgbcolor = (databuf[0] & 0x0000ff00) >> 8;
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, i + 1, aw20036->rgbcolor);

		aw20036->rgbcolor = (databuf[0] & 0x000000ff);
		aw20036->rgbcolor = (aw20036->rgbcolor * 64) / 256;
		aw20036_i2c_write(aw20036, i + 2, aw20036->rgbcolor);
		pr_debug("%s: addr:0x%02x, data:0x%02x\n", __func__, i,
			 aw20036->rgbcolor);
	}
	return len;
}

static ssize_t
allrgbbrightness_store(struct device *dev, struct device_attribute *attr,
			const char *buf, size_t len)
{
	unsigned int databuf[1] = { 0 };
	unsigned int i;
	int ret = -1;
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct aw20036 *aw20036 = container_of(led_cdev, struct aw20036, cdev);

	ret = kstrtou32(buf, 0, &databuf[0]);
	if (ret < 0) {
		dev_err(aw20036->dev, "%s: input data invalid!", __func__);
		return ret;
	}

	aw20036_i2c_write(aw20036, 0xF0, 0xC0);
	aw20036_i2c_write(aw20036, 0x01, 0x00);
	/*Set pag 2 PAD0-PAD35 */
	aw20036_i2c_write(aw20036, 0xF0, 0xC2);
	for (i = 0; i < AW20036_REG_NUM_PAG2; i += 3) {
		aw20036->rgbbrightness = (databuf[0] & 0x00ff0000) >> 16;
		aw20036_i2c_write(aw20036, i, aw20036->rgbbrightness);

		aw20036->rgbbrightness = (databuf[0] & 0x0000ff00) >> 8;
		aw20036_i2c_write(aw20036, i + 1, aw20036->rgbbrightness);

		aw20036->rgbbrightness = (databuf[0] & 0x000000ff);
		aw20036_i2c_write(aw20036, i + 2, aw20036->rgbbrightness);
		pr_debug("%s: addr:0x%02x, data:0x%02x\n", __func__, i, aw20036->rgbbrightness);
	}
	return len;
}

static DEVICE_ATTR_RW(reg);
static DEVICE_ATTR_RW(hwen);
static DEVICE_ATTR_RW(effect);
static DEVICE_ATTR_RW(imax);
static DEVICE_ATTR_WO(rgbcolor);
static DEVICE_ATTR_WO(rgbbrightness);
static DEVICE_ATTR_WO(allrgbcolor);
static DEVICE_ATTR_WO(allrgbbrightness);

static struct attribute *aw20036_attributes[] = {
	&dev_attr_reg.attr,
	&dev_attr_hwen.attr,
	&dev_attr_effect.attr,
	&dev_attr_imax.attr,
	&dev_attr_rgbcolor.attr,
	&dev_attr_allrgbcolor.attr,
	&dev_attr_rgbbrightness.attr,
	&dev_attr_allrgbbrightness.attr,
	NULL
};

static struct attribute_group aw20036_attribute_group = {
	.attrs = aw20036_attributes
};

/******************************************************
 *
 * led class dev
 *
 ******************************************************/
static int aw20036_led_init(struct aw20036 *aw20036)
{
	aw20036_i2c_write(aw20036, 0xF0, 0xC0);
	aw20036_i2c_write(aw20036, 0x02, 0x01);
	usleep_range(2000, 3000);
	aw20036_i2c_write(aw20036, 0X03, 0x10);	/*SET default */
	aw20036_imax_cfg(aw20036, aw20036->imax);/*set imax */
	aw20036_i2c_write_bits(aw20036, REG_GCCR, BIT_ALLON_MASK, BIT_GCR_ALLON_ENABLE);
	aw20036_i2c_write(aw20036, 0x80, 0x02);

	return 0;
}

static int aw20036_parse_led_cdev(struct aw20036 *aw20036, struct device_node *np)
{
	struct device_node *temp;
	int ret = -1;

	for_each_child_of_node(np, temp) {
		ret = of_property_read_string(temp, "aw20036,name", &aw20036->cdev.name);
		if (ret < 0) {
			dev_err(aw20036->dev, "Failure reading led name, ret = %d\n", ret);
			goto free_pdata;
		}
		ret = of_property_read_u32(temp, "aw20036,imax", &aw20036->imax);
		if (ret < 0) {
			dev_err(aw20036->dev, "Failure reading imax, ret = %d\n", ret);
			goto free_pdata;
		}
		ret = of_property_read_u32(temp, "aw20036,brightness", &aw20036->cdev.brightness);
		if (ret < 0) {
			dev_err(aw20036->dev, "Failure reading brightness, ret = %d\n", ret);
			goto free_pdata;
		}
		ret = of_property_read_u32(temp, "aw20036,max_brightness",
					   &aw20036->cdev.max_brightness);
		if (ret < 0) {
			dev_err(aw20036->dev,
				"Failure reading max brightness, ret = %d\n",
				ret);
			goto free_pdata;
		}
	}
	aw20036_led_init(aw20036);
	INIT_WORK(&aw20036->brightness_work, aw20036_brightness_work);
	aw20036->cdev.brightness_set = aw20036_set_brightness;
	ret = led_classdev_register(aw20036->dev, &aw20036->cdev);
	if (ret) {
		dev_err(aw20036->dev, "unable to register led ret=%d\n", ret);
		goto free_pdata;
	}
	ret =
		sysfs_create_group(&aw20036->cdev.dev->kobj, &aw20036_attribute_group);
	if (ret) {
		dev_err(aw20036->dev, "led sysfs ret: %d\n", ret);
		goto free_class;
	}
	return 0;

 free_class:
	led_classdev_unregister(&aw20036->cdev);
 free_pdata:
	return ret;
}

/******************************************************
 *
 * i2c driver
 *
 ******************************************************/
static int aw20036_i2c_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct aw20036 *aw20036;
	struct device_node *np = i2c->dev.of_node;
	int ret;
	int irq_flags;
	pr_info("%s probe start!\n", __func__);
	if (!i2c_check_functionality(i2c->adapter, I2C_FUNC_I2C)) {
		dev_err(&i2c->dev, "check_functionality failed\n");
		return -EIO;
	}

	aw20036 = devm_kzalloc(&i2c->dev, sizeof(struct aw20036), GFP_KERNEL);
	if (aw20036 == NULL)
		return -ENOMEM;

	aw20036->dev = &i2c->dev;
	aw20036->i2c = i2c;

	i2c_set_clientdata(i2c, aw20036);

	mutex_init(&aw20036->cfg_lock);

	/* aw20036 rst & int */
	if (np) {
		ret = aw20036_parse_dt(&i2c->dev, aw20036, np);
		pr_err("%s: aw20036 rst & int ret: %d", __func__, ret);
		if (ret) {
			dev_err(&i2c->dev, "%s: failed to parse device tree node\n", __func__);
			goto err_parse_dt;
		}
	} else {
		aw20036->reset_gpio = -1;
		aw20036->irq_gpio = -1;
	}

	if (gpio_is_valid(aw20036->reset_gpio)) {
		ret = devm_gpio_request_one(&i2c->dev, aw20036->reset_gpio,
						GPIOF_OUT_INIT_LOW, "aw20036_rst");
		if (ret) {
			dev_err(&i2c->dev, "%s: rst request failed\n", __func__);
			goto err_gpio_request;
		}
	}

	if (gpio_is_valid(aw20036->irq_gpio)) {
		ret = devm_gpio_request_one(&i2c->dev, aw20036->irq_gpio,
						GPIOF_DIR_IN, "aw20036_int");
		if (ret) {
			dev_err(&i2c->dev, "%s: int request failed\n", __func__);
			goto err_gpio_request;
		}
	}

	/* hardware reset */
	aw20036_hw_reset(aw20036);

	/* aw20036 chip id */
	ret = aw20036_read_chipid(aw20036);
	if (ret < 0) {
		dev_err(&i2c->dev, "%s: aw20036_read_chipid failed ret=%d\n", __func__, ret);
		goto err_id;
	}

	/* aw22xxx irq */
	if (gpio_is_valid(aw20036->irq_gpio) &&
		!(aw20036->flags & AW20036_FLAG_SKIP_INTERRUPTS)) {
		/* register irq handler */
		irq_flags = IRQF_TRIGGER_FALLING | IRQF_ONESHOT;
		ret = devm_request_threaded_irq(&i2c->dev,
						gpio_to_irq(aw20036->irq_gpio),
						NULL, aw20036_irq, irq_flags,
						"aw20036", aw20036);
		if (ret != 0) {
			dev_err(&i2c->dev, "%s: failed to request IRQ %d: %d\n",
				__func__, gpio_to_irq(aw20036->irq_gpio), ret);
			goto err_irq;
		}
	} else {
		dev_info(&i2c->dev, "%s skipping IRQ registration\n", __func__);
		/* disable feature support if gpio was invalid */
		aw20036->flags |= AW20036_FLAG_SKIP_INTERRUPTS;
	}

	dev_set_drvdata(&i2c->dev, aw20036);

	aw20036_parse_led_cdev(aw20036, np);
	if (ret < 0) {
		dev_err(&i2c->dev, "%s error creating led class dev\n", __func__);
		goto err_sysfs;
	}

	pr_info("%s probe completed successfully!\n", __func__);
	return 0;

 err_sysfs:
	devm_free_irq(&i2c->dev, gpio_to_irq(aw20036->irq_gpio), aw20036);
 err_irq:
 err_id:
	gpio_free(aw20036->reset_gpio);
	gpio_free(aw20036->irq_gpio);
 err_gpio_request:
 err_parse_dt:
	devm_kfree(&i2c->dev, aw20036);
	aw20036 = NULL;
	return ret;
}

static int aw20036_i2c_remove(struct i2c_client *i2c)
{
	struct aw20036 *aw20036 = i2c_get_clientdata(i2c);

	sysfs_remove_group(&aw20036->cdev.dev->kobj, &aw20036_attribute_group);
	led_classdev_unregister(&aw20036->cdev);

	devm_free_irq(&i2c->dev, gpio_to_irq(aw20036->irq_gpio), aw20036);

	if (gpio_is_valid(aw20036->reset_gpio))
		gpio_free(aw20036->reset_gpio);
	if (gpio_is_valid(aw20036->irq_gpio))
		gpio_free(aw20036->irq_gpio);

	devm_kfree(&i2c->dev, aw20036);
	aw20036 = NULL;

	return 0;
}

static const struct i2c_device_id aw20036_i2c_id[] = {
	{AW20036_I2C_NAME, 0},
	{}
};

MODULE_DEVICE_TABLE(i2c, aw20036_i2c_id);

static const struct of_device_id aw20036_dt_match[] = {
	{.compatible = "awinic,aw20036_led"},
	{},
};

static struct i2c_driver aw20036_i2c_driver = {
	.driver = {
		   .name = AW20036_I2C_NAME,
		   .owner = THIS_MODULE,
		   .of_match_table = of_match_ptr(aw20036_dt_match),
		   },
	.probe = aw20036_i2c_probe,
	.remove = aw20036_i2c_remove,
	.id_table = aw20036_i2c_id,
};

static int __init aw20036_i2c_init(void)
{
	int ret = 0;

	pr_info("aw20036 driver version %s\n", AW20036_DRIVER_VERSION);
	ret = i2c_add_driver(&aw20036_i2c_driver);
	pr_err("ret= %d\n", ret);
	if (ret) {
		pr_err("fail to add aw20036 device into i2c\n");
		return ret;
	}
	return 0;
}

module_init(aw20036_i2c_init);

static void __exit aw20036_i2c_exit(void)
{
	i2c_del_driver(&aw20036_i2c_driver);
}

module_exit(aw20036_i2c_exit);

MODULE_DESCRIPTION("AW20036 LED Driver");
MODULE_LICENSE("GPL v2");
