/**
 * Sensortek SIP1221LR1S/STK3311 Ambient Light and Proximity Sensor
 *
 * Copyright (c) 2015, Intel Corporation.
 *
 * This file is subject to the terms and conditions of version 2 of
 * the GNU General Public License. See the file COPYING in the main
 * directory of this archive for more details.
 *
 * IIO driver for SIP1221LR1S/STK3311. 7-bit I2C address: 0x48.
 */

#include <linux/acpi.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/regmap.h>
#include <linux/gpio/consumer.h>
#include <linux/iio/events.h>
#include <linux/iio/iio.h>
#include <linux/iio/sysfs.h>
#include <linux/delay.h>
#include "sip1221lr1s.h"

#define SIP1221LR1S_DRIVER_NAME "sip1221lr1s"
#define SIP1221LR1S_EVENT "sip1221lr1s_event"
#define SIP1221LR1S_SCALE_AVAILABLE "0.25 0.5 1 2 4 8 16 32 64 128 256 512 1024"
#define SIP1221LR1S_IT_AVAILABLE    "20 50 70 100 200"

// data[0] delay timer  data[1] inte time
static const int sip1221lr1s_it_table[][2] = {
    {25, 20},
    {55, 50},
    {75, 70},
    {105, 100},
    {205, 200},
};

enum POLLING_PERIOD
{
    ALS_DELAY_20MS,
    ALS_DELAY_50MS,
    ALS_DELAY_70MS,
    ALS_DELAY_100MS,
    ALS_DELAY_200MS,
    ALS_DELAY_MAX,
};

struct sip1221lr1s_data
{
    struct i2c_client *client;
    struct mutex lock;
    struct sip1221lr1s_device *dev;

    bool als_enabled;
    bool ps_enabled;
    u64 timestamp;

    uint16_t als_raw;
    uint16_t ir_raw;
    uint16_t inte_time_index;
    uint16_t inte_time;
    uint32_t max_range;
};

static int sip_read_regs(uint8_t reg, uint8_t *value, uint8_t len);
static int sip_write_regs(uint8_t reg, uint8_t *value, uint8_t len);
static int sip_write_reg(uint8_t reg, uint8_t value);
static int sip_read_reg(uint8_t reg, uint8_t *p_value);
static int sip_modify_reg(uint8_t reg, uint8_t value, uint8_t mask);
static int sip_check_reg(uint8_t reg, uint8_t value, uint8_t mask);

#if USE_SERIAL_WORK_MODE_OPTION
static int sip1221lr1s_ps_is_enabled(sip_i2c_info_t *i2c_info);
#endif

static struct sip1221lr1s_device g_sensor_device =
{
        .als_data_is_ready = 0,
        .als_data_is_ready = 0,
        .iinfo = {
            .write_reg = sip_write_reg,
            .write_regs = sip_write_regs,
            .read_reg = sip_read_reg,
            .read_regs = sip_read_regs,
            .modify_reg = sip_modify_reg,
            .check_reg = sip_check_reg,
        },

        .ainfo = {.conf = {
                      .is_dri = false,
                      .gain0_count = 10,
                      .gain1_count = 10,
                      .short_inte_time = 15,
#if USE_SERIAL_WORK_MODE_OPTION
                      .long_inte_time = 80,
#else
                      .long_inte_time = 100,
#endif /* USE_SERIAL_WORK_MODE_OPTION */
                  },
                  .out = {.raw = {0, 0}, .lux_raw = 0, .ir_ratio = 0, .lux = 0},
                  .cali = {
                      .target_ch0 = 1900,
                      .target_ch1 = 1200,
                      .target_lux = 500,
                      .dark_ch0_coef = 0,
                      .dark_ch1_coef = 0,
                      .dark_lux_coef = 0,
                      .light_ch0_coef = 1000,
                      .light_ch1_coef = 1000,
                      .light_lux_coef = 1000,
                      .cali_result = ALS_CALI_INIT,
                  },
                  .sp = NULL,
                  .stat = {.data_valid = 0x00, .int_status = 0x00, .enable_status = 0x00, .enabled = false, .first_frame_flag_after_enable = false, .data_changed = false}},
};
static const sip_als_gain_info_t als_gains[] =
{
    {250, 250},
    {500, 500},
    {1000, 1000},
    {2000, 2000},
    {4000, 4000},
    {8000, 8000},
    {16000, 16000},
    {32000, 32000},
    {64000, 64000},
    {128000, 128000},
    {256000, 256000},
    {512000, 512000},
    {1024000, 1024000},
};

static const sip1221lr1s_spectral_param sp_primary = {
    1000, 0, 0, 1000, 0, 0,
    1000, 0, 0, 1000, 1000, 0,
    1000, 0, 0, 1000, 1000, 0,
    505, 558, 7000,
    0, 236994, 0, 226569, 0, 224912, 0, 224912,
};

static int i2c_write_regs(struct i2c_client *client, uint8_t reg, uint8_t *value, int len)
{
    int ret;
    struct i2c_msg msg;
    uint8_t data[64];
    if (len > 63)
    {
        log_e("write buffer to long\n");
        return -1;
    }

    data[0] = reg;
    memcpy(&data[1], value, len);
    msg.addr = client->addr; /*I2C addr*/
    msg.flags = 0;           /*write*/
    msg.buf = data;          /*reg*/
    msg.len = len + 1;       /*len*/
    ret = i2c_transfer(client->adapter, &msg, 1);

    if (ret < 0)
    {
        log_e(" i2c transfer error,ret = %d\n", ret);
    }
    return ret;
}

static int i2c_read_regs(struct i2c_client *client, uint8_t reg, uint8_t *value, int len)
{
    int ret;
    struct i2c_msg msg[2];

    msg[0].addr = client->addr; /*I2C addr*/
    msg[0].flags = 0;           /*write*/
    msg[0].buf = &reg;          /*reg*/
    msg[0].len = 1;             /*len*/

    /*msg[1]*/
    msg[1].addr = client->addr; /*I2C addr*/
    msg[1].flags = I2C_M_RD;    /*read*/
    msg[1].buf = value;         /*read reg*/
    msg[1].len = len;           /*reg len*/

    ret = i2c_transfer(client->adapter, msg, 2);
    if (ret < 0)
    {
        log_e(" i2c transfer error,ret = %d\n", ret);
    }
    return ret;
}

static int sip_read_regs(uint8_t reg, uint8_t *value, uint8_t len)
{
    struct i2c_client *client = g_sensor_device.client;
    return i2c_read_regs(client, reg, value, len);
}

static int sip_write_regs(uint8_t reg, uint8_t *value, uint8_t len)
{
    struct i2c_client *client = g_sensor_device.client;

    return i2c_write_regs(client, reg, value, len);
}

static int sip_write_reg(uint8_t reg, uint8_t value)
{
    return sip_write_regs(reg, &value, 1);
}

static int sip_read_reg(uint8_t reg, uint8_t *p_value)
{
    return sip_read_regs(reg, p_value, 1);
}

static int sip_modify_reg(uint8_t reg, uint8_t value, uint8_t mask)
{
    int rv = -1;
    uint8_t rw_buffer = 0;

    if (0xFF == mask)
    {
        rv = sip_write_reg(reg, value);
    }
    else
    {
        /* Read current value from this register */
        rv = sip_read_reg(reg, &rw_buffer);
        if (rv >= 0)
        {
            rw_buffer &= ~mask;
            rw_buffer |= (value & mask);
            rv = sip_write_reg(reg, rw_buffer);
        }
    }

    return rv;
}

static int sip_check_reg(uint8_t reg, uint8_t value, uint8_t mask)
{
    int rv = -1;
    uint8_t r_value = 0;

    rv = sip_read_reg(reg, &r_value);

    log_i("sip_check_reg, sip_read_reg: %d,%d", reg, r_value);

    if (rv >= 0)
    {
        if ((r_value & mask) != (value & mask))
        {
            rv = -2;
        }
    }

    return rv;
}

sip_reg_t esd_regs[] = {
    {SIP1221_CLK_CTRL, 0x33},
    {SIP1221_CTRL, 0x03},
};

int sip1221lr1s_esd_check_conf(sip_i2c_info_t *i2c_info)
{
    uint8_t i;
    uint8_t value = 0x00;

    for (i = 0; i < ARR_SIZE(esd_regs); ++i)
    {
        if (0 == esd_regs[i].reg)
        {
            continue;
        }

        i2c_info->read_reg(esd_regs[i].reg, &value);
        if (value != esd_regs[i].value)
        {
            log_i("read 0x%02X: 0x%02X", esd_regs[i].reg, value);
            return -1;
        }
    }

    return 0;
}

static void timer_enable(struct timer_list *timer, int ms)
{
    timer->expires = jiffies + msecs_to_jiffies(ms);
    add_timer(timer);
}

static void timer_disable(struct timer_list *timer)
{
    del_timer_sync(timer);
}

#if SIP1221LR1S_SUPPORT_DUMP
int sip1221lr1s_dump_register(sip_i2c_info_t *i2c_info)
{
    int i;
    uint8_t reg_value;
    sip_reg_t sip1221_dump_regs[] = {
        {SIP1221_CLK_CTRL, 0x00},
        {SIP1221_CTRL, 0x00},
        {SIP1221_CTRL, 0x00},
        {SIP1221_INT_CTRL, 0x00},
        {SIP1221_ALS_CTRL0, 0x00},
        {SIP1221_ALS_CTRL1, 0x00},
        {SIP1221_ALS_INTEN, 0x00},
        {SIP1221_ALS_AZ_CTRL, 0x00},
        {SIP1221_ALS_THLOW_H, 0x00},
        {SIP1221_ALS_THLOW_L, 0x00},
        {SIP1221_ALS_THHIGH_H, 0x00},
        {SIP1221_ALS_THHIGH_L, 0x00},
        {SIP1221_ALS_INTE_TIME_H, 0x00},
        {SIP1221_ALS_INTE_TIME_M, 0x00},
        {SIP1221_ALS_INTE_TIME_L, 0x00},
        {SIP1221_ALS0_1_GAIN, 0x00},
        {SIP1221_ALS_AZ_EN, 0x00},
        {SIP1221_RESERVE1, 0x00},
        {SIP1221_ALS0_1_GAIN, 0x00},
        {0x6F, 0x56},
        {0x70, 0x35},
    };

    for (i = 0; i < ARR_SIZE(sip1221_dump_regs); ++i)
    {
        i2c_info->read_reg(sip1221_dump_regs[i].reg, &reg_value);
        log_i("reg[0x%02x]=0x%02x", sip1221_dump_regs[i].reg, reg_value);
    }
    return 0;
}
#endif

static int sip1221lr1s_als_check_trim(sip_i2c_info_t *i2c_info)
{
    unsigned int i;
    uint8_t zero_cnt = 0;
    sip_reg_t trim_regs[] = {
        {0xd4, 0x00},
        {0xd5, 0x00},
        {0xd6, 0x00},
        {0xd7, 0x00},
    };

    for (i = 0; i < ARR_SIZE(trim_regs); ++i)
    {
        i2c_info->read_reg(trim_regs[i].reg, &trim_regs[i].value);
        if (trim_regs[i].value == 0)
        {
            zero_cnt++;
        }
    }

    if (zero_cnt == 4)
    {
        i2c_info->write_reg(0xB0, 0x00);
        i2c_info->write_reg(0xB1, 0x00);
    }

    return 0;
}

static int sip1221lr1s_als_set_lcd_type(sip1221lr1s_als_info_t *als_info, int als_lcd_type)
{
    als_info->tmp.lcd_type = als_lcd_type;
    log_i("lcd_type: %d", als_info->tmp.lcd_type);

    switch (als_lcd_type)
    {
    case PRIMARY_LCD_PANEL:
    default:
        log_d("PRIMARY_LCD_PANEL");
        als_info->sp = &sp_primary;
        break;
    }

    return 0;
}

static int sip1221lr1s_als_load_conf(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
    unsigned int i;
    uint8_t is_dri = 0x00;
    uint8_t als_int_enable = 0x00;
    sip_reg_t init_regs[] = {
        {SIP1221_CTRL, 0x03},
        {SIP1221_INT_CTRL, 0x00},
        {SIP1221_ALS_CTRL0, 0xA0},
        {SIP1221_ALS_CTRL1, 0x1C},
        {SIP1221_ALS_INTEN, 0x80},
        {SIP1221_ALS0_1_GAIN, (uint8_t)(als_info->conf.gain0_count << 4 | als_info->conf.gain1_count)},
        {SIP1221_ALS_AZ_CTRL, 0xFF},
        {SIP1221_ALS_THLOW_H, 0x00},
        {SIP1221_ALS_THLOW_L, 0x0A},
        {SIP1221_ALS_THHIGH_H, 0x00},
        {SIP1221_ALS_THHIGH_L, 0x1A},
        {SIP1221_ALS_AZ_EN, 0xC1},
        {SIP1221_RESERVE1, 0x05},
        {SIP1221_ALS0_1_GAIN, 0x77},
        {0x6F, 0x56},
        {0x70, 0x35},
    };

#if USE_SERIAL_WORK_MODE_OPTION
    is_dri |= WORK_MODE;
#endif

    if (als_info->conf.is_dri == true)
    {
        is_dri |= INT_EN;
        als_int_enable = 0x01;
    }

    SIP_VERIFY2(i2c_info->write_reg(SIP1221_INT_CTRL, is_dri) >= 0,
                "light sensor init- failed!");
    SIP_VERIFY2(i2c_info->write_reg(SIP1221_ALS_INTEN, als_int_enable) >= 0,
                "light sensor init- failed!");

    als_info->tmp.gain0_count = als_info->conf.gain0_count;
    als_info->tmp.gain1_count = als_info->conf.gain1_count;

    for (i = 0; i < ARR_SIZE(init_regs); ++i)
    {
        SIP_VERIFY2(i2c_info->write_reg(init_regs[i].reg, init_regs[i].value) >= 0,
                    "light sensor init-%d failed!", i);
    }

    sip1221lr1s_als_set_lcd_type(als_info, 0);

    log_i("light sensor init success!");
    return 0;
}

static int sip1221lr1s_als_set_inte_time(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info, uint32_t inte_time)
{
    uint32_t al_inte_step_time = 0;
    uint8_t tx_buf[3] = {0};

    al_inte_step_time = (uint32_t)(inte_time * 1000000 / 1358 - 1);
    tx_buf[0] = (unsigned char)((al_inte_step_time >> 16) & 0x03);
    tx_buf[1] = (unsigned char)((al_inte_step_time >> 8) & 0xFF);
    tx_buf[2] = (unsigned char)(al_inte_step_time & 0xFF);

    SIP_ASSERT(i2c_info->write_regs(SIP1221_ALS_INTE_TIME_H, tx_buf, sizeof(tx_buf)));

    als_info->tmp.inte_time = inte_time;
    if (tx_buf[0] > 0)
    {
        als_info->tmp.max_range = MAX_ALS_VALUE;
    }
    else
    {
        als_info->tmp.max_range = min((((tx_buf[1] << 8) | tx_buf[2]) + 1), MAX_ALS_VALUE);
    }
    return 0;
}

static int sip1221lr1s_als_enable_private(sip_i2c_info_t *i2c_info, bool bo)
{
    // if (bo)
    // {
    //     SIP_ASSERT(i2c_info->write_reg(SIP1221l_ALS_AZ_EN, 0xC1));
    //     mdelay(1);
    //     SIP_ASSERT(i2c_info->write_reg(SIP1221_ALS_AZ_EN, 0xF1));
    // }
    SIP_ASSERT(i2c_info->modify_reg(SIP1221_ALS_ENABLE, bo ? 0x1F : 0x00, 0x1F));
    return 0;
}

#if USE_SERIAL_WORK_MODE_OPTION
static int sip1221lr1s_als_is_enabled(sip_i2c_info_t *i2c_info)
{
    uint8_t reg_als_enable = 0x00;
    SIP_ASSERT(i2c_info->read_reg(SIP1221_ALS_ENABLE, &reg_als_enable));
    return (reg_als_enable & (ALS0_EN | ALS1_EN | ALS_EN)) == (ALS0_EN | ALS1_EN | ALS_EN);
}

#endif /* USE_SERIAL_WORK_MODE_OPTION */
static int sip1221lr1s_als_set_enable(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info, bool bo)
{
    if (als_info->stat.enabled == bo)
    {
        log_e("repeatedly enable als: %d", bo);
        return 0;
    }

    if (bo)
    {
#if !ALS_WAIT_TIMEOUT_OPTION
        sip1221lr1s_als_set_inte_time(i2c_info, als_info, als_info->conf.short_inte_time);
#endif /* ALS_WAIT_TIMEOUT_OPTION */
        als_info->stat.first_frame_flag_after_enable = true;
    }

#if USE_SERIAL_WORK_MODE_OPTION

#endif /* USE_SERIAL_WORK_MODE_OPTION */

    als_info->stat.enabled = bo;

    sip1221lr1s_als_enable_private(i2c_info, bo);

    log_i("status = %d", bo);

    return 0;
}

static int sip1221lr1s_als_get_stat(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
#if ALS_WAIT_TIMEOUT_OPTION
    for (int i = 0; i < 10; ++i)
    {
        SIP_ASSERT(i2c_info->read_reg(SIP1221_DATA_VALID, &(als_info->stat.data_valid)));

        if (!als_info->stat.first_frame_flag || (als_info->stat.data_valid & ALS_DATA_VALID))
        {
            break;
        }
        else
        {
            mdelay(10);
        }
    }
#else
    SIP_ASSERT(i2c_info->read_reg(SIP1221_DATA_VALID, &(als_info->stat.data_valid)));
#endif /* ALS_WAIT_TIMEOUT_OPTION */

    SIP_ASSERT(i2c_info->read_reg(SIP1221_ALS_INT_STATUS, &(als_info->stat.int_status)));
    SIP_ASSERT(i2c_info->read_reg(SIP1221_ALS_ENABLE, &(als_info->stat.enable_status)));

    return 0;
}

static int sip1221lr1s_soft_reset(sip_i2c_info_t *i2c_info)
{
    SIP_ASSERT(i2c_info->write_reg(SIP1221_CTRL, 0x00));
    return 0;
}

static int sip1221lr1s_als_analyse_stat(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
    int ret = 0;
    if ((als_info->stat.data_valid & ALS_DATA_VALID))
    {
        als_info->stat.data_invalid_times = 0;
    }
    else
    {
        als_info->stat.data_invalid_times++;
        ret = -1;
#if SIP1221LR1S_SUPPORT_DUMP
        sip1221lr1s_dump_register(i2c_info);
#endif
    }

    if (sip1221lr1s_esd_check_conf(i2c_info))
    {
        als_info->stat.data_invalid_times++;
        ret = -1;
    }

    if (als_info->stat.data_invalid_times > 10)
    {
        log_i("trigger esd check");
        als_info->stat.data_invalid_times = 0;
        sip1221lr1s_soft_reset(i2c_info);
        mdelay(5);
        sip1221lr1s_als_load_conf(i2c_info, als_info);
        if (als_info->stat.enabled)
        {
            sip1221lr1s_als_enable_private(i2c_info, true);
        }
    }
    return ret;
}

static int sip1221lr1s_als_read_raw_data(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
    uint8_t rx_buf[8] = {0};
    SIP_ASSERT(i2c_info->read_regs(SIP1221_ALS_DATA, rx_buf, sizeof(rx_buf)));

    als_info->out.raw[0] = (uint16_t)((uint16_t)rx_buf[4] << 8) | rx_buf[5]; // als
    als_info->out.raw[1] = (uint16_t)((uint16_t)rx_buf[6] << 8) | rx_buf[7]; // wb

    return 0;
}

static int sip1221lr1s_als_set_gain(sip_i2c_info_t *i2c_info, uint16_t channel, uint32_t again)
{
    uint8_t count = 0;
    uint32_t tmp_gain = (uint32_t)(again * 4 / 1000);

    while ((tmp_gain / 2) != 0)
    {
        count++;
        tmp_gain /= 2;
    }

    SIP_ASSERT(sip1221lr1s_als_enable_private(i2c_info, false));
    if (channel)
    {
        i2c_info->modify_reg(SIP1221_ALS0_1_GAIN, (uint8_t)(count << 4), ALS1_GAIN_MASK);
    }
    else
    {
        i2c_info->modify_reg(SIP1221_ALS0_1_GAIN, count, ALS0_GAIN_MASK);
    }
    SIP_ASSERT(sip1221lr1s_als_enable_private(i2c_info, true));

    return 0;
}

static uint32_t find_highest_bit(uint32_t value)
{
    uint32_t bit_index;
    if (value == 0)
        return 0;

    bit_index = INTEGER_SIZE - 1; // 假设整数为10位

    while ((value & (1 << bit_index)) == 0)
    {
        bit_index--;
    }

    return bit_index;
}

static int sip1221lr1s_als_optimize_gain(uint32_t maximum_adc, uint32_t max,
                                         uint8_t *gain_index, uint8_t *saturation)
{
    uint32_t gain_change = 0;
    if (max == 0)
        max = 1;
    if (max >= maximum_adc)
    {
        if (*gain_index > LOW_AUTO_GAIN_VALUE) // Lower_Auto_gain_value is 3
            *gain_index /= AUTO_GAIN_DIV;
        else
            *gain_index = LOWEST_GAIN_ID;
        *saturation = 1;
    }
    else
    {
        gain_change = (uint32_t)((maximum_adc * SATURATION_LOW_PERCENT) /
                                 (max * SATURATION_HIGH_PERCENT));
        if (gain_change == 0 && (*gain_index) != 0)
        {
            (*gain_index)--;
        }
        else
        {
            gain_change = (uint32_t)find_highest_bit(gain_change);
            if (((*gain_index) + gain_change) > MAX_GAIN_ID)
                *gain_index = MAX_GAIN_ID;
            else
                *gain_index += gain_change;
        }
        *saturation = 0;
    }
    return 0;
}

static bool sip1221lr1s_als_auto_gain(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
    uint8_t old_gain0_count, old_gain1_count, new_gain0_count, new_gain1_count;
    uint8_t saturation_status = 0;
    uint32_t rgain0, rgain1;
    uint16_t max;
#if EN_SHUNT_GAIN
    uint8_t ch0_saturation_status = 0, ch1_saturation_status = 0;
#endif /* EN_SHUNT_GAIN */

    old_gain0_count = new_gain0_count = als_info->tmp.gain0_count;
    old_gain1_count = new_gain1_count = als_info->tmp.gain1_count;

    rgain0 = als_gains[old_gain0_count].tgain;
    rgain1 = als_gains[old_gain1_count].rgain;

    als_info->tmp.tgain0 = als_gains[old_gain0_count].tgain;
    als_info->tmp.tgain1 = als_gains[old_gain1_count].tgain;

    als_info->tmp.cpl0 = rgain0 * als_info->tmp.inte_time / 1000;
    als_info->tmp.cpl1 = rgain1 * als_info->tmp.inte_time / 1000;

#if EN_SHUNT_GAIN
    SIP_ASSERT(sip1221lr1s_als_optimize_gain(als_info->tmp.max_range, als_info->out.raw[0], &new_gain0_count, &ch0_saturation_status));
    if (old_gain0_count != new_gain0_count)
    {
        SIP_ASSERT(sip1221lr1s_als_set_gain(i2c_info, 0, (uint16_t)als_gains[new_gain0_count].tgain));
        als_info->tmp.gain0_count = new_gain0_count;
    }

    SIP_ASSERT(sip1221lr1s_als_optimize_gain(als_info->tmp.max_range, als_info->out.raw[1], &new_gain1_count, &ch1_saturation_status));
    if (old_gain1_count != new_gain1_count)
    {
        SIP_ASSERT(sip1221lr1s_als_set_gain(i2c_info, 1, (uint16_t)als_gains[new_gain1_count].tgain));
        als_info->tmp.gain1_count = new_gain1_count;
    }
    saturation_status = ch0_saturation_status || ch1_saturation_status;
    log_i("old_gain0_count = %d, new_gain0_count = %d, old_gain1_count = %d, new_gain1_count = %d",
          old_gain0_count, new_gain0_count, old_gain1_count, new_gain1_count);
    log_i("ch0_saturation_status = %d, ch1_saturation_status = %d",
          ch0_saturation_status, ch1_saturation_status);
#else
    max = MAX(als_info->out.raw[0], als_info->out.raw[1]);
    if(!als_info->stat.cali_enable) {
        SIP_ASSERT(sip1221lr1s_als_optimize_gain(als_info->tmp.max_range, max, &new_gain0_count, &saturation_status));
        if (old_gain0_count != new_gain0_count)
        {
            SIP_ASSERT(sip1221lr1s_als_set_gain(i2c_info, 0, (uint32_t)als_gains[new_gain0_count].tgain));
            als_info->tmp.gain0_count = new_gain0_count;
            new_gain1_count = new_gain0_count;
            SIP_ASSERT(sip1221lr1s_als_set_gain(i2c_info, 1, (uint32_t)als_gains[new_gain0_count].tgain));
            als_info->tmp.gain1_count = new_gain1_count;
        }
    }
#endif /* EN_SHUNT_GAIN */

    return saturation_status;
}

static int sip1221lr1s_als_compute_lux(sip1221lr1s_als_info_t *als_info)
{
    uint32_t ch0_raw = als_info->out.raw[1];
    uint32_t ch1_raw = als_info->out.raw[0];
    uint32_t k_lux = als_info->cali.light_lux_coef;
    uint32_t ratio0 = 0, ratio1 = 0, ratio2 = 0;
    int k_ch0_A = 0, k_ch1_A = 1;
    int k_ch0_D = 0, k_ch1_D = 1;
    int k_ch0_H = 0, k_ch1_H = 1;
    uint32_t cpl0 = als_info->tmp.cpl0;
    static int light_type = C_LIGHT;

    if (als_info->sp == NULL)
    {
        log_e("sp is NULL");
        return -1;
    }

    if (ch0_raw <= 0)
    {
        ch0_raw = 1;
    }

    if (cpl0 <= 0 || ch1_raw < 2)
    {
        als_info->out.lux = als_info->out.lux_raw = 0;
    }
    else
    {
        ratio0 = ((ch0_raw * 1000) / ch1_raw);

        if (ratio0 < als_info->sp->t_ratio_CD)
        {
            als_info->out.lux = (uint64_t)((uint64_t)ch0_raw * als_info->sp->k_C_ch0 + (uint64_t)ch1_raw * als_info->sp->k_C_ch1) * k_lux / 1000000;
            light_type = C_LIGHT;
        }
        else
        {
            k_ch0_A = als_info->sp->k_DA_01 +
                      als_info->sp->k_DA_02 * (k_lux - 1000) / 1000 +
                      als_info->sp->k_DA_03 * (k_lux - 1000) * (k_lux - 1000) / 1000000;
            k_ch1_A = als_info->sp->k_DA_11 +
                      als_info->sp->k_DA_12 * (k_lux - 1000) / 1000 +
                      als_info->sp->k_DA_13 * (k_lux - 1000) * (k_lux - 1000) / 1000000;

            ratio1 = ratio0 * k_ch0_A / k_ch1_A;

            if (ratio1 < als_info->sp->t_ratio_DA)
            {
                k_ch0_D = als_info->sp->k_D_01 +
                          als_info->sp->k_D_02 * (k_lux - 1000) / 1000 +
                          als_info->sp->k_D_03 * (k_lux - 1000) * (k_lux - 1000) / 1000000;
                k_ch1_D = als_info->sp->k_D_11 +
                          als_info->sp->k_D_12 * (k_lux - 1000) / 1000 +
                          als_info->sp->k_D_13 * (k_lux - 1000) * (k_lux - 1000) / 1000000;

                als_info->out.lux = (uint64_t)((uint64_t)ch0_raw * k_lux * als_info->sp->k_D_ch0 + (uint64_t)ch1_raw * k_lux * als_info->sp->k_D_ch1) / 1000000;
                light_type = D65_LIGHT;
            }
            else
            {
                k_ch0_H = als_info->sp->k_AH_01 +
                          als_info->sp->k_AH_02 * (k_lux - 1000) / 1000 +
                          als_info->sp->k_AH_03 * (k_lux - 1000) * (k_lux - 1000) / 1000000;
                k_ch1_H = als_info->sp->k_AH_11 +
                          als_info->sp->k_AH_12 * (k_lux - 1000) / 1000 +
                          als_info->sp->k_AH_13 * (k_lux - 1000) * (k_lux - 1000) / 1000000;

                ratio2 = ratio0 * k_ch1_H / k_ch0_H;

                if (ratio2 < als_info->sp->t_ratio_AH)
                {
                    als_info->out.lux = (uint64_t)((uint64_t)ch0_raw * k_lux * als_info->sp->k_A_ch0 + (uint64_t)ch1_raw * k_lux * als_info->sp->k_A_ch1) / 1000000;
                    light_type = A_LIGHT;
                }
                else
                {
                    als_info->out.lux = (uint64_t)((uint64_t)ch0_raw * k_lux * als_info->sp->k_H_ch0 + (uint64_t)ch1_raw * k_lux * als_info->sp->k_H_ch1) / 1000000;
                    light_type = H_LIGHT;
                }
            }
        }

        als_info->out.lux /= cpl0;

        if (als_info->out.lux < 0)
        {
            als_info->out.lux = 0;
        }

        als_info->out.lux_raw = als_info->out.lux / k_lux;
    }

    als_info->stat.data_changed = true;

    return 0;
}

static int sip1221lr1s_als_get_data(sip_i2c_info_t *i2c_info, sip1221lr1s_als_info_t *als_info)
{
    bool saturation_status;

    als_info->stat.data_changed = false;

    SIP_ASSERT(sip1221lr1s_als_get_stat(i2c_info, als_info));
    SIP_ASSERT(sip1221lr1s_als_analyse_stat(i2c_info, als_info));

    SIP_ASSERT(sip1221lr1s_als_read_raw_data(i2c_info, als_info));

    saturation_status = sip1221lr1s_als_auto_gain(i2c_info, als_info);
    if (!saturation_status)
    {
        if (als_info->stat.first_frame_flag_after_enable)
        {
#if !ALS_WAIT_TIMEOUT_OPTION
            SIP_ASSERT(sip1221lr1s_als_enable_private(i2c_info, false));
            SIP_ASSERT(sip1221lr1s_als_set_inte_time(i2c_info, als_info, als_info->conf.long_inte_time));
            SIP_ASSERT(sip1221lr1s_als_enable_private(i2c_info, true));
#endif /* ALS_WAIT_TIMEOUT_OPTION */

            als_info->stat.first_frame_flag_after_enable = false;
        }

        SIP_ASSERT(sip1221lr1s_als_compute_lux(als_info));
    }

    return 0;
}

static int sip1221lr1s_als_enable(struct sip1221lr1s_device *dev, bool bo)
{
    if (dev->ainfo.stat.enabled == bo)
        return 0;

    if (dev->ainfo.conf.is_dri == false)
    {
        if (bo)
        {
            timer_enable(&dev->als_timer, ALS_POLLING_TIME);
        }
        else
        {
            timer_disable(&dev->als_timer);
        }
    }
    sip1221lr1s_als_set_enable(&dev->iinfo, &dev->ainfo, bo);
    return 0;
}

static int sip1221lr1s_als_cali_enable(struct sip1221lr1s_device *dev, uint32_t cali_target)
{
    // SIP_ASSERT(sip1221lr1s_als_set_enable(&dev->iinfo, &dev->ainfo, bo));
    log_i("als cali target = %d\n", cali_target);
    if (dev->ainfo.stat.enabled)
    {
        SIP_ASSERT(sip1221lr1s_als_enable_private(&dev->iinfo, false));
        SIP_ASSERT(sip1221lr1s_als_set_gain(&dev->iinfo, 0, als_gains[9].tgain));
        SIP_ASSERT(sip1221lr1s_als_set_gain(&dev->iinfo, 1, als_gains[9].tgain));
        dev->ainfo.tmp.gain0_count = 9;
        dev->ainfo.tmp.gain1_count = 9;
        SIP_ASSERT(sip1221lr1s_als_enable_private(&dev->iinfo, true));
    }
    else
    {
        SIP_ASSERT(sip1221lr1s_als_set_inte_time(&dev->iinfo, &dev->ainfo, dev->ainfo.conf.long_inte_time));
        SIP_ASSERT(sip1221lr1s_als_set_gain(&dev->iinfo, 0, als_gains[10].tgain));
        SIP_ASSERT(sip1221lr1s_als_set_gain(&dev->iinfo, 1, als_gains[10].tgain));
        dev->ainfo.tmp.gain0_count = 10;
        dev->ainfo.tmp.gain1_count = 10;
        SIP_ASSERT(sip1221lr1s_als_enable_private(&dev->iinfo, true));
    }

    dev->ainfo.cali.target_lux = cali_target;
    dev->ainfo.stat.cali_enable = true;
    dev->ainfo.cali.cali_count = 0;
    dev->ainfo.cali.cali_result = ALS_CALI_FAIL;
    dev->ainfo.cali.light_lux_coef = 1000;
    timer_enable(&dev->als_cali_timer, ALS_POLLING_TIME);
    return 0;
}

static void sip1221lr1s_als_config(struct sip1221lr1s_device *dev, uint32_t k_lux)
{
    dev->ainfo.cali.light_lux_coef = k_lux;
}

static int sip1221lr1s_als_cali_disable(struct sip1221lr1s_device *dev)
{
    dev->ainfo.stat.cali_enable = false;
    if (!dev->ainfo.stat.enabled)
    {
        SIP_ASSERT(sip1221lr1s_als_enable_private(&dev->iinfo, false));
    }
    return 0;
}

static int sip1221lr1s_detect(struct sip1221lr1s_device *dev)
{
    int ret = 0, i = 0;

    for (i = 0; i < MAX_DETECT_RETRY_TIMES + 1; i++)
    {
        ret = dev->iinfo.check_reg(SIP1221_ID_REG, SIP1221_ID, 0xFF);
        if (ret >= 0)
        {
            log_i("detect success, cost-time: %d", i);
            return 0;
        }

        if (i != MAX_DETECT_RETRY_TIMES)
        {
            mdelay(1);
        }
    }

    log_e("fail to detect");
    return -ENODEV;
}

static int sip1221lr1s_chip_probe(struct sip1221lr1s_device *dev)
{
    int ret = 0;
    ret = sip1221lr1s_soft_reset(&dev->iinfo);
    if (ret < 0)
        return ret;
    mdelay(10);

    ret = sip1221lr1s_als_load_conf(&dev->iinfo, &dev->ainfo);
    dev->ainfo.stat.enabled = false;
    if (ret < 0)
        return ret;

    ret = sip1221lr1s_als_check_trim(&dev->iinfo);
    if (ret < 0)
        return ret;

//  init_waitqueue_head(&dev->wait);
    return 0;
}

static void als_work_func(struct work_struct *work)
{
    struct sip1221lr1s_device *dev = container_of(work, struct sip1221lr1s_device, als_work);
    if (dev->ainfo.stat.cali_enable)
    {
        log_i("sip1221lr1s is doing cali,immediate return\n");
        return;
    }

    if (!sip1221lr1s_als_get_data(&dev->iinfo, &dev->ainfo))
    {
        dev->als_data_is_ready = true;
//      wake_up_interruptible(&dev->wait);
    }
}

static void als_cali_work_func(struct work_struct *work)
{
    static int enter_times = 0;
    static uint32_t lux_sum = 0;
    uint32_t avg_lux = 0;
    struct sip1221lr1s_device *dev = container_of(work, struct sip1221lr1s_device, als_cali_work);

    enter_times++;
    if (enter_times > 10)
    {
        enter_times = 0;
        lux_sum = 0;
        dev->ainfo.cali.cali_count = 0;
        timer_disable(&dev->als_cali_timer);
        sip1221lr1s_als_cali_disable(dev);
        log_i("sip1221lr1s_als_cali_ failed\n");
    }

    if (!sip1221lr1s_als_get_data(&dev->iinfo, &dev->ainfo))
    {
        dev->ainfo.cali.cali_count++;
        if (dev->ainfo.cali.cali_count == 1)
        {
            enter_times = 0;
            lux_sum = 0;
        };
        lux_sum += dev->ainfo.out.lux;
        log_i("sip1221_als_cali_ lux_raw = %llu, lux_sum = %d, cali_count = %d\n", dev->ainfo.out.lux, lux_sum, dev->ainfo.cali.cali_count)
    }

    if (dev->ainfo.cali.cali_count >= SIP1221LR1S_LUX_AVERAGE_COUNT)
    {

        avg_lux = (lux_sum * 1000) / SIP1221LR1S_LUX_AVERAGE_COUNT / 1000;
        dev->ainfo.cali.cali_count = 0;
        if (avg_lux == 0)
        {
            avg_lux = 10;
        }
        dev->ainfo.cali.light_lux_coef = dev->ainfo.cali.target_lux * 1000 / avg_lux;
        log_i("sip1221_als_cali_ avg_lux = %d,light_lux_coef = %d, lux_sum = %d, target_lux = %d\n", avg_lux, dev->ainfo.cali.light_lux_coef, lux_sum, dev->ainfo.cali.target_lux);
        enter_times = 0;
        lux_sum = 0;
        dev->ainfo.cali.cali_result = ALS_CALI_SUCCESS;
        sip1221lr1s_als_cali_disable(dev);
        timer_disable(&dev->als_cali_timer);
    }
}

static void als_timer_function(struct timer_list *t)
{
    struct sip1221lr1s_device *dev = container_of(t, struct sip1221lr1s_device, als_timer);

    mod_timer(&dev->als_timer, jiffies + msecs_to_jiffies(ALS_POLLING_TIME));
    schedule_work(&dev->als_work);
}

static void als_cali_timer_function(struct timer_list *t)
{
    struct sip1221lr1s_device *dev = container_of(t, struct sip1221lr1s_device, als_cali_timer);

    mod_timer(&dev->als_cali_timer, jiffies + msecs_to_jiffies(ALS_POLLING_TIME));
    schedule_work(&dev->als_cali_work);
}

static void sip1221lr1s_timer_init(struct sip1221lr1s_device *dev)
{

    struct timer_list *timer;
    timer = &dev->als_timer;
#if 0
    init_timer(timer);
    timer->expires = jiffies + msecs_to_jiffies(ALS_POLLING_TIME);
    timer->function = als_timer_function;
    timer->data = (unsigned long)dev;
#else
    timer->expires = jiffies + msecs_to_jiffies(ALS_POLLING_TIME);
    timer_setup(&dev->als_timer, als_timer_function, 0);
#endif
    // init work function
    INIT_WORK(&dev->als_work, als_work_func);

    // als cali timer init
    timer = &dev->als_cali_timer;
    timer->expires = jiffies + msecs_to_jiffies(ALS_POLLING_TIME);
    timer_setup(&dev->als_cali_timer, als_cali_timer_function, 0);
    INIT_WORK(&dev->als_cali_work, als_cali_work_func);
}


#define SIP1221LR1S_LIGHT_CHANNEL {                       \
    .type = IIO_LIGHT,                                    \
    .indexed = 1,                                         \
    .channel = 0,                                         \
    .info_mask_separate = BIT(IIO_CHAN_INFO_PROCESSED)  | \
                          BIT(IIO_CHAN_INFO_CALIBSCALE) | \
                          BIT(IIO_CHAN_INFO_ENABLE)     | \
                          BIT(IIO_CHAN_INFO_OFFSET),      \
}

static const struct iio_chan_spec sip1221lr1s_channels[] = {
    SIP1221LR1S_LIGHT_CHANNEL,
};

static IIO_CONST_ATTR(in_illuminance_scale_available, SIP1221LR1S_SCALE_AVAILABLE);

static IIO_CONST_ATTR(in_illuminance_integration_time_available,
                    SIP1221LR1S_IT_AVAILABLE);

static struct attribute *sip1221lr1s_attributes[] = {
    &iio_const_attr_in_illuminance_scale_available.dev_attr.attr,
    &iio_const_attr_in_illuminance_integration_time_available.dev_attr.attr,
    NULL,
};

static const struct attribute_group sip1221lr1s_attribute_group = {
    .attrs = sip1221lr1s_attributes
};

static int sip1221lr1s_read_raw(struct iio_dev *indio_dev,
                                struct iio_chan_spec const *chan,
                                int *val, int *val2, long mask)
{
    struct sip1221lr1s_data *data = iio_priv(indio_dev);
    struct sip1221lr1s_device *dev = data->dev;

    switch (mask)
    {
    case IIO_CHAN_INFO_CALIBSCALE:
        *val = dev->ainfo.cali.light_lux_coef;
        return IIO_VAL_INT;
    case IIO_CHAN_INFO_PROCESSED:
        if (chan->type == IIO_LIGHT)
        {
            *val = dev->ainfo.out.lux;
        }
        return IIO_VAL_INT;
    case IIO_CHAN_INFO_OFFSET:
        *val = dev->ainfo.cali.cali_result;
        return IIO_VAL_INT;
    }

    return -EINVAL;
}

static int sip1221lr1s_write_raw(struct iio_dev *indio_dev,
                                struct iio_chan_spec const *chan,
                                int val, int val2, long mask)
{
    int ret = 0;
    struct sip1221lr1s_data *data = iio_priv(indio_dev);
    struct sip1221lr1s_device *dev = data->dev;
    switch (mask)
    {
        case IIO_CHAN_INFO_CALIBSCALE:
            ret = sip1221lr1s_als_cali_enable(dev, val);
        break;

        case IIO_CHAN_INFO_ENABLE:
            ret = sip1221lr1s_als_enable(dev, val);
        break;
        case IIO_CHAN_INFO_OFFSET:
            sip1221lr1s_als_config(dev, val);
        break;
    }

    return ret;
}

static const struct iio_info sip1221lr1s_info = {
    .read_raw = sip1221lr1s_read_raw,
    .write_raw = sip1221lr1s_write_raw,
    .attrs = &sip1221lr1s_attribute_group,
};

static int sip1221lr1s_probe(struct i2c_client *client,
                             const struct i2c_device_id *id)
{
    int ret;
    struct iio_dev *indio_dev;
    struct sip1221lr1s_data *data;
    struct sip1221lr1s_device *dev = &g_sensor_device;

    if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C))
    {
        log_e("I2C check functionality failed.");
        return -ENODEV;
    }
    dev->client = client;

    ret = sip1221lr1s_detect(dev);
    if (ret < 0)
    {
        goto err_standby;
    }

    indio_dev = devm_iio_device_alloc(&client->dev, sizeof(*data));
    if (!indio_dev)
    {
        dev_err(&client->dev, "iio allocation failed!\n");
        return -ENOMEM;
    }

    data = iio_priv(indio_dev);
    data->client = client;
    data->dev = &g_sensor_device;
    i2c_set_clientdata(client, indio_dev);
    mutex_init(&data->lock);

    indio_dev->dev.parent = &client->dev;
    indio_dev->info = &sip1221lr1s_info;
    indio_dev->name = SIP1221LR1S_DRIVER_NAME;
    indio_dev->modes = INDIO_DIRECT_MODE;
    indio_dev->channels = sip1221lr1s_channels;
    indio_dev->num_channels = ARRAY_SIZE(sip1221lr1s_channels);

    ret = iio_device_register(indio_dev);
    if (ret < 0)
    {
        log_e("device_register failed\n");
        goto err_standby;
    }

    sip1221lr1s_timer_init(data->dev);

    ret = sip1221lr1s_chip_probe(data->dev);
    if (ret < 0)
    {
        log_e("sip1221lr1s probe error");
        return ret;
    }

    dev_err(&client->dev, "probe success\n");
    return 0;

err_standby:

    return ret;
}

static int sip1221lr1s_remove(struct i2c_client *client)
{
    struct iio_dev *indio_dev = i2c_get_clientdata(client);
    iio_device_unregister(indio_dev);
    return 0;
}


static const struct i2c_device_id sip1221lr1s_i2c_id[] = {
    {"sip1221lr1s", 0},
    {}
};

MODULE_DEVICE_TABLE(i2c, sip1221lr1s_i2c_id);

static const struct of_device_id sip1221lr1s_of_match[] = {
    {.compatible = "si,sip1221lr1s"},
    {}
};

static struct i2c_driver sip1221lr1s_driver = {
    .driver = {
        .name = "sip1221lr1s",
        .of_match_table = sip1221lr1s_of_match,
    },
    .probe = sip1221lr1s_probe,
    .remove = sip1221lr1s_remove,
    .id_table = sip1221lr1s_i2c_id,
};

module_i2c_driver(sip1221lr1s_driver);

MODULE_AUTHOR("si");
MODULE_DESCRIPTION("SIP1221LR1S Ambient Light Sensor driver");
MODULE_LICENSE("GPL v2");
