// SPDX-License-Identifier: GPL-2.0
/*
 * Raydium RM67191 MIPI-DSI panel driver
 *
 * Copyright 2019 NXP
 */

#include <linux/backlight.h>
#include <linux/delay.h>
#include <linux/gpio/consumer.h>
#include <linux/media-bus-format.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/regulator/consumer.h>

#include <video/mipi_display.h>
#include <video/of_videomode.h>
#include <video/videomode.h>

#include <drm/drm_crtc.h>
#include <drm/drm_mipi_dsi.h>
#include <drm/drm_panel.h>
#include <linux/of_gpio.h>

/* Panel specific color-format bits */
#define COL_FMT_16BPP 0x55
#define COL_FMT_18BPP 0x66
#define COL_FMT_24BPP 0x77

/* Write Manufacture Command Set Control */
#define WRMAUCCTR 0xFE

int lcd_sel_vendor_in_panel;
int lcd_sel_vendor_in_panel_gpio_value;

/* Manufacturer Command Set pages (CMD2) */
struct cmd_set_entry {
	u8 cmd;
	u8 param;
};

static const struct cmd_set_entry page_5[] = {
{0xB0,0x05},{0xB3,0x72},
};

static const u32 rad_bus_formats[] = {
	MEDIA_BUS_FMT_RGB888_1X24,
	MEDIA_BUS_FMT_RGB666_1X18,
	MEDIA_BUS_FMT_RGB565_1X16,
};

static const u32 rad_bus_flags = DRM_BUS_FLAG_DE_LOW |
				 DRM_BUS_FLAG_PIXDATA_SAMPLE_POSEDGE;

struct rad_panel {
	struct drm_panel panel;
	struct mipi_dsi_device *dsi;

	struct gpio_desc *reset,*bl_en;
	struct backlight_device *backlight;

	struct regulator_bulk_data *supplies;
	unsigned int num_supplies;

	bool prepared;
	bool enabled;

	const struct rad_platform_data *pdata;
};

struct rad_platform_data {
	int (*enable)(struct rad_panel *panel);
};

static int rad_panel_push_cmd_list(struct mipi_dsi_device *dsi,
				   struct cmd_set_entry const *cmd_set,
				   size_t count)
{
	size_t i;
	int ret = 0;

	for (i = 0; i < count; i++) {
		const struct cmd_set_entry *entry = cmd_set++;
		u8 buffer[2] = { entry->cmd, entry->param };

		ret = mipi_dsi_generic_write(dsi, &buffer, sizeof(buffer));
		if (ret < 0)
			return ret;
	}

	return ret;
};

static const struct drm_display_mode default_mode = {
	/* For CBG LCD(1200x1920) */
	.clock = 150000,
	.hdisplay = 1200,
	.hsync_start = 1200 + 40,		//hsync_start=hdisplay+hfp
	.hsync_end = 1200 + 40 + 10,	//hsync_end=hsync_start+hsync
	.htotal = 1200 + 40 + 10 + 100,	//htotal=hsync_end+hbp
	.vdisplay = 1920,
	.vsync_start = 1920 + 18,		//vsync_start=vdisplay+vfp
	.vsync_end = 1920 + 18 + 4,		//vsync_end=vsync_start+vsync
	.vtotal = 1920 + 18 + 4 + 70,	//vtotal=vsync_end+vbp
	.width_mm = 135,
	.height_mm = 217,

	.flags = DRM_MODE_FLAG_NHSYNC |
		 DRM_MODE_FLAG_NVSYNC,
};

static const struct drm_display_mode willtron_default_mode = {
    /* For Willtron LCD(1200x1920) */
	.clock = 159261,
	.hdisplay = 1200,
	.hsync_start = 1200 + 60, 		//hsync_start=hdisplay+hfp
	.hsync_end = 1200 + 60 + 24,	//hsync_end=hsync_start+hsync
	.htotal = 1200 + 60 + 24 + 80,	//htotal=hsync_end+hbp
	.vdisplay = 1920,
	.vsync_start = 1920 + 14,		//vsync_start=vdisplay+vfp
	.vsync_end = 1920 + 14 + 2,		//vsync_end=vsync_start+vsync
	.vtotal = 1920 + 14 + 2 + 10,	//vtotal=vsync_end+vbp
	.width_mm = 135,
	.height_mm = 217,

	.flags = DRM_MODE_FLAG_NHSYNC |
		 DRM_MODE_FLAG_NVSYNC,
};

static inline struct rad_panel *to_rad_panel(struct drm_panel *panel)
{
	return container_of(panel, struct rad_panel, panel);
}

static int rad_panel_prepare(struct drm_panel *panel)
{
	struct rad_panel *rad = to_rad_panel(panel);
	int ret;

	if (rad->prepared)
		return 0;

	ret = regulator_bulk_enable(rad->num_supplies, rad->supplies);
	if (ret)
		return ret;

	/* At lest 10ms needed between power-on and reset-out as RM specifies */
	usleep_range(10000, 12000);

	if (rad->reset) {
		if (lcd_sel_vendor_in_panel_gpio_value == 0) {
		  printk("[MIPI]rad_panel_prepare-Set reset low");
		  gpiod_set_value_cansleep(rad->reset, 0);
		  usleep_range(3000, 3500);
	    }
		printk("[MIPI]rad_panel_prepare-Set reset high");
		gpiod_set_value_cansleep(rad->reset, 1);
		/*
		 * 50ms delay after reset-out, as per manufacturer initalization
		 * sequence.
		 */
		msleep(50);
	}

	rad->prepared = true;

	return 0;
}

static int rad_panel_unprepare(struct drm_panel *panel)
{
	struct rad_panel *rad = to_rad_panel(panel);
	int ret;

	if (!rad->prepared)
		return 0;

	/*
	 * Right after asserting the reset, we need to release it, so that the
	 * touch driver can have an active connection with the touch controller
	 * even after the display is turned off.
	 */
	if (rad->reset) {
		if (lcd_sel_vendor_in_panel_gpio_value == 0) {
			gpiod_set_value_cansleep(rad->reset, 0);
			printk("[MIPI]rad_panel_prepare-Set reset low");
			usleep_range(15000, 17000);
			gpiod_set_value_cansleep(rad->reset, 1);
			printk("[MIPI]rad_panel_prepare-Set reset high");
		}
	}

	ret = regulator_bulk_disable(rad->num_supplies, rad->supplies);
	if (ret)
		return ret;

	rad->prepared = false;

	return 0;
}

static int proto_enable(struct rad_panel *panel)
{
	struct mipi_dsi_device *dsi = panel->dsi;
	struct device *dev = &dsi->dev;
	int ret;

	if (panel->enabled)
		return 0;

	dsi->mode_flags |= MIPI_DSI_MODE_LPM;

	/* Exit sleep mode */
	printk("[MIPI]GN LCM initial-Sleep Out");
	ret = mipi_dsi_dcs_exit_sleep_mode(dsi);
	if (ret < 0) {
		dev_err(dev, "Failed to exit sleep mode (%d)\n", ret);
		goto fail;
	}

	usleep_range(120000, 125000);

	printk("[MIPI]GN LCM initial-Display On");
	ret = mipi_dsi_dcs_set_display_on(dsi);
	if (ret < 0) {
		dev_err(dev, "Failed to set display ON (%d)\n", ret);
		goto fail;
	}

	usleep_range(50000, 50500);

	/* Fix Willtron LCD abnormal display issue */
	if(lcd_sel_vendor_in_panel_gpio_value == 1){
		ret = rad_panel_push_cmd_list(dsi,
				      &page_5[0],
				      ARRAY_SIZE(page_5));
		if (ret < 0) {
			dev_err(dev, "Failed to send MCS (%d)\n", ret);
			goto fail;
		}
	}

	gpiod_set_value_cansleep(panel->bl_en, 1);
	printk("[MIPI]Set BL_EN to on");
	backlight_enable(panel->backlight);

	panel->enabled = true;

	return 0;

fail:
	gpiod_set_value_cansleep(panel->reset, 0);
	printk("[MIPI]rad_panel_prepare-Set reset low");

	return ret;
}

static int rad_panel_enable(struct drm_panel *panel)
{
	struct rad_panel *rad = to_rad_panel(panel);

	return rad->pdata->enable(rad);
}

static int rad_panel_disable(struct drm_panel *panel)
{
	struct rad_panel *rad = to_rad_panel(panel);
	struct mipi_dsi_device *dsi = rad->dsi;
	struct device *dev = &dsi->dev;
	int ret;

	printk("[MIPI]rad_panel_disable");
	if (!rad->enabled)
		return 0;

	dsi->mode_flags |= MIPI_DSI_MODE_LPM;

	gpiod_set_value_cansleep(rad->bl_en, 0);
	printk("[MIPI]Set BL_EN to off");
	
	backlight_disable(rad->backlight);

	usleep_range(20000, 25000);

	printk("[MIPI]GN LCM initial-Display Off");
	ret = mipi_dsi_dcs_set_display_off(dsi);
	if (ret < 0) {
		dev_err(dev, "Failed to set display OFF (%d)\n", ret);
		return ret;
	}

	usleep_range(5000, 5500);

	if (lcd_sel_vendor_in_panel_gpio_value == 0) {
		printk("[MIPI]GN LCM initial-Sleep In");
		ret = mipi_dsi_dcs_enter_sleep_mode(dsi);
			if (ret < 0) {
				dev_err(dev, "Failed to enter sleep mode (%d)\n", ret);
				return ret;
			}
		usleep_range(130000, 135000);
    }

	rad->enabled = false;

	return 0;
}

static int rad_panel_get_modes(struct drm_panel *panel,
			       struct drm_connector *connector)
{
	struct drm_display_mode *mode;

	if (lcd_sel_vendor_in_panel_gpio_value == 0)
	{
		mode = drm_mode_duplicate(connector->dev, &default_mode);
		if (!mode) {
			dev_err(panel->dev, "failed to add mode %ux%u@%u\n",
			default_mode.hdisplay, default_mode.vdisplay,
			drm_mode_vrefresh(&default_mode));
			return -ENOMEM;
		}
	}
	else
	{
		mode = drm_mode_duplicate(connector->dev, &willtron_default_mode);
		if (!mode) {
			dev_err(panel->dev, "failed to add mode %ux%u@%u\n",
			willtron_default_mode.hdisplay, willtron_default_mode.vdisplay,
			drm_mode_vrefresh(&willtron_default_mode));
			return -ENOMEM;
		}
	}

	drm_mode_set_name(mode);
	mode->type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED;
	drm_mode_probed_add(connector, mode);

	connector->display_info.width_mm = mode->width_mm;
	connector->display_info.height_mm = mode->height_mm;
	connector->display_info.bus_flags = rad_bus_flags;

	drm_display_info_set_bus_formats(&connector->display_info,
					 rad_bus_formats,
					 ARRAY_SIZE(rad_bus_formats));
	return 1;
}

static int rad_bl_get_brightness(struct backlight_device *bl)
{
	struct mipi_dsi_device *dsi = bl_get_data(bl);
	struct rad_panel *rad = mipi_dsi_get_drvdata(dsi);
	u16 brightness;
	int ret;

	if (!rad->prepared)
		return 0;

	dsi->mode_flags &= ~MIPI_DSI_MODE_LPM;

	ret = mipi_dsi_dcs_get_display_brightness(dsi, &brightness);
	if (ret < 0)
		return ret;

	bl->props.brightness = brightness;

	return brightness & 0xff;
}

static int rad_bl_update_status(struct backlight_device *bl)
{
	struct mipi_dsi_device *dsi = bl_get_data(bl);
	struct rad_panel *rad = mipi_dsi_get_drvdata(dsi);
	int ret = 0;

	if (!rad->prepared)
		return 0;

	dsi->mode_flags &= ~MIPI_DSI_MODE_LPM;

	ret = mipi_dsi_dcs_set_display_brightness(dsi, bl->props.brightness);
	if (ret < 0)
		return ret;

	return 0;
}

static const struct backlight_ops rad_bl_ops = {
	.update_status = rad_bl_update_status,
	.get_brightness = rad_bl_get_brightness,
};

static const struct drm_panel_funcs rad_panel_funcs = {
	.prepare = rad_panel_prepare,
	.unprepare = rad_panel_unprepare,
	.enable = rad_panel_enable,
	.disable = rad_panel_disable,
	.get_modes = rad_panel_get_modes,
};

static const char * const rad_supply_names[] = {
	"v3p3",
	"v1p8",
};

static int rad_init_regulators(struct rad_panel *rad)
{
	struct device *dev = &rad->dsi->dev;
	int i;

	rad->num_supplies = ARRAY_SIZE(rad_supply_names);
	rad->supplies = devm_kcalloc(dev, rad->num_supplies,
				     sizeof(*rad->supplies), GFP_KERNEL);
	if (!rad->supplies)
		return -ENOMEM;

	for (i = 0; i < rad->num_supplies; i++)
		rad->supplies[i].supply = rad_supply_names[i];

	return devm_regulator_bulk_get(dev, rad->num_supplies, rad->supplies);
};

static const struct rad_platform_data gn_proto = {
	.enable = &proto_enable,
};

static const struct of_device_id rad_of_match[] = {
	{ .compatible = "gn,proto", .data = &gn_proto },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, rad_of_match);

static int rad_panel_probe(struct mipi_dsi_device *dsi)
{
	struct device *dev = &dsi->dev;
	const struct of_device_id *of_id = of_match_device(rad_of_match, dev);
	struct device_node *np = dev->of_node;
	struct rad_panel *panel;
	int ret;
	u32 video_mode;

	if (!of_id || !of_id->data)
		return -ENODEV;
	panel = devm_kzalloc(&dsi->dev, sizeof(*panel), GFP_KERNEL);
	if (!panel)
		return -ENOMEM;
	mipi_dsi_set_drvdata(dsi, panel);
	panel->dsi = dsi;
	panel->pdata = of_id->data;

	dsi->format = MIPI_DSI_FMT_RGB888;
	dsi->mode_flags = MIPI_DSI_MODE_VIDEO_HSE | MIPI_DSI_MODE_NO_EOT_PACKET;
	ret = of_property_read_u32(np, "video-mode", &video_mode);
	if (!ret) {
		switch (video_mode) {
		case 0:
			/* burst mode */
			dsi->mode_flags |= MIPI_DSI_MODE_VIDEO_BURST |
					   MIPI_DSI_MODE_VIDEO;
			break;
		case 1:
			/* non-burst mode with sync event */
			dsi->mode_flags |= MIPI_DSI_MODE_VIDEO;
			break;
		case 2:
			/* non-burst mode with sync pulse */
			dsi->mode_flags |= MIPI_DSI_MODE_VIDEO_SYNC_PULSE |
					   MIPI_DSI_MODE_VIDEO;
			break;
		case 3:
			/* command mode */
			dsi->mode_flags |= MIPI_DSI_CLOCK_NON_CONTINUOUS |
					   MIPI_DSI_MODE_VSYNC_FLUSH;
			break;
		default:
			dev_warn(dev, "invalid video mode %d\n", video_mode);
			break;
		}
	}

	lcd_sel_vendor_in_panel = of_get_named_gpio(np, "lcd_sel-gpio", 0);

	printk("lcd_sel_vendor_in_panel = %d", lcd_sel_vendor_in_panel);

	if (gpio_request(lcd_sel_vendor_in_panel, "lcd-sel") < 0)
	{
		printk("Failed to request 'lcd_sel-gpio' in device tree");
	}
	else
	{
		gpio_direction_input(lcd_sel_vendor_in_panel);
		lcd_sel_vendor_in_panel_gpio_value = gpio_get_value(lcd_sel_vendor_in_panel);
		printk("lcd_sel_vendor_in_panel's gpio value = %d", lcd_sel_vendor_in_panel_gpio_value);
		gpio_free(lcd_sel_vendor_in_panel);
	}

	ret = of_property_read_u32(np, "dsi-lanes", &dsi->lanes);
	if (ret) {
		dev_err(dev, "Failed to get dsi-lanes property (%d)\n", ret);
		return ret;
	}

	panel->reset = devm_gpiod_get_optional(dev, "reset",
					       GPIOD_OUT_LOW |
					       GPIOD_FLAGS_BIT_NONEXCLUSIVE);
	if (IS_ERR(panel->reset)) {
		ret = PTR_ERR(panel->reset);
		dev_err(dev, "Failed to get reset gpio (%d)\n", ret);
		return ret;
	}
	gpiod_set_value_cansleep(panel->reset, 0);
	printk("[MIPI]rad_panel_probe-Set reset low");

	panel->bl_en = devm_gpiod_get_optional(dev, "bl_en",
					       GPIOD_OUT_LOW |
					       GPIOD_FLAGS_BIT_NONEXCLUSIVE);
	if (IS_ERR(panel->bl_en)) {
		ret = PTR_ERR(panel->bl_en);
		dev_err(dev, "Failed to get bl_en gpio (%d)\n", ret);
		return ret;
	}
	gpiod_set_value_cansleep(panel->bl_en, 0);
	printk("[MIPI]rad_panel_probe-Set bl_en low");

	ret = rad_init_regulators(panel);
	if (ret)
		return ret;

	drm_panel_init(&panel->panel, dev, &rad_panel_funcs,
		       DRM_MODE_CONNECTOR_DSI);
	dev_set_drvdata(dev, panel);

	drm_panel_add(&panel->panel);

	ret = mipi_dsi_attach(dsi);
	if (ret)
		drm_panel_remove(&panel->panel);

	return ret;
}

static int rad_panel_remove(struct mipi_dsi_device *dsi)
{
	struct rad_panel *rad = mipi_dsi_get_drvdata(dsi);
	struct device *dev = &dsi->dev;
	int ret;

	ret = mipi_dsi_detach(dsi);
	if (ret)
		dev_err(dev, "Failed to detach from host (%d)\n", ret);

	drm_panel_remove(&rad->panel);

	return 0;
}

static void rad_panel_shutdown(struct mipi_dsi_device *dsi)
{
	struct rad_panel *rad = mipi_dsi_get_drvdata(dsi);

	rad_panel_disable(&rad->panel);
	rad_panel_unprepare(&rad->panel);
}

static struct mipi_dsi_driver rad_panel_driver = {
	.driver = {
		.name = "panel-gn-proto",
		.of_match_table = rad_of_match,
	},
	.probe = rad_panel_probe,
	.remove = rad_panel_remove,
	.shutdown = rad_panel_shutdown,
};
module_mipi_dsi_driver(rad_panel_driver);

MODULE_AUTHOR("Robert Chiras <robert.chiras@nxp.com>");
MODULE_DESCRIPTION("DRM Driver for Raydium RM67191 MIPI DSI panel");
MODULE_LICENSE("GPL v2");
